/*
 * cramfs.c
 *
 * Copyright (C) 1999 Linus Torvalds
 *
 * Copyright (C) 2000-2002 Transmeta Corporation
 *
 * Copyright (C) 2003 Kai-Uwe Bloem,
 * Auerswald GmbH & Co KG, <linux-development@auerswald.de>
 * - adapted from the www.tuxbox.org u-boot tree, added "ls" command
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License (Version 2) as
 * published by the Free Software Foundation.
 *
 * Compressed ROM filesystem for Linux.
 *
 * TODO:
 * add support for resolving symbolic links
 */

/*
 * These are the VFS interfaces to the compressed ROM filesystem.
 * The actual compression is based on zlib, see the other files.
 */

#include <common.h>
#include <malloc.h>
#include <asm/byteorder.h>
#include <linux/stat.h>
#include <jffs2/jffs2.h>
#include <jffs2/load_kernel.h>
#include <cramfs/cramfs_fs.h>

#include <spi.h>
#include <spi_flash.h>
#include <nand.h>
 
/* These two macros may change in future, to provide better st_ino
   semantics. */
#define CRAMINO(x)	(CRAMFS_GET_OFFSET(x) ? CRAMFS_GET_OFFSET(x)<<2 : 1)
#define OFFSET(x)	((x)->i_ino)

struct cramfs_super super;
// --> liu_hongcong
#ifdef CONFIG_FSLOAD
typedef struct {
	ulong	size;			/* total bank size in bytes		*/
	ushort	sector_count;		/* number of erase units		*/
	ulong	flash_id;		/* combined device & manufacturer code	*/
	ulong	start[CONFIG_SYS_MAX_FLASH_SECT];   /* virtual sector start address */
	uchar	protect[CONFIG_SYS_MAX_FLASH_SECT]; /* sector protection status	*/
#ifdef CONFIG_SYS_FLASH_CFI
	uchar	portwidth;		/* the width of the port		*/
	uchar	chipwidth;		/* the width of the chip		*/
	ushort	buffer_size;		/* # of bytes in write buffer		*/
	ulong	erase_blk_tout;		/* maximum block erase timeout		*/
	ulong	write_tout;		/* maximum write timeout		*/
	ulong	buffer_write_tout;	/* maximum buffer write timeout		*/
	ushort	vendor;			/* the primary vendor id		*/
	ushort	cmd_reset;		/* vendor specific reset command	*/
	ushort	interface;		/* used for x8/x16 adjustments		*/
	ushort	legacy_unlock;		/* support Intel legacy (un)locking	*/
	ushort	manufacturer_id;	/* manufacturer id			*/
	ushort	device_id;		/* device id				*/
	ushort	device_id2;		/* extended device id			*/
	ushort	ext_addr;		/* extended query table address		*/
	ushort	cfi_version;		/* cfi version				*/
	ushort	cfi_offset;		/* offset for cfi query			*/
	ulong   addr_unlock1;		/* unlock address 1 for AMD flash roms  */
	ulong   addr_unlock2;		/* unlock address 2 for AMD flash roms  */
	const char *name;		/* human-readable name	                */
#endif
} flash_info_t;
#endif
// <-- liu_hongcong

/* CPU address space offset calculation macro, struct part_info offset is
 * device address space offset, so we need to shift it by a device start address. */
extern flash_info_t flash_info[];
// --> liu_hongcong
#ifndef CONFIG_FSLOAD
#define PART_OFFSET(x)	(x->offset + flash_info[x->dev->id->num].start[0])
#else
#define PART_OFFSET(x)	(x->ram_img)
#endif

static int cramfs_read_super (struct part_info *info)
{
	unsigned long root_offset;

	/* Read the first block and get the superblock from it */
	memcpy (&super, (void *) PART_OFFSET(info), sizeof (super));

	/* Do sanity checks on the superblock */
	if (super.magic != CRAMFS_32 (CRAMFS_MAGIC)) {
		/* check at 512 byte offset */
		memcpy (&super, (void *) PART_OFFSET(info) + 512, sizeof (super));
		if (super.magic != CRAMFS_32 (CRAMFS_MAGIC)) {
			printf ("cramfs: wrong magic\n");
			return -1;
		}
	}

	/* flags is reused several times, so swab it once */
	super.flags = CRAMFS_32 (super.flags);
	super.size = CRAMFS_32 (super.size);

	/* get feature flags first */
	if (super.flags & ~CRAMFS_SUPPORTED_FLAGS) {
		printf ("cramfs: unsupported filesystem features\n");
		return -1;
	}

	/* Check that the root inode is in a sane state */
	if (!S_ISDIR (CRAMFS_16 (super.root.mode))) {
		printf ("cramfs: root is not a directory\n");
		return -1;
	}
	root_offset = CRAMFS_GET_OFFSET (&(super.root)) << 2;
	if (root_offset == 0) {
		printf ("cramfs: empty filesystem");
	} else if (!(super.flags & CRAMFS_FLAG_SHIFTED_ROOT_OFFSET) &&
		   ((root_offset != sizeof (struct cramfs_super)) &&
		    (root_offset != 512 + sizeof (struct cramfs_super)))) {
		printf ("cramfs: bad root offset %lu\n", root_offset);
		return -1;
	}

	return 0;
}

static unsigned long cramfs_resolve (unsigned long begin, unsigned long offset,
				     unsigned long size, int raw,
				     char *filename)
{
	unsigned long inodeoffset = 0, nextoffset;

	while (inodeoffset < size) {
		struct cramfs_inode *inode;
		char *name;
		int namelen;

		inode = (struct cramfs_inode *) (begin + offset +
						 inodeoffset);

		/*
		 * Namelengths on disk are shifted by two
		 * and the name padded out to 4-byte boundaries
		 * with zeroes.
		 */
		namelen = CRAMFS_GET_NAMELEN (inode) << 2;
		name = (char *) inode + sizeof (struct cramfs_inode);

		nextoffset =
			inodeoffset + sizeof (struct cramfs_inode) + namelen;

		for (;;) {
			if (!namelen)
				return -1;
			if (name[namelen - 1])
				break;
			namelen--;
		}

		if (!strncmp (filename, name, namelen)) {
			char *p = strtok (NULL, "/");

			if (raw && (p == NULL || *p == '\0'))
				return offset + inodeoffset;

			if (S_ISDIR (CRAMFS_16 (inode->mode))) {
				return cramfs_resolve (begin,
						       CRAMFS_GET_OFFSET
						       (inode) << 2,
						       CRAMFS_24 (inode->
								  size), raw,
						       p);
			} else if (S_ISREG (CRAMFS_16 (inode->mode))) {
				return offset + inodeoffset;
			} else {
				printf ("%*.*s: unsupported file type (%x)\n",
					namelen, namelen, name,
					CRAMFS_16 (inode->mode));
				return 0;
			}
		}

		inodeoffset = nextoffset;
	}

	printf ("can't find corresponding entry\n");
	return 0;
}

static int cramfs_uncompress (unsigned long begin, unsigned long offset,
			      unsigned long loadoffset)
{
	struct cramfs_inode *inode = (struct cramfs_inode *) (begin + offset);
	unsigned long *block_ptrs = (unsigned long *)
		(begin + (CRAMFS_GET_OFFSET (inode) << 2));
	unsigned long curr_block = (CRAMFS_GET_OFFSET (inode) +
				    (((CRAMFS_24 (inode->size)) +
				      4095) >> 12)) << 2;
	int size, total_size = 0;
	int i;

	cramfs_uncompress_init ();

	for (i = 0; i < ((CRAMFS_24 (inode->size) + 4095) >> 12); i++) {
		size = cramfs_uncompress_block ((void *) loadoffset,
						(void *) (begin + curr_block),
						(CRAMFS_32 (block_ptrs[i]) -
						 curr_block));
		if (size < 0)
			return size;
		loadoffset += size;
		total_size += size;
		curr_block = CRAMFS_32 (block_ptrs[i]);
	}

	cramfs_uncompress_exit ();
	return total_size;
}

int cramfs_load (char *loadoffset, struct part_info *info, char *filename)
{
	unsigned long offset;

	if (cramfs_read_super (info))
		return -1;

	offset = cramfs_resolve ((unsigned long)PART_OFFSET(info),
				 CRAMFS_GET_OFFSET (&(super.root)) << 2,
				 CRAMFS_24 (super.root.size), 0,
				 strtok (filename, "/"));

	if (offset <= 0)
		return offset;

	return cramfs_uncompress ((unsigned long)PART_OFFSET(info), offset,
				  (unsigned long) loadoffset);
}

#ifdef CONFIG_PARTS	
static int cramfs_get_super (char *buf, struct cramfs_super *super)
{
	unsigned long root_offset;
        
    memcpy((char *)super, buf, sizeof(struct cramfs_super));

	/* Do sanity checks on the superblock */
	if (super->magic != CRAMFS_32 (CRAMFS_MAGIC)) {
		/* check at 512 byte offset */
        memcpy((char *)super, buf + 512, sizeof(struct cramfs_super));
		if (super->magic != CRAMFS_32 (CRAMFS_MAGIC)) {
			printf ("cramfs: wrong magic\n");
            return -1;
        }
	}

	/* flags is reused several times, so swab it once */
	super->flags = CRAMFS_32 (super->flags);
	super->size = CRAMFS_32 (super->size);

	/* get feature flags first */
	if (super->flags & ~CRAMFS_SUPPORTED_FLAGS) {
		printf ("cramfs: unsupported filesystem features\n");
        return -1;
	}

	/* Check that the root inode is in a sane state */
	if (!S_ISDIR (CRAMFS_16 (super->root.mode))) {
		printf ("cramfs: root is not a directory\n");
        return -1;
	}
	root_offset = CRAMFS_GET_OFFSET (&(super->root)) << 2;
	if (root_offset == 0) {
		printf ("cramfs: empty filesystem");
	} else if (!(super->flags & CRAMFS_FLAG_SHIFTED_ROOT_OFFSET) &&
		   ((root_offset != sizeof (struct cramfs_super)) &&
		    (root_offset != 512 + sizeof (struct cramfs_super)))) {
		printf ("cramfs: bad root offset %lu\n", root_offset);
        return -1;
	}

    return 0;
}

#if 0
static int my_cramfs_uncompress (unsigned long begin, unsigned long offset,
			      unsigned long loadoffset)
{
	struct cramfs_inode *inode = (struct cramfs_inode *) (begin + offset);
	unsigned long *block_ptrs = (unsigned long *)
		(begin + (CRAMFS_GET_OFFSET (inode) << 2));
	unsigned long curr_block = (CRAMFS_GET_OFFSET (inode) +
				    (((CRAMFS_24 (inode->size)) +
				      4095) >> 12)) << 2;
	int size, total_size = 0;
	int i;

	cramfs_uncompress_init ();

	for (i = 0; i < ((CRAMFS_24 (inode->size) + 4095) >> 12); i++) {
		size = cramfs_uncompress_block ((void *) loadoffset,
						(void *) (begin + curr_block),
						(CRAMFS_32 (block_ptrs[i]) -
						 curr_block));
		if (size < 0)
			return size;
		loadoffset += size;
		total_size += size;
		curr_block = CRAMFS_32 (block_ptrs[i]);
	}

	cramfs_uncompress_exit ();
	return total_size;
}
#endif

static inline void * flash_dev_get(void)
{
#ifdef CONFIG_SPI_FLASH
    return spi_flash_probe(PARTITION_SPI_BUS, PARTITION_SPI_CS,
			PARTITION__SPI_MAX_HZ, PARTITION_SPI_MODE);
#elif CONFIG_NAND
    return &nand_info[nand_curr_device];
#else
error need flash dev
#endif
}

static inline int flash_dev_release(void *flash)
{
#ifdef CONFIG_SPI_FLASH
    return spi_flash_free((struct spi_flash *)flash);
#elif CONFIG_NAND
    return 0;
#else
error need flash dev
#endif
}


static inline int flash_read(void *flash, u32 offset,
		size_t len, void *buf)
{
    
#ifdef CONFIG_SPI_FLASH
        return spi_flash_read((struct spi_flash *)flash, offset,
                len, buf);
#elif CONFIG_NAND
        return nand_read_skip_bad((nand_info_t *)flash,offset,&len,buf);
#else
    error need flash dev
#endif
}

/* cramfsļϵͳжȡϢļ */
int cramfs_load_with_name (void *loadoffset, char *filename, u32 part_offset)
{
    unsigned long offset;
    void *buf;
    int ret = 0;
    static void *flash;
    struct cramfs_super *super;
        
	flash = flash_dev_get();
	if (!flash)
	{
        printf("fail to probe flash\n");
		return -1;
	}

	buf = malloc(CRAMFS_SUPER_READ_SIZE);
    if (!buf)
    {
        printf("fail to malloc partinfo head\n");
		goto free_flash;
    }

    //ȡļϵͳͷݣڽsuperϢ
    if (flash_read(flash, part_offset, CRAMFS_SUPER_READ_SIZE, buf))
    {
        printf("fail to read partinfo head\n");
		ret = -1;
		goto free_buf;
    }

	super = malloc(sizeof(struct cramfs_super));
    if (super == NULL)
    {
        printf("fail to malloc super\n");
		ret = -1;
		goto free_buf;
    }
    memset(super, 0, sizeof(struct cramfs_super));
    
    //ļϵͳͷݽsuperϢ
	if (cramfs_get_super(buf, super))
	{
        printf("fail to read super\n");
		ret = -1;
		goto free_super;
	}
    
    //ļϵͳڶȡͷϢʱĴС¶ȡļϵͳ
    if (super->size > CRAMFS_SUPER_READ_SIZE)
    {
        free(buf);
        
        buf = malloc(super->size);
        if (!buf)
        {
            printf("fail to malloc partinfo body\n");
            free(super);
            ret = -1;
            goto free_flash;
        }

        if (flash_read(flash, part_offset, super->size, buf))
        {
            printf("fail to read partinfo body\n");
    		ret = -1;
    		goto free_super;
        }
    }
    
    //ļϵͳнϢļ
	offset = cramfs_resolve ((unsigned long)buf,
				 CRAMFS_GET_OFFSET (&(super->root)) << 2,
				 CRAMFS_24 (super->root.size), 0,
				 strtok (filename, "/"));
    
	if (offset <= 0)
	{
        printf("fail to resolve\n");
		ret = -1;
		goto free_super;
	}

	if (cramfs_uncompress ((unsigned long)buf, offset,
				  (unsigned long) loadoffset) <= 0)
	{
        printf("fail to uncompress\n");
		ret = -1;
		goto free_super;
	}
    
free_super:
    free(super);
    
free_buf:
    free(buf);
    
free_flash:
	flash_dev_release(flash);
    return ret;
}
#endif

static int cramfs_list_inode (struct part_info *info, unsigned long offset)
{
	struct cramfs_inode *inode = (struct cramfs_inode *)
		(PART_OFFSET(info) + offset);
	char *name, str[20];
	int namelen, nextoff;

	/*
	 * Namelengths on disk are shifted by two
	 * and the name padded out to 4-byte boundaries
	 * with zeroes.
	 */
	namelen = CRAMFS_GET_NAMELEN (inode) << 2;
	name = (char *) inode + sizeof (struct cramfs_inode);
	nextoff = namelen;

	for (;;) {
		if (!namelen)
			return namelen;
		if (name[namelen - 1])
			break;
		namelen--;
	}

	printf (" %s %8d %*.*s", mkmodestr (CRAMFS_16 (inode->mode), str),
		CRAMFS_24 (inode->size), namelen, namelen, name);

	if ((CRAMFS_16 (inode->mode) & S_IFMT) == S_IFLNK) {
		/* symbolic link.
		 * Unpack the link target, trusting in the inode's size field.
		 */
		unsigned long size = CRAMFS_24 (inode->size);
		char *link = malloc (size);

		if (link != NULL && cramfs_uncompress ((unsigned long)PART_OFFSET(info), offset,
						       (unsigned long) link)
		    == size)
			printf (" -> %*.*s\n", (int) size, (int) size, link);
		else
			printf (" [Error reading link]\n");
		if (link)
			free (link);
	} else
		printf ("\n");

	return nextoff;
}

int cramfs_ls (struct part_info *info, char *filename)
{
	struct cramfs_inode *inode;
	unsigned long inodeoffset = 0, nextoffset;
	unsigned long offset, size;

	if (cramfs_read_super (info))
		return -1;

	if (strlen (filename) == 0 || !strcmp (filename, "/")) {
		/* Root directory. Use root inode in super block */
		offset = CRAMFS_GET_OFFSET (&(super.root)) << 2;
		size = CRAMFS_24 (super.root.size);
	} else {
		/* Resolve the path */
		offset = cramfs_resolve ((unsigned long)PART_OFFSET(info),
					 CRAMFS_GET_OFFSET (&(super.root)) <<
					 2, CRAMFS_24 (super.root.size), 1,
					 strtok (filename, "/"));

		if (offset <= 0)
			return offset;

		/* Resolving was successful. Examine the inode */
		inode = (struct cramfs_inode *) (PART_OFFSET(info) + offset);
		if (!S_ISDIR (CRAMFS_16 (inode->mode))) {
			/* It's not a directory - list it, and that's that */
			return (cramfs_list_inode (info, offset) > 0);
		}

		/* It's a directory. List files within */
		offset = CRAMFS_GET_OFFSET (inode) << 2;
		size = CRAMFS_24 (inode->size);
	}

	/* List the given directory */
	while (inodeoffset < size) {
		inode = (struct cramfs_inode *) (PART_OFFSET(info) + offset +
						 inodeoffset);

		nextoffset = cramfs_list_inode (info, offset + inodeoffset);
		if (nextoffset == 0)
			break;
		inodeoffset += sizeof (struct cramfs_inode) + nextoffset;
	}

	return 1;
}

int cramfs_info (struct part_info *info)
{
	if (cramfs_read_super (info))
		return 0;

	printf ("size: 0x%x (%u)\n", super.size, super.size);

	if (super.flags != 0) {
		printf ("flags:\n");
		if (super.flags & CRAMFS_FLAG_FSID_VERSION_2)
			printf ("\tFSID version 2\n");
		if (super.flags & CRAMFS_FLAG_SORTED_DIRS)
			printf ("\tsorted dirs\n");
		if (super.flags & CRAMFS_FLAG_HOLES)
			printf ("\tholes\n");
		if (super.flags & CRAMFS_FLAG_SHIFTED_ROOT_OFFSET)
			printf ("\tshifted root offset\n");
	}

	printf ("fsid:\n\tcrc: 0x%x\n\tedition: 0x%x\n",
		super.fsid.crc, super.fsid.edition);
	printf ("name: %16s\n", super.name);

	return 1;
}

int cramfs_check (struct part_info *info)
{
	struct cramfs_super *sb;

	if (info->dev->id->type != MTD_DEV_TYPE_NOR)
		return 0;

	sb = (struct cramfs_super *) PART_OFFSET(info);
	if (sb->magic != CRAMFS_32 (CRAMFS_MAGIC)) {
		/* check at 512 byte offset */
		sb = (struct cramfs_super *) (PART_OFFSET(info) + 512);
		if (sb->magic != CRAMFS_32 (CRAMFS_MAGIC))
			return 0;
	}
	return 1;
}
