/*=======================================================================
?Copyright (C) Microsoft Corporation.?All rights reserved.
?
  THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
  PARTICULAR PURPOSE.
=======================================================================*/

using System;
using System.Globalization;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Web;
using System.Diagnostics;

namespace IPSD.WebServer
{
    internal class Connection
    {
        #region Private Fields

        private Host _host;
        private Socket _socket;
        private Request currentRequest;
        bool keepAlive = false;
        private static int s_ActiveCount = 0;
        private static object s_SyncObj = new object();

        #endregion

        #region Constructor

        public Connection(Host host, Socket socket)
        {
            this._host = host;
            this._socket = socket;
            CountConections(1);
        }

        #endregion

        #region Properties

        public bool Connected
        {
            get
            {
                return this._socket.Connected;
            }
        }

        public bool IsLocal
        {
            get
            {
                return (this.LocalIP == this.RemoteIP);
            }
        }

        public string LocalIP
        {
            get
            {
                IPEndPoint localEndPoint = (IPEndPoint)this._socket.LocalEndPoint;
                if ((localEndPoint != null) && (localEndPoint.Address != null))
                {
                    return localEndPoint.Address.ToString();
                }
                return "127.0.0.1";
            }
        }

        public string RemoteIP
        {
            get
            {
                IPEndPoint remoteEndPoint = (IPEndPoint)this._socket.RemoteEndPoint;
                if ((remoteEndPoint != null) && (remoteEndPoint.Address != null))
                {
                    return remoteEndPoint.Address.ToString();
                }
                return "127.0.0.1";
            }
        }

        public string RemoteEndPoint
        {
            get
            {
                IPEndPoint remoteEndPoint = (IPEndPoint)this._socket.RemoteEndPoint;
                if ((remoteEndPoint != null) && (remoteEndPoint.Address != null))
                {
                    return remoteEndPoint.ToString();
                }
                return "127.0.0.1";
            }
        }

        public bool KeepAlive
        {
            get
            {
                keepAlive = false;
                if (currentRequest != null)
                {
                    string keepAliveValue = this.currentRequest.GetKnownRequestHeader(3);
                    if (!string.IsNullOrEmpty(keepAliveValue) && keepAliveValue.Equals("true", StringComparison.OrdinalIgnoreCase))
                    {
                        keepAlive = true;
                    }
                    else
                    {
                        keepAlive = false;
                    }
                }
                return keepAlive;
            }
            set
            {
                keepAlive = value;
            }
        }

        #endregion

        #region Private helper functions

        private static void CountConections(int increase)
        {
            lock (s_SyncObj)
            {
                s_ActiveCount += increase;
                WebEventLog.GetInstance().WriteInfomation(string.Format("\r\n IPSD Web Server \r\n Connection Count: {0}", s_ActiveCount.ToString()));
            }
        }

        #endregion

        #region System Methods

        /// <summary>
        /// Close the web connction
        /// </summary>
        public void Close()
        {
            if (!KeepAlive)
            {

                try
                {
                    if (this.currentRequest != null)
                    {
                        this.currentRequest = null;
                    }
                    this._socket.Shutdown(SocketShutdown.Both);

                    this._socket.Close();
                    CountConections(-1);

                }
                catch (Exception ex)
                {
                   // WebEventLog.GetInstance().WriteException("Failed to close connection", ex);
                }
                finally
                {
                    this._socket = null;
                }
            }
        }

        public void CloseAll()
        {
            try
            {
                if (this.currentRequest != null)
                {
                    this.currentRequest = null;
                }
                this._socket.Shutdown(SocketShutdown.Both);
                this._socket.Close();
                CountConections(-1);
            }
            catch (Exception ex)
            {
                //WebEventLog.GetInstance().WriteException("Failed to close keep-alive connection", ex);
            }
            finally
            {
                this._socket = null;
            }
        }

        /// <summary>
        /// Set the "Content-Type" head in response
        /// </summary>
        /// <param name="fileName">fileName</param>
        /// <returns>Content-Type Header</returns>
        private static string MakeContentTypeHeader(string fileName)
        {
            string str = null;
            int startIndex = fileName.LastIndexOf('.');
            if (startIndex >= 0)
            {
                string str3 = fileName.Substring(startIndex);
                if (str3 != null)
                {
                    if (!(str3 == ".js"))
                    {
                        if (str3 == ".gif")
                        {
                            str = "image/gif";
                        }
                        else if (str3 == ".jpg")
                        {
                            str = "image/jpeg";
                        }
                    }
                    else
                    {
                        str = "application/x-javascript";
                    }
                }
            }
            if (str == null)
            {
                return null;
            }
            return ("Content-Type: " + str + "\r\n");
        }

        /// <summary>
        /// Set the headers in the response
        /// </summary>
        /// <param name="statusCode">statusCode</param>
        /// <param name="moreHeaders">moreHeaders</param>
        /// <param name="contentLength">contentLength</param>
        /// <param name="keepAlive">keepAlive</param>
        /// <returns></returns>
        private static string MakeResponseHeaders(int statusCode, string moreHeaders, int contentLength, bool keepAlive)
        {
            StringBuilder builder = new StringBuilder();
            builder.Append(string.Concat(new object[] { "HTTP/1.1 ", statusCode, " ", HttpWorkerRequest.GetStatusDescription(statusCode), "\r\n" }));
            builder.Append("Server: IPSD -Web Server/" + Messages.VersionString + "\r\n");
            builder.Append("Date: " + DateTime.Now.ToUniversalTime().ToString("R", DateTimeFormatInfo.InvariantInfo) + "\r\n");
            //if (!moreHeaders.ToLower().Contains("content-length:") && contentLength >= 0)
            //{
            //    builder.Append("Content-Length: " + contentLength + "\r\n");
            //}
            if (moreHeaders != null)
            {
                builder.Append(moreHeaders);
            }
            if (!keepAlive)
            {
                builder.Append("Connection: Close\r\n");
            }
            builder.Append("\r\n");
            return builder.ToString();
        }

        /// <summary>
        /// Set the headers in the PSIA response
        /// </summary>
        /// <param name="statusCode">statusCode</param>
        /// <param name="moreHeaders">moreHeaders</param>
        /// <param name="contentLength">contentLength</param>
        /// <param name="keepAlive">keepAlive</param>
        /// <returns></returns>
        private static string MakePSIAResponseHeaders(int statusCode, string moreHeaders, int contentLength)
        {
            StringBuilder builder = new StringBuilder();
            builder.Append(string.Concat(new object[] { "HTTP/1.1 ", statusCode, " ", HttpWorkerRequest.GetStatusDescription(statusCode), "\r\n" }));
            builder.Append("Server: IPSD -Web Server/" + Messages.VersionString + "\r\n");
            builder.Append("Date: " + DateTime.Now.ToUniversalTime().ToString("R", DateTimeFormatInfo.InvariantInfo) + "\r\n");

            if (moreHeaders != null)
            {
                builder.Append(moreHeaders);
            }

            builder.Append("Connection: Close\r\n");

            builder.Append("\r\n");
            return builder.ToString();
        }

        /// <summary>
        /// Process web request
        /// </summary>
        public void ProcessOneRequest()
        {
            if (this.WaitForRequestBytes() == 0)
            {
                this.WriteErrorAndClose(400);
            }
            else
            {
                currentRequest = new Request(this._host, this);
                currentRequest.Process();
            }
        }
        
        /// <summary>
        /// Read /Get request by the array of byte 
        /// </summary>
        /// <param name="maxBytes">maxBytes</param>
        /// <returns>The array of byte </returns>
        public byte[] ReadRequestBytes(int maxBytes)
        {
            try
            {
                if (this.WaitFoAllrRequestBytes() == 0)
                {
                    return null;
                }
                int available = this._socket.Available;
                if (available > maxBytes)
                {
                    available = maxBytes;
                }
                int count = 0;
                byte[] buffer = new byte[available];
                if (available > 0)
                {
                    count = this._socket.Receive(buffer, 0, available, SocketFlags.None);
                }
                if (count < available)
                {
                    byte[] dst = new byte[count];
                    if (count > 0)
                    {
                        Buffer.BlockCopy(buffer, 0, dst, 0, count);
                    }
                    buffer = dst;
                }
                return buffer;
            }
            catch
            {
                return null;
            }
        }

        private int WaitForRequestBytes()
        {
            int available = 0;
            string msg;
            try
            {
                if (this._socket.Available == 0)
                {
                    //this._socket.Poll(0x186a0, SelectMode.SelectRead);
                    this._socket.Poll(100000 /* 100ms */, SelectMode.SelectRead);
                    if ((this._socket.Available == 0) && this._socket.Connected)
                    {
                        this._socket.Poll(5000000 /* 5sec */, SelectMode.SelectRead);

                    }
                }
                available = this._socket.Available;

            }
            catch
            {
            }
            return available;
        }

        private int WaitFoAllrRequestBytes()
        {
            int available = 0;
            string msg;
            try
            {
                if (this._socket.Available == 0)
                {
                    //this._socket.Poll(0x186a0, SelectMode.SelectRead);
                    this._socket.Poll(100000 /* 100ms */, SelectMode.SelectRead);
                    if ((this._socket.Available == 0) && this._socket.Connected)
                    {
                        //this._socket.Poll(0x989680, SelectMode.SelectRead);

                        this._socket.Poll(5000000 /* 5sec */, SelectMode.SelectRead);

                    }
                }
                available = this._socket.Available;

            }
            catch
            {
            }
            return available;
        }

        public void Write100Continue()
        {
            this.WriteEntireResponseFromString(100, null, null, true);
        }

        /// <summary>
        /// Write the response body
        /// </summary>
        /// <param name="data">Response Data</param>
        /// <param name="offset"></param>
        /// <param name="length">Length</param>
        public void WriteBody(byte[] data, int offset, int length)
        {
            try
            {
                this._socket.Send(data, offset, length, SocketFlags.None);
            }
            catch (Exception ex)
            {
                this.KeepAlive = false;
                if (this.currentRequest != null)
                {
                    this.currentRequest.CloseConnection();
                    this.currentRequest = null;
                }
                WebEventLog.GetInstance().WriteException("Failed to write body. Body Content:\r\n" + Encoding.UTF8.GetString(data), ex);
            }
        }

        /// <summary>
        /// Write the entire response from the request file
        /// </summary>
        /// <param name="fileName">File Name</param>
        /// <param name="keepAlive">whether keep-alive request</param>
        public void WriteEntireResponseFromFile(string fileName, bool keepAlive)
        {
            if (!System.IO.File.Exists(fileName))
            {
                this.WriteErrorAndClose(0x194);
            }
            else
            {
                bool flag = false;
                FileStream stream = null;
                try
                {
                    stream = new FileStream(fileName, FileMode.Open, FileAccess.Read, FileShare.Read);
                    int length = (int)stream.Length;
                    byte[] buffer = new byte[length];
                    int contentLength = stream.Read(buffer, 0, length);
                    string s = MakeResponseHeaders(200, MakeContentTypeHeader(fileName), contentLength, keepAlive);
                    this._socket.Send(Encoding.UTF8.GetBytes(s));
                    this._socket.Send(buffer, 0, contentLength, SocketFlags.None);
       
                    flag = true;
                }
                catch (Exception ex)
                {
                    WebEventLog.GetInstance().WriteException("Failed to write the entire response:", ex);
                }
                finally
                {
                    if (!(keepAlive && flag))
                    {
                        this.Close();
                    }
                    if (stream != null)
                    {
                        stream.Close();
                    }
                }
            }
        }

        /// <summary>
        /// Write the PSIA response
        /// </summary>
        /// <param name="psiaResponse">psia response content</param>
        /// <param name="keepAlive">whether is keep-alive request</param>
        public void WritePSIAResponse(string psiaResponse, bool keepAlive)
        {
            if (string.IsNullOrEmpty(psiaResponse))
            {
                this.WriteErrorAndClose(0x194);
            }
            else
            {
                bool flag = false;
                try
                {

                    byte[] buffer = System.Text.Encoding.UTF8.GetBytes(psiaResponse);
                    int length = buffer.Length;
                    int contentLength = psiaResponse.Length;
                    string s = MakePSIAResponseHeaders(200, "Content-Type: text/plain\r\nContent-Length:" + contentLength.ToString() + "\r\n", contentLength);
                    this._socket.Send(Encoding.UTF8.GetBytes(s));

                    this._socket.Send(buffer, 0, contentLength, SocketFlags.None);

                    flag = true;

                }
                catch (Exception ex)
                {
                    WebEventLog.GetInstance().WriteException("Failed to write PSIA Request:\r\n PSIA Content:" + psiaResponse, ex);
                }
                finally
                {
                    this.CloseAll();
                }
            }
        }

        /// <summary>
        /// Write entire response from string content
        /// </summary>
        /// <param name="statusCode">Status Code</param>
        /// <param name="extraHeaders">Extra Headers</param>
        /// <param name="body">response body</param>
        /// <param name="keepAlive">whether is keep-alive request</param>
        public void WriteEntireResponseFromString(int statusCode, string extraHeaders, string body, bool keepAlive)
        {
            try
            {
                int contentLength = (body != null) ? Encoding.UTF8.GetByteCount(body) : 0;
                string str = MakeResponseHeaders(statusCode, extraHeaders, contentLength, keepAlive);
                string b = str + body;
                this._socket.Send(Encoding.UTF8.GetBytes(b));

            }
            catch (SocketException e)
            {
                this.CloseAll();
                WebEventLog.GetInstance().WriteException("Fail to write response \r\n Content:" + body, e);
            }
            finally
            {
                if (statusCode != 100)
                {
                    this.Close();
                }
            }
        }

        /// <summary>
        /// Write error code and close connection
        /// </summary>
        /// <param name="statusCode"></param>
        public void WriteErrorAndClose(int statusCode)
        {
            this.WriteErrorAndClose(statusCode, null);
        }

        /// <summary>
        /// Write error code and close connection
        /// </summary>
        /// <param name="statusCode"></param>
        public void WriteErrorAndClose(int statusCode, string message)
        {
            string body = Messages.FormatErrorMessageBody(statusCode, this._host.VirtualPath);
            if ((message != null) && (message.Length > 0))
            {
                body = body + "\r\n<!--\r\n" + message + "\r\n-->";
            }
            this.WriteEntireResponseFromString(statusCode, null, body, false);
        }

        /// <summary>
        /// Write the response headers
        /// </summary>
        /// <param name="statusCode">Status Code</param>
        /// <param name="extraHeaders">Extra Headers</param>
        public void WriteHeaders(int statusCode, string extraHeaders)
        {
            try
            {
                string s = MakeResponseHeaders(statusCode, extraHeaders, -1, false);
                this._socket.Send(Encoding.UTF8.GetBytes(s));
            }
            catch (SocketException)
            {
            }
        }

        #endregion

    }
}
