/*=======================================================================
?Copyright (C) Microsoft Corporation.?All rights reserved.
?
  THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
  PARTICULAR PURPOSE.
=======================================================================*/

using Microsoft.Win32;
using System;
using System.Diagnostics;
using System.Threading;
using System.Web;
using System.Web.Hosting;
using System.Collections.Generic;
using System.Collections.ObjectModel;

namespace IPSD.WebServer
{
    /// <summary>
    /// The web request data changed notification handler
    /// </summary>
    public delegate void DataChangedNotificationHandler();

    /// <summary>
    /// The client message send handler
    /// </summary>
    /// <param name="message">The client message</param>
    /// <returns>Callback message content</returns>
    public delegate string SendMessageToClientHandler(string message);

    /// <summary>
    /// The handler for sending PSIA request
    /// </summary>
    /// <param name="psiaRequest">The specified PSIA Request</param>
    /// <returns></returns>
    public delegate string SendPSIARequestHander(PSIARequest psiaRequest);

    internal class Server : MarshalByRefObject
    {
        #region Private Fields

        private Host _host;
        private string _installPath;
        private string _physicalPath;
        private int _port;
        public WaitCallback _restartCallback;
        private string _virtualPath;
        private int keepAliveInterval;

        #endregion

        #region External Delegate/Event

        /// <summary>
        /// The notification event for web client data changed.
        /// </summary>
        public event DataChangedNotificationHandler DataChangedNotificationEvent;

        /// <summary>
        /// The event for send message for WinForm Client
        /// </summary>
        public event SendMessageToClientHandler SendMessageToClientEvent;

        /// <summary>
        /// The event for sending PSIA request 
        /// </summary>
        public event SendPSIARequestHander SendPSIARequestEvent;

        #endregion

        #region Properties

        public string InstallPath
        {
            get
            {
                return this._installPath;
            }
        }

        public string PhysicalPath
        {
            get
            {
                return this._physicalPath;
            }
        }

        public int Port
        {
            get
            {
                return this._port;
            }
        }

        public string RootUrl
        {
            get
            {
                return ("http://localhost:" + this._port + this._virtualPath);
            }
        }

        public string VirtualPath
        {
            get
            {
                return this._virtualPath;
            }
        }

        #endregion

        #region System Methods

        public Server(int port, string virtualPath, string physicalPath,int keepAliveInterval)
        {
            this._port = port;
            this._virtualPath = virtualPath;
            this.keepAliveInterval = keepAliveInterval;
            this._physicalPath = physicalPath.EndsWith(@"\") ? physicalPath : (physicalPath + @"\");
            this._restartCallback = new WaitCallback(this.RestartCallback);
            this._installPath = this.GetInstallPathAndConfigureAspNetIfNeeded();
            this.CreateHost();
        }

        private void CreateHost()
        {
            this._host = (Host)ApplicationHost.CreateApplicationHost(typeof(Host), this._virtualPath, this._physicalPath);
            this._host.Configure(this, this._port, this._virtualPath, this._physicalPath, this._installPath,this.keepAliveInterval);
        }

        private string GetInstallPathAndConfigureAspNetIfNeeded()
        {
            RegistryKey key = null;
            RegistryKey key2 = null;
            RegistryKey key3 = null;
            string str = null;
            try
            {
                FileVersionInfo versionInfo = FileVersionInfo.GetVersionInfo(typeof(HttpRuntime).Module.FullyQualifiedName);
                string str2 = string.Format("{0}.{1}.{2}.{3}", new object[] { versionInfo.FileMajorPart, versionInfo.FileMinorPart, versionInfo.FileBuildPart, versionInfo.FilePrivatePart });
                string name = @"Software\Microsoft\ASP.NET\" + str2;
                if (!str2.StartsWith("1.0."))
                {
                    name = name.Substring(0, name.LastIndexOf('.') + 1) + "0";
                }
                key2 = Registry.LocalMachine.OpenSubKey(name);
                if (key2 != null)
                {
                    return (string)key2.GetValue("Path");
                }
                key = Registry.LocalMachine.OpenSubKey(@"Software\Microsoft\ASP.NET");
                if (key == null)
                {
                    key = Registry.LocalMachine.CreateSubKey(@"Software\Microsoft\ASP.NET");
                    key.SetValue("RootVer", str2);
                }
                string str4 = "v" + str2.Substring(0, str2.LastIndexOf('.'));
                key3 = Registry.LocalMachine.OpenSubKey(@"Software\Microsoft\.NETFramework");
                string str5 = (string)key3.GetValue("InstallRoot");
                if (str5.EndsWith(@"\"))
                {
                    str5 = str5.Substring(0, str5.Length - 1);
                }
                key2 = Registry.LocalMachine.CreateSubKey(name);
                str = str5 + @"\" + str4;
                key2.SetValue("Path", str);
                key2.SetValue("DllFullPath", str + @"\aspnet_isapi.dll");
                return str;
            }
            catch (Exception)
            { 
              //Nothing to do
            }
            finally
            {
                if (key2 != null)
                {
                    key2.Close();
                }
                if (key != null)
                {
                    key.Close();
                }
                if (key3 != null)
                {
                    key3.Close();
                }
            }
            return str;
        }

        public override object InitializeLifetimeService()
        {
            return null;
        }

        public void Restart()
        {
            ThreadPool.QueueUserWorkItem(this._restartCallback);
        }

        private void RestartCallback(object unused)
        {
            this.Stop();
            this.CreateHost();
            this.Start();
        }

        public void Start()
        {
            if (this._host != null)
            {
                try
                {
                    this._host.Start();
                    WebEventLog.GetInstance().WriteInfomation("IPSD Web Server Started: \r\n PhysicalPath:" + this._physicalPath + "\r\n Port:" + this._port.ToString());
                }
                catch (Exception ex)
                {
                    WebEventLog.GetInstance().WriteException(ex);
                }
            }
        }

        public void Stop()
        {
            if (this._host != null)
            {
                try
                {
                    this._host.Stop();
                    WebEventLog.GetInstance().WriteInfomation("IPSD Web Server Stopped: \r\n PhysicalPath:" + this._physicalPath + "\r\n Port:" + this._port.ToString());
                }
                catch(Exception ex)
                {
                    WebEventLog.GetInstance().WriteException(ex);
                }
            }
        }

        #endregion

        #region External APIs

        /// <summary>
        /// Notify the host client data changed
        /// </summary>
        public void DataChangedNotification()
        {
            if (DataChangedNotificationEvent != null)
            {
                this.DataChangedNotificationEvent();
            }
        }

        /// <summary>
        /// Send message to host client by http web request
        /// </summary>
        /// <param name="message">The specified message</param>
        /// <returns>Callback Message</returns>
        public string SendMessageToClient(string message)
        {
            string returnValue = string.Empty;
            if ( SendMessageToClientEvent != null)
            {
                returnValue = SendMessageToClientEvent(message);
            }
            return returnValue;
        }

        /// <summary>
        /// Send the psia request 
        /// </summary>
        /// <param name="paisRequest">The specified PSIARequest</param>
        /// <returns>Callback Message</returns>
        public string SendPSIARequest(PSIARequest paisRequest)
        {
            string returnValue = string.Empty;
            if (SendPSIARequestEvent != null)
            {
                returnValue = SendPSIARequestEvent(paisRequest);
            }
            return returnValue;
        }

        #endregion

    }
}
