/*
 *
 * @APPLE_LICENSE_HEADER_START@
 *
 * Copyright (c) 1999-2008 Apple Inc.  All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this
 * file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 *
 */
/*
    File:       ClientSession.cpp

    Contains:   .  
                    
    
    
*/

//#include <arpa/inet.h>
#include <stdlib.h>
#include "ClientSession.h"
#include "OSMemory.h"
#include "SafeStdLib.h"
#include "OSHeaders.h"
#include "OS.h"
#include "RTPPacket.h"
#include "RTCPPacket.h"
#include "RTCPRRPacket.h"
#include "RTCPAckPacketFmt.h"
#include "RTCPNADUPacketFmt.h"

//////////////////////////////////////////////////////////////////////////
// The following code is added by Henry CHEN for testing

#include "StreamingClientImp.h"

#include "PracticalSocket.h"
#include "ext_string.h"
# define CLIENTSESSION_LOG_DEBUG 0
#if CLIENTSESSION_LOG_DEBUG
class log_print
{
public:
    log_print(char * str){strcpy(x, str); qtss_txprintf("->log_print Enter: %s\n", x);}
    ~log_print(){qtss_txprintf("->log_print Leave: %s\n", x);}
    char x[MAX_PATH];
};
#endif
// Henry code end
//////////////////////////////////////////////////////////////////////////

//These two parameters governs how the client determines whether an incoming RTP packet is within the range of the sequence number or not.
enum {
    kMaxDropOut = 3000,             //The sequence number can be kMaxDropOut ahead of the reference.
    kMaxMisorder = 100,             //The sequence number can be kMaxMisorder behind of the reference.
    kMaxUDPPacketSize = 1450
};

#define CLIENT_SESSION_DEBUG 1

static const SInt64 kMaxWaitTimeInMsec = 200;
static const SInt64 kIdleTimeoutInMsec = 20000; // Time out in 20 seconds if nothing's doing
static const SInt16 kSanitySeqNumDifference = 3000;


UInt32          ClientSession::sActiveConnections = 0;
UInt32          ClientSession::sPlayingConnections = 0;
UInt32          ClientSession::sTotalConnectionAttempts = 0;

UInt32          ClientSession::sBytesReceived = 0;
UInt32          ClientSession::sPacketsReceived = 0;

char* ConvertBytesToCHexString( void* inValue, const UInt32 inValueLen)
{
    static const char* kHEXChars={ "0123456789ABCDEF" };

    UInt8* theDataPtr = (UInt8*) inValue;
    UInt32 len = inValueLen *2;
    
    char *theString = NEW char[len+1];
    char *resultStr = theString;
    if (theString != NULL)
    {
        UInt8 temp;
        UInt32 count = 0;
        for (count = 0; count < inValueLen; count++)
        {
            temp = *theDataPtr++;
            *theString++ = kHEXChars[temp >> 4];
            *theString++ = kHEXChars[temp & 0xF];
        }
        *theString = 0;
    }
    return resultStr;
}

// Modify by Ding Hai Tao 2011-5-18: 
// 1.The Static/Global lock is not suitable
// 2.Move this lock to StreamingClientImp
// HUS::kernel_mutex<FALSE> ClientSession::GMutex;

#define PORTSNUM 4
const static int g_Ports[] = {554, 7070, 8000, 8001};
ClientSession::ClientSession(   UInt32 inAddr, UInt16 inPort, char* inURL,
                                ClientType inClientType,
                                UInt32 inDurationInSec, UInt32 inStartPlayTimeInSec,
                                UInt32 inRTCPIntervalInMS, UInt32 inOptionsIntervalInSec,
                                UInt32 inHTTPCookie, Bool16 inAppendJunkData, UInt32 inReadInterval,
                                UInt32 inSockRcvBufSize, Float32 inLateTolerance, char* inMetaInfoFields,
                                Float32 inSpeed, UInt32 verboseLevel, char* inPacketRangePlayHeader, UInt32 inOverbufferWindowSizeInK,
                                Bool16 sendOptions, Bool16 requestRandomData, SInt32 randomDataSize, Bool16 enable3GPP,
                                UInt32 GBW, UInt32 MBW, UInt32 MTD, Bool16 enableForcePlayoutDelay, UInt32 playoutDelay, 
								UInt32 bandwidth, UInt32 bufferSpace, UInt32 delayTime, UInt32 startPlayDelay,
                                char *controlID, char *name, char *password,
//////////////////////////////////////////////////////////////////////////
// The following code added by Henry CHEN, HTSL
								StreamingClientImp *streamingClient,
								Float32 inScale, UInt32 inCmdType
// Henry CHEN code end 
//////////////////////////////////////////////////////////////////////////
                                )
:   fSocket(NULL),
    fClient(NULL),
    fTimeoutTask(this,0/* kIdleTimeoutInMsec*/),

    fDurationInSec(inDurationInSec),
    fStartPlayTimeInSec(inStartPlayTimeInSec),
    fRTCPIntervalInMs(inRTCPIntervalInMS),
    fOptionsIntervalInSec(inOptionsIntervalInSec),
    
    fOptions(sendOptions),
    fOptionsRequestRandomData(requestRandomData),
    fOptionsRandomDataSize(randomDataSize),
    fTransactionStartTimeMilli(0),
	fState(kSendingOptions),
	fCmdType(inCmdType),
    fDeathReason(kDiedNormally),
    fNumSetups(0),
    fUDPSocketArray(NULL),
    
    fPlayTime(0),
    fTotalPlayTime(0),
    fLastRTCPTime(0),
    fTeardownImmediately(false),
    fAppendJunk(inAppendJunkData),
    fReadInterval(inReadInterval),
    fSockRcvBufSize(inSockRcvBufSize),
    
    fSpeed(inSpeed),
    fPacketRangePlayHeader(inPacketRangePlayHeader),
	
    fGuarenteedBitRate(GBW),
    fMaxBitRate(MBW),
    fMaxTransferDelay(MTD),
	fEnableForcePlayoutDelay(enableForcePlayoutDelay),
	fPlayoutDelay(playoutDelay),
    fBandwidth(bandwidth),
	fBufferSpace(bufferSpace),
	fDelayTime(delayTime),
	fStartPlayDelay(startPlayDelay),
	fEnable3GPP(enable3GPP),
	
    //fStats(NULL),
    fOverbufferWindowSizeInK(inOverbufferWindowSizeInK),
    fCurRTCPTrack(0),
    fNumPacketsReceived(0),
	fNumBytesReceived(0),
    fVerboseLevel(verboseLevel),
	fPlayerSimulator(verboseLevel)
//////////////////////////////////////////////////////////////////////////
// The following code added by Henry CHEN, HTSL
    , fStreamingClient(streamingClient), m_bTearDowned(false)
	, m_bReConnect(true),m_bHistorical(false),m_nStartTimePos(0),
// Henry CHEN code end 
//////////////////////////////////////////////////////////////////////////
	fScale(inScale),
    m_InitLocker(TRUE, FALSE, NULL),
    fStatus(RTSP_OK),
    m_StatusLocker(TRUE, TRUE, NULL),
    fPacketSeq(1),
	current_rtsp_state_(NULL)
{
	if(theURL.Len != 0)
		delete []theURL.Ptr;
	if(inURL != NULL)
	{
		theURL.Ptr = NEW char[strlen(inURL)+1];
		memset(theURL.Ptr,0,strlen(inURL) +1);
		strcpy(theURL.Ptr,inURL);
		theURL.Len = strlen(inURL);

		//real streamer relay		rtsp://192.168.250.50/{guid}/0
		//historical play			rtsp://192.168.250.50/{guid}/SessionID?user=xxx&download=0
		//mannual record			rtsp://192.168.250.50/{guid}/0

		//historical play
		if(theURL.FindString("?"))
		{
			m_bHistorical = true;

			//TOTO,when reconnecting,replace start time and len
			m_tmStart =CTime(static_cast<time_t>( fStartPlayTimeInSec));
			m_tmEnd = CTime(static_cast<time_t>(fStartPlayTimeInSec + fDurationInSec));
		}
	}

	fInClientType = inClientType;
	fInHTTPCookie = inHTTPCookie;
	fInAddr = inAddr;
	fInPort = inPort;
	fInLateTolerance = inLateTolerance;
	
	if(fInMetaInfoFields.Len != 0)
		fInMetaInfoFields.Delete();
	if(inMetaInfoFields != NULL)
	{
		fInMetaInfoFields.Ptr = NEW char[strlen(inMetaInfoFields)+1];
		memset(fInMetaInfoFields.Ptr,0,strlen(inMetaInfoFields) +1);
		strcpy(fInMetaInfoFields.Ptr,inMetaInfoFields);
		fInMetaInfoFields.Len = strlen(inMetaInfoFields);
	}
	
	if(fControlID.Len != 0)
		fControlID.Delete();
	if(controlID != NULL)
	{
		fControlID.Ptr = NEW char[strlen(controlID)+1];
		memset(fControlID.Ptr,0,strlen(controlID) +1);
		strcpy(fControlID.Ptr,controlID);
		fControlID.Len = strlen(controlID);
	}
	
	if(fName.Len != 0)
		fName.Delete();
	if(name != NULL)
	{
		fName.Ptr = NEW char[strlen(name)+1];
		memset(fName.Ptr,0,strlen(name) + 1);
		strcpy(fName.Ptr,name);
		fName.Len = strlen(name);
	}
	
	if(fPassword.Len != 0)
		fPassword.Delete();
	if(password != NULL)
	{
		fPassword.Ptr = NEW char[strlen(password) + 1];
		memset(fPassword.Ptr,0,strlen(password) + 1);
		strcpy(fPassword.Ptr,password);
		fPassword.Len = strlen(password);
	}

    this->SetTaskName("RTSPClientLib:ClientSession");
	
	//Init();

	m_nTimeMilliSecond = 0;
	m_nLastMilliSecond = 0;
    // Start the connection process going
   // this->Signal(Task::kStartEvent);
}

void ClientSession::Init( )
{
    fPacketSeq = 1;
    _ClientLocker.UnlockAll();
    m_StatusLocker.do_unlock();
    m_InitLocker.Reset();
    fStatus = RTSP_OK;
	fTransactionStartTimeMilli = 0;
	fNumSetups = 0;
	fPlayTime = 0;
	fTotalPlayTime = 0;
	fLastRTCPTime = 0;
	fCurRTCPTrack = 0;
	fNumPacketsReceived = 0;
	fNumBytesReceived = 0;

	fTimeoutTask.SetTimeout(0);

    if (fOptions)
       fState = kSendingOptions;
    
#if CLIENT_SESSION_DEBUG
    fVerboseLevel = kUInt32_Max;  //maximum possible unsigned int value in 2's complement
#endif

	char szHostName[256];
	gethostname(szHostName,256);
	struct   hostent   *   pHost;       
	pHost   =   gethostbyname(szHostName);  	
	qtss_printf("ClientSession::this=%ld,StreamingClient=%ld,IP=%u.%u.%u.%u,URL=%s\n", this, fStreamingClient,
		(unsigned   int)((unsigned char)pHost->h_addr_list[0][0]),
		(unsigned   int)((unsigned char)pHost->h_addr_list[0][1]),
		(unsigned   int)((unsigned char)pHost->h_addr_list[0][2]),
		(unsigned   int)((unsigned char)pHost->h_addr_list[0][3]),
		theURL.Ptr);

    //
    // Construct the appropriate ClientSocket type depending on what type of client we are supposed to be
    switch (fInClientType)
    {
        case kRTSPUDPClientType:
        {
            fControlType = kRawRTSPControlType;
            fTransportType = kUDPTransportType;
            fSocket = NEW TCPClientSocket(Socket::kNonBlockingSocketType);
            break;
        }
        case kRTSPTCPClientType:
        {
            fControlType = kRawRTSPControlType;
            fTransportType = kTCPTransportType;
            fSocket = NEW TCPClientSocket(Socket::kNonBlockingSocketType);
            break;
        }
        case kRTSPHTTPClientType:
        {
            fControlType = kRTSPHTTPControlType;
            fTransportType = kTCPTransportType;
            fSocket = NEW HTTPClientSocket(theURL, fInHTTPCookie, Socket::kNonBlockingSocketType);
            break;
        }
        case kRTSPHTTPDropPostClientType:
        {
            fControlType = kRTSPHTTPDropPostControlType;
            fTransportType = kTCPTransportType;
            fSocket = NEW HTTPClientSocket(theURL, fInHTTPCookie, Socket::kNonBlockingSocketType);
            break;
        }
        case kRTSPReliableUDPClientType:
        {
            fControlType = kRawRTSPControlType;
            fTransportType = kReliableUDPTransportType;
            fSocket = NEW TCPClientSocket(Socket::kNonBlockingSocketType);
            break;
        }
        default:
        {
            qtss_printf("ClientSession: Attempt to create unsupported client type.\n");
            //::exit(-1);
            Assert(FALSE);
        }
    }

	if(fSocket == NULL)
		return;
    int nIndex = rand()%PORTSNUM;
    fInPort =g_Ports[nIndex];
    fSocket->Set(fInAddr, fInPort);
    
    //
    // Construct the client object using this socket.
    fClient = NEW RTSPClient(fSocket);
	if(fClient == NULL)
		return;
    fClient->SetVerboseLevel(fVerboseLevel);
    fClient->Set(theURL);
    fClient->SetSetupParams(fInLateTolerance, fInMetaInfoFields.Ptr);
	fClient->SetBandwidth(fBandwidth);
	if(fControlID.Len != 0)
		fClient->SetControlID(fControlID.Ptr);

    if (fEnable3GPP)
    {
        fClient->Set3GPPLinkChars(fGuarenteedBitRate, fMaxBitRate, fMaxTransferDelay);
	    fClient->Set3GPPRateAdaptation(fBufferSpace, fDelayTime);
    }
    
    //user name and password
    if (fName.Len!= 0 && fPassword.Len != 0)
    {
        fClient->SetName(fName.Ptr);
        fClient->SetPassword(fPassword.Ptr);
    }
	m_nNoErrTime = GetTickCount();
	this->Signal(Task::kStartEvent);
}

ClientSession::~ClientSession()
{
	qtss_printf("ClientSession::~ClientSession,begin this=%ld,IP=?\r\n",this);
    m_InitLocker.do_unlock();
    _ClientLocker.UnlockAll();
	// Modify by Ding Hai Tao 2011-5-18: 
	// 1.The Static/Global lock is not suitable
	// 2.Move this lock to StreamingClientImp
	// 3.Tell StreamingClientImp I am Destroyed
	//_KernalMutex::scoped_lock lock(ClientSession::GMutex);
	{
		_KernalMutex::scoped_lock lock (m_mtxProData);
		if(fStreamingClient != NULL)
		{
			fStreamingClient->SetClientSessionNull();
		}
	}

	if(theURL.Len != 0)
		theURL.Delete();
	if(fInMetaInfoFields.Len != 0)
		fInMetaInfoFields.Delete();
	if(fControlID.Len != 0)
		fControlID.Delete();
	if(fName.Len != 0)
		fName.Delete();
	if(fPassword.Len != 0)
		fPassword.Delete();

    if (fUDPSocketArray != NULL)
    {
        for (UInt32 x = 0; x < fSDPParser.GetNumStreams() * 2; x++)
        {
            OS_Error theErr = OS_NoErr;
            
            while (theErr == OS_NoErr)
            {
                    UInt32 theRemoteAddr = 0;
                    UInt32 theLength = 0;
                    UInt16 theRemotePort = 0;
                    char thePacketBuf[2048];
            
                    // Get a packet from one of the UDP sockets.
                    theErr = fUDPSocketArray[x]->RecvFrom(&theRemoteAddr, &theRemotePort,
                                                                &thePacketBuf[0], 2048,
                                                                &theLength);
            }
            delete fUDPSocketArray[x];
        }
    }

    if ( fUDPSocketArray != NULL )
    {
        delete [] fUDPSocketArray;
        fUDPSocketArray = NULL;
    }

	{
		_KernalMutex::scoped_lock lock(m_ClientLocker);
		if ( fClient != NULL )
		{
			delete fClient;
			fClient = NULL;
		}
	}

    if ( fSocket != NULL )
    {
        delete fSocket;
        fSocket = NULL;
    }	

#if CLIENT_SESSION_DEBUG
   // qtss_printf("Connecting to: %s, port %d\n", inet_ntoa(inAddr), inPort);
	char szHostName[256];
	gethostname(szHostName,256);
	struct   hostent   *   pHost;       
	pHost   =   gethostbyname(szHostName); 
	if(pHost != NULL)
	{
		qtss_printf("ClientSession::~ClientSession this=%ld,IP=%u.%u.%u.%u\n", this, 
			(unsigned   int)((unsigned char)pHost->h_addr_list[0][0]),
			(unsigned   int)((unsigned char)pHost->h_addr_list[0][1]),
			(unsigned   int)((unsigned char)pHost->h_addr_list[0][2]),
			(unsigned   int)((unsigned char)pHost->h_addr_list[0][3]));
	}
	else
	{
		qtss_printf("ClientSession::~ClientSession this=%ld,IP=?",this);
	}
#endif 
}

void ClientSession::TearDown() {
    //OSMutexLocker locker(&fMutex);
	qtss_printf("TearDown LockMutex begin, this = %ld\n",this);
    _KernalMutex::scoped_lock lock (m_mtxProData);

	qtss_printf("TearDown begin, this = %ld\n",this);

	// Modify by Ding Hai Tao 2011-5-18: fStreamingClient->clientSession == NULL is not meaningful
    if(fStreamingClient == NULL /*|| (fStreamingClient != NULL && fStreamingClient->clientSession == NULL)*/)
	{
        return;
	}

	fTimeoutTask.SetTimeout(1 * 1000/*kIdleTimeoutInMsec*/);    
    OutputDebugStringA("TearDown begin\n");
   
    long nSession = (long)fStreamingClient;    

	// Modify by Ding Hai Tao 2011-5-18: fStreamingClient->clientSession == NULL is not meaningful
	//fStreamingClient->clientSession = NULL;
    fStreamingClient = NULL;

    if (fState != kDone) {
        //fState = kSendingTeardown;	
        Signal(ClientSession::kTeardownEvent);
    }
    
    m_bTearDowned = true;

    char ss[100];
    sprintf(ss, "TearDown end, Flag:%d, Session:%d, This:%d\n", (int)m_bTearDowned, (int)nSession, (int)this);
   	qtss_printf(ss);
 
	OutputDebugStringA(ss);
}

// static void dbg_printf(const char *format, ...)
// {
// 	char StrBuf[512];
// 	va_list ap;
// 	va_start(ap, format);
// 	_vsnprintf_s(StrBuf, sizeof(StrBuf), format, ap);
// 	va_end(ap);
// 	OutputDebugStringA(StrBuf);
// }

SInt64 ClientSession::Run()
{
    // Get Events from the ClientSession Events Address
	EventFlags theEvents = this->GetEvents();
#if CLIENTSESSION_LOG_DEBUG
    char s[MAX_PATH] = {0};
    sprintf(s, "this = %d, thread ID = %d, Events = %d, fState = %d\n", this, GetCurrentThreadId(), theEvents, fState);
    log_print logx(s);
#endif
    // kStartEvent Process

    if (theEvents & Task::kStartEvent)
    {
        sActiveConnections++;
        sTotalConnectionAttempts++;
		//Sometimes the clientSession can be told to stop before it has a chance to start the connection
		if (theEvents & ClientSession::kTeardownEvent)
			fTeardownImmediately = true;
		else
		{
			Assert(theEvents == Task::kStartEvent);
			// Determine a random connection interval, and go away until that time comes around.
			// Next time the event received would be Task::kIdleEvent, and the initial state is kSendingDescribe
            qtss_printf("ClientSession::Run() Leave 1...this:%d %d %d\n", this, theEvents, fState);
			return ((UInt32) ::rand()) % kMaxWaitTimeInMsec + 1;
		}
    }

    // kTimeoutEvent Process
    if (theEvents & Task::kTimeoutEvent)
    {
#if CLIENT_SESSION_DEBUG
        qtss_printf("ClientSession:: this = %ld,Session timing out.\n",this);
#endif

        if(fState == kDone)
            return 0;
		_KernalMutex::scoped_lock lock (m_mtxProData);
        if (fStreamingClient)
        {
            qtss_printf("ClientSession:: StreamingClient = %ld,Session timing out.\n",fStreamingClient);

			// Modify by Ding Hai Tao 2011-5-18: fStreamingClient->clientSession == NULL is not meaningful
            fStreamingClient->SetClientSessionNull();
            fStreamingClient = NULL;
        }

		if ( fVerboseLevel >= 2)
        	qtss_printf("Session timing out.\n");
        fDeathReason = kSessionTimedout;
        fState = kDone;
        return -1; //Gavin make change, in case the session time out, delete itself
    }

    // kTeardownEvent Process --- If we've been told to TEARDOWN, do so.
    if (theEvents & ClientSession::kTeardownEvent)
    {
		if ( fVerboseLevel >= 2)
        	qtss_printf("Session tearing down immediately.\n");
        fTeardownImmediately = true;
    }
    
    // kKillEvent Process --- We have been told to delete ourselves. Do so... NOW!!!!!!!!!!!!!!!
    if (theEvents & Task::kKillEvent)
    {
		if ( fVerboseLevel >= 2)
        	qtss_printf("Session killed.\n");
        sActiveConnections--;
        return -1;
    }   
   
    // Refresh the timeout. There is some legit activity going on...
   // fTimeoutTask.RefreshTimeout();
    OS_Error theErr = OS_NoErr;    
    while ((theErr == OS_NoErr) && (fState != kDone))
    {
        // Do the appropriate thing depending on our current state
        switch (fState)
        {
			case kSendingRecord:
			{

                {
                    HUS::mutex::scoped_lock lock(_ClientLocker);
                    fClient->ExchangeFSocket();
                }
				theErr = fClient->SendRecord(fStartPlayTimeInSec, fDurationInSec);
				if(theErr == OS_NoErr)
				{
                    m_nNoErrTime = GetTickCount();
					if ((fStatus =fClient->GetStatus()) != 200)
					{
                        _KernalMutex::scoped_lock lock (m_mtxProData);
                        if(fStreamingClient)
						    fStreamingClient->SendAlarm(STREAMING_SERVER, fStatus, CMD_RECORD, NULL );
						theErr = ENOTCONN; // Exit the state machine
						break;
					}
					fState = kPlaying;
				}
				break;
			}
            case kSendingOptions:
            {
                if(fTeardownImmediately)
                {
                    qtss_printf("ClientSession::Run() kSendingOptions...kSendingTeardown!!!!this:%d\n", this);
                    fState = kSendingTeardown;
                    break;
                }

                {
                    qtss_printf("ClientSession::Run() kSendingOptions..._ClientLocker In this:%d\n", this);
                    HUS::mutex::scoped_lock lock(_ClientLocker);
                    qtss_printf("ClientSession::Run() kSendingOptions..._ClientLocker Out this:%d\n", this);
                    fClient->ExchangeFSocket();
                }
                    if (fOptionsRequestRandomData)
                        theErr = fClient->SendOptionsWithRandomDataRequest(fOptionsRandomDataSize);
                    else
                        theErr = fClient->SendOptions();

                    if ( fVerboseLevel >= 3)
                        qtss_printf("Sent OPTIONS. Result = %"_U32BITARG_". Response code = %"_U32BITARG_"   this: %d\n", theErr, fStatus, this);
                    if (0 == fTransactionStartTimeMilli) 
                       fTransactionStartTimeMilli = OS::Milliseconds();

                if(theErr == ENOTCONN)
                {
                    qtss_printf("Sending Option ENOTCONN, this = %d\n", this);
                    fStatus = RTSP_SERVICE_UNAVALIBLE;
                    {
                        _KernalMutex::scoped_lock lock (m_mtxProData);
                        if(fStreamingClient)
                            fStreamingClient->SendAlarm(STREAMING_SERVER,fStatus, CMD_OPTION, NULL );
                    }
                    m_InitLocker.do_unlock();
                    m_StatusLocker.do_lock();
                }

                if (theErr == OS_NoErr)
                {
                     m_nNoErrTime = GetTickCount();
                    // Check that the OPTIONS response is a RTSP_OK. If not, bail
                    if ((fStatus = fClient->GetStatus()) != RTSP_OK)
                    {
                        _KernalMutex::scoped_lock lock (m_mtxProData);
                        if(fStreamingClient)
						    fStreamingClient->SendAlarm(STREAMING_SERVER,fStatus, CMD_OPTION, NULL );
                        theErr = ENOTCONN; // Exit the state machine
                        m_InitLocker.do_unlock();
                        m_StatusLocker.do_lock();
                        break;
                    }
                    else
                    {
						if ( fVerboseLevel >= 3)
                        {
                            qtss_printf("--- Options transaction time ms = %qd  ---\n", (SInt64) ( OS::Milliseconds() - fTransactionStartTimeMilli) );
                            SInt32 receivedLength = (SInt32) fClient->GetContentLength();
                            if (receivedLength != 0)
                                qtss_printf("--- Options Request Random Data Received requested = %"_S32BITARG_" received = %"_S32BITARG_"  ---\n", fOptionsRandomDataSize, receivedLength);
                        }
                        fState = kSendingDescribe;
                    }
                }
                qtss_printf("ClientSession::Run() kSendingOptions...Break this:%d  theErr: %d\n", this, theErr);
                break;
            }
            case kSendingDescribe:
            {
				if(fCmdType == kSendingRecord)
					theErr = fClient->SendDescribe(0, 0, fAppendJunk);
				else
					theErr = fClient->SendDescribe(fStartPlayTimeInSec, fStartPlayTimeInSec+fDurationInSec, fAppendJunk);

				//dbg_printf("[query][senddescirbe][error:%d]\n", theErr);

				if ( fVerboseLevel >= 3)
                	qtss_printf("Sent DESCRIBE. Result = %"_U32BITARG_". Response code = %"_U32BITARG_",    This: %d\n", theErr,fStatus, this);
                if (theErr == OS_NoErr)
                {
                    m_nNoErrTime = GetTickCount();
                    // Check that the DESCRIBE response is a RTSP_OK. If not, bail
                    if ((fStatus = fClient->GetStatus()) != RTSP_OK)
                    {
                        qtss_printf("ClientSession::Run() kSendDescribe...Enter m_mtxProData. fStatus: %d, this: %d\n", fStatus, this);
                        _KernalMutex::scoped_lock lock (m_mtxProData);
                        if(fStreamingClient)
						    fStreamingClient->SendAlarm(STREAMING_SERVER, fStatus,CMD_DESCRIBE,  NULL);

                        m_InitLocker.do_unlock();
                        m_StatusLocker.do_lock();
                        theErr = ENOTCONN; // Exit the state machine
                        qtss_printf("ClientSession::Run() kSendDescribe...Leave m_mtxProData. fStatus: %d, this: %d\n", fStatus, this);
                        break;
                    }
                    else
                    {
                        //
                        // We've sent a describe and gotten a response from the server.
                        // Parse the response and look for track information.

                        fSDPParser.Parse(fClient->GetContentBody(), fClient->GetContentLength());

                        //////////////////////////////////////////////////////////////////////////
                        // The following code is added by Henry CHEN, HTSL
                         // To computing the duration time in second
                        StrPtrLen * theSDPPtr = fSDPParser.GetSDPData();
                        StringParser theRangeParser(theSDPPtr);
                        theRangeParser.GetThru(NULL, '=');//consume "npt="
                        theRangeParser.ConsumeWhitespace();
                        Float64 fMyStartTime, fMyStopTime;
                        UInt32 fMyDuration;
                        fMyStartTime = (Float64)theRangeParser.ConsumeFloat();
                        //see if there is a stop time as well.
                        if (theRangeParser.GetDataRemaining() > 1)
                        {
                            theRangeParser.GetThru(NULL, '-');
                            theRangeParser.ConsumeWhitespace();
                            fMyStopTime = (Float64)theRangeParser.ConsumeFloat();
                        }
                        
                        fMyDuration = (UInt32)ceil(fMyStopTime - fMyStartTime);
                        
                        fDurationInSec = fMyDuration > fDurationInSec? fMyDuration: fDurationInSec;
                        
                        // Henry CHEN code ends
                        //////////////////////////////////////////////////////////////////////////

                        // The SDP must have been misformatted.
                        if (fSDPParser.GetNumStreams() == 0)
                            fDeathReason = kBadSDP;
                            
                        // We have valid SDP. If this is a UDP connection, construct a UDP
                        // socket array to act as incoming sockets.
                        if ((fTransportType == kUDPTransportType) || (fTransportType == kReliableUDPTransportType))
                            this->SetupUDPSockets();
                            
                        // Setup client stats
                        fStats.resize(fSDPParser.GetNumStreams());
                    }
					fPlayerSimulator.Setup(fSDPParser.GetNumStreams(), fDelayTime, fStartPlayDelay);
                    fState = kSendingSetup;
                }
                break;
            }
            case kSendingSetup:
            {
                // The SETUP request is different depending on whether we are interleaving or not
                if (fTransportType == kUDPTransportType)
                {
                    theErr = fClient->SendUDPSetup(fSDPParser.GetStreamInfo(fNumSetups)->fTrackID,
                                                fUDPSocketArray[fNumSetups*2]->GetLocalPort());
                }
                else if (fTransportType == kTCPTransportType)
                {
                    fSocket->SetRcvSockBufSize(fSockRcvBufSize); // Make the rcv buf really big
					theErr = fClient->SendTCPSetup(1, 2, 3);
                }
                else if (fTransportType == kReliableUDPTransportType)
                {
                    theErr = fClient->SendReliableUDPSetup(fSDPParser.GetStreamInfo(fNumSetups)->fTrackID,
                                                fUDPSocketArray[fNumSetups*2]->GetLocalPort());
                }
				if ( fVerboseLevel >= 3)
                	qtss_printf("Sent SETUP #%"_U32BITARG_". Result = %"_U32BITARG_". Response code = %"_U32BITARG_", this:%d\n",
							fNumSetups, theErr,fStatus, this);
                //
                // If this SETUP request / response is complete, check for errors, and if
                // it succeeded, move onto the next SETUP. If we're done setting up all tracks,
                // move onto PLAY.
                if (theErr == OS_NoErr)
                {
                    m_nNoErrTime = GetTickCount();
                    if ((fStatus = fClient->GetStatus() )!= RTSP_OK)
                    {
                       qtss_printf("ClientSession::Run() kSendSetup...Enter m_mtxProData. fStatus: %d, this: %d\n", fStatus, this);
                        _KernalMutex::scoped_lock lock (m_mtxProData);
                       qtss_printf("ClientSession::Run() kSendSetup...Leave m_mtxProData. fStatus: %d, this: %d\n", fStatus, this);
                        if(fStreamingClient)
						    fStreamingClient->SendAlarm(STREAMING_SERVER, fStatus, CMD_SETUP,  NULL);
                        theErr = ENOTCONN; // Exit the state machine
                        m_InitLocker.do_unlock();
                        m_StatusLocker.do_lock();
                        break;
                    }
                    else
                    {
                        // Record the server port for RTCPs.
                        fStats[fNumSetups].fDestRTCPPort = fClient->GetServerPort() + 1;
                        
						//obtain the sampling rate of this stream
						StringParser parser = StringParser(&fSDPParser.GetStreamInfo(fNumSetups)->fPayloadName);
						parser.GetThru(NULL, '/');
						UInt32 samplingRate = parser.ConsumeInteger(NULL);
						Assert(samplingRate != 0);
						
						//Generates the client SSRC
						SInt64 ms = OS::Microseconds();
						UInt32 ssrc = static_cast<UInt32>((ms >> 32) ^ ms) + ::rand();
						fStats[fNumSetups].fClientSSRC = ssrc + fNumSetups;
						
						fPlayerSimulator.SetupTrack(fNumSetups, samplingRate, fBufferSpace);
                        fNumSetups++;
                        if (fNumSetups == fSDPParser.GetNumStreams())
                           fState = kSendingPlay;
                    }               
                }
                break;
            }
            case kSendingPlay:
			{
                if (fPacketRangePlayHeader != NULL)
                    theErr = fClient->SendPacketRangePlay(fPacketRangePlayHeader, fSpeed);
                else
                {
					if(fCmdType == kSendingRecord)
						theErr = fClient->SendPlay(0, 0, fScale);
					else
					   theErr = fClient->SendPlay(fStartPlayTimeInSec, fStartPlayTimeInSec + fDurationInSec, fScale);
				}
				if ( fVerboseLevel >= 3)
                	qtss_printf("Sent PLAY. Result = %"_U32BITARG_". Response code = %"_U32BITARG_" this: %d\n", theErr, fStatus, this);
				//
                // If this PLAY request / response is complete, then we are done with setting
                // up all the client crap. Now all we have to do is receive the data until it's
                // time to send the TEARDOWN
                if (theErr == OS_NoErr)
                {
                    m_nNoErrTime = GetTickCount();
                    if ((fStatus = fClient->GetStatus()) != RTSP_OK)
                    {
                       qtss_printf("ClientSession::Run() kSendingPlay...Enter m_mtxProData. fStatus: %d, this: %d\n", fStatus, this);
                        _KernalMutex::scoped_lock lock (m_mtxProData);
                        qtss_printf("ClientSession::Run() kSendingPlay...Leave m_mtxProData. fStatus: %d, this: %d\n", fStatus, this);
                        if(fStreamingClient)
						    fStreamingClient->SendAlarm(STREAMING_SERVER, fStatus, CMD_PLAY, NULL);
                        theErr = ENOTCONN; // Exit the state machine
                        m_InitLocker.do_unlock();
                        m_StatusLocker.do_lock();
                        break;
                    }
                        
                    // Mark down the SSRC for each track, if possible. 
                    for (UInt32 ssrcCount = 0; ssrcCount < fSDPParser.GetNumStreams(); ssrcCount++)
                        fStats[ssrcCount].fServerSSRC = fClient->GetSSRCByTrack(fSDPParser.GetStreamInfo(ssrcCount)->fTrackID);
                    
					HUS::mutex::scoped_lock lock(_ClientLocker);
					fClient->ExchangeFakeSocket();

					sPlayingConnections++;
					fState = kPlaying;
                    
                    //
                    // Start our duration timer. Use this to figure out when to send a teardown
                    fPlayTime = fLastRTCPTime = OS::Milliseconds();   
					
					if(fVerboseLevel >= 1)
					{
						for (UInt32 i = 0; i < fSDPParser.GetNumStreams(); i++)
						{
							QTSS_RTPPayloadType type = fSDPParser.GetStreamInfo(i)->fPayloadType;
							qtss_printf("Receiving track %"_U32BITARG_", trackID=%"_U32BITARG_", %s at time %"_S64BITARG_"\n",
								i, fSDPParser.GetStreamInfo(i)->fTrackID,
								type == qtssVideoPayloadType ? "video" : type == qtssAudioPayloadType ? "audio" : "unknown",
								OS::Milliseconds());
						}
					}
                }
                break;
            }
			case kSendingPing:
			{
                m_InitLocker.Reset();
				if(_ClientLocker.TryEnter() == FALSE)
				{
					fState = kPlaying;
                    qtss_printf("ClientSession::Run() kSendingPing...TryLock false. this: %d\n",this);
					break;
				}
                    qtss_printf("ClientSession::Run() kSendingPing...TryLock Successed.  this: %d\n", this);
                if(fTeardownImmediately)
                {
                    sPlayingConnections--;
                    fState = kSendingTeardown;
                    _ClientLocker.UnlockAll();
                    qtss_printf("ClientSession::Run() kSendingPing...kSendingTeardown...UnlockAll _ClientLocker.  this: %d\n", this);
                    break;
                }
                if(fStatus != RTSP_OK)
                {
                    theErr = ENOTCONN;
                    _ClientLocker.UnlockAll();
                    qtss_printf("ClientSession::Run() kSendingPing...UnlockAll _ClientLocker.  this: %d, fStatus: %d\n", this, fStatus);
                    break;
                }

				fClient->ExchangeFSocket(); 
                 theErr = Ping();
				if(theErr == OS_NoErr)
				{
                    m_nNoErrTime = GetTickCount();
                    fStatus = fClient->GetStatus();
                    {
                        _KernalMutex::scoped_lock lock (m_mtxProData);
                        if(fStreamingClient)
                            fStreamingClient->SendAlarm(STREAMING_SERVER,fStatus, CMD_PLAYING, NULL);
                    }
					if(fStatus != RTSP_OK)
					{
						theErr = ENOTCONN;
					}
					else
					{
						fState = kPlaying;
                    }
                    qtss_printf("-> Sent Ping ....this=%d the Err = %d fState = %d\n", this, theErr, fState);
                    m_nNoErrTime = GetTickCount();
                    fClient->ExchangeFakeSocket();
                    _ClientLocker.UnlockAll();
				}
                else if(theErr == EAGAIN || theErr == EINPROGRESS)
                {
                    qtss_printf("-> Sending Ping EAGAIN!\n");
                    if(GetTickCount() - m_nNoErrTime > 3000 )
                    {
                        theErr = ENOTCONN;
                        _ClientLocker.UnlockAll();
                    }
                }
                else 
                {
                    _KernalMutex::scoped_lock lock (m_mtxProData);
                    if(fStreamingClient)
                        fStreamingClient->SendAlarm(STREAMING_SERVER,RTSP_SERVICE_UNAVALIBLE, CMD_PLAYING, NULL);
                    fClient->ExchangeFakeSocket();
                   m_nNoErrTime = GetTickCount();

                   qtss_printf("-> Sent Ping ....this=%d the Err = %d fState = %d\n", this, theErr, fState);
                    _ClientLocker.UnlockAll();
                    qtss_printf("ClientSession::Run() kSendingPing...UnlockAll _ClientLocker.  this: %d, theErr:%d\n", this, theErr);
                }
				break;
			}
            case kPlaying:
			{
                
                // Should we send a teardown? We should if either we've been told to teardown, or if our time has run out
                SInt64 curTime = OS::Milliseconds();
                fTotalPlayTime = curTime - fPlayTime;

                //////////////////////////////////////////////////////////////////////////////
                // add by wanghao
                // fDuration is 0 in StreamingClientDll, which means do not check end time
                //if ( ((fDurationInSec != 0) && (fTotalPlayTime > fDurationInSec * 1000)) || (fTeardownImmediately))//wanghao
                //if ( (fTotalPlayTime > fDurationInSec * 1000) || (fTeardownImmediately) )//old
                // end wanghao

				if(fTeardownImmediately)//modify by dingweicheng
                {
                    sPlayingConnections--;
                    fState = kSendingTeardown;
                     qtss_printf("ClientSession::Run() kPlaying...kSendingTeardown.  this: %d\n", this);
                    break;
                }

                //Send RTCP if necessary; if we are using TCP for media transport, than 1 set of RTCP total is enough
				Bool16 sendRTCP = ((curTime - fLastRTCPTime) > fRTCPIntervalInMs) && (fTransportType != kTCPTransportType);
				sendRTCP |= (fPlayTime == fLastRTCPTime);		//send the first RTCP ASAP.
                if (sendRTCP)
                {
                    //(void) fClient->SendSetParameter(); // test for keep alives and error responses
                    //(void) fClient->SendOptions(); // test for keep alives  and error responses
                    for ( ; fCurRTCPTrack < fSDPParser.GetNumStreams(); fCurRTCPTrack++)
                    {
                        OS_Error err = this->SendRTCPPackets(fCurRTCPTrack);
						if (fTransportType == kTCPTransportType && err != OS_NoErr)
						{
							theErr = err; //if error happens on a TCP RTCP socket, then bail
							break;
						}
                    }
					if (theErr != OS_NoErr)
						break;
					
                    //Done sending the RTCP's
                    fCurRTCPTrack = 0;
                    fLastRTCPTime = (curTime == fLastRTCPTime) ? curTime + 1 : curTime;
					
                    //Drop the POST portion of the HTTP connection after every send
                    if (fControlType == kRTSPHTTPDropPostControlType)
						((HTTPClientSocket*)fSocket)->ClosePost();
                }
                m_InitLocker.do_unlock(); // To send commands,  We should wait for RTCP Packet had been sent out, or it will got SOCKET ERROR
                //Now read the media data
                theErr = this->ReadMediaData();
                if ((theErr == EINPROGRESS) || (theErr == EAGAIN) || (theErr == OS_NoErr))
				{
					if(theErr == EAGAIN)
					{
						if((GetTickCount() - m_nNoErrTime) > 6000)
						{
							m_nNoErrTime = GetTickCount();
							fState = kSendingPing;
                            qtss_printf("ClientSession::Run() kSendingPlay...kSendingPing.  this: %d\n", this);       
							break;
						}
					}

					theErr = OS_NoErr; //ignore control flow errors here
				}
				else
				{
                    qtss_printf("ClientSession::Run() kSendingPlay...Enter m_mtxProdata lock.  this: %d\n", this);       
                    _KernalMutex::scoped_lock lock (m_mtxProData);
                    if(fStreamingClient)
					    fStreamingClient->SendAlarm(STREAMING_SERVER,RTSP_NETWORK_ERROR , CMD_PLAYING, NULL);
                    sPlayingConnections--;
                    qtss_printf("ClientSession::Run() kSendingPlay...Leave  m_mtxProdata lock.  this: %d\n", this);   
					break;
				}

				curTime = OS::Milliseconds();
				SInt64 nextRTCPTime = fLastRTCPTime + fRTCPIntervalInMs;
                //return curTime < nextRTCPTime ? nextRTCPTime - curTime : fReadInterval;
				return curTime < nextRTCPTime ? MIN(nextRTCPTime - curTime, fReadInterval) : 1;
               // return kMaxWaitTimeInMsec/5;
            }			
            case kSendingTeardown:
			{                
				// [Yunfeng] Restore to primary fSocket.
                {
                    qtss_printf("ClientSession::Run() kSendingTeardown...Enter _ClientLocker lock.  this: %d\n", this);   
                    HUS::mutex::scoped_lock lock(_ClientLocker);
                    fClient->ExchangeFSocket();
                    qtss_printf("ClientSession::Run() kSendingTeardown...Leave _ClientLocker lock.  this: %d\n", this);   
                }
                m_InitLocker.do_unlock();
				//dbg_printf("[query][sendteardown]\n");
				theErr = fClient->SendTeardown();
				if ( fVerboseLevel >= 3)
					qtss_printf("Sending TEARDOWN. Result = %"_U32BITARG_". Response code = %"_U32BITARG_" this:%d\n", theErr, fClient->GetStatus(), this);
				// Once the TEARDOWN is complete, we are done, so mark ourselves as dead, and wait
				// for the owner of this object to delete us
				if (theErr == OS_NoErr)
                {
                    m_nNoErrTime = GetTickCount();
                    m_bReConnect = false;
                    fState = kDone;
                }
				break;
            }
            case kSendingParam:
			{
				// [Yunfeng] Restore to primary fSocket.
				HUS::mutex::scoped_lock lock(_ClientLocker);
					fClient->ExchangeFSocket();

                if (theErr == OS_NoErr)
                {
                    m_nNoErrTime = GetTickCount();
                    fState = kPlaying;
                }
			}
        }
    }

    if(theErr == EINPROGRESS || theErr == EAGAIN)
    {
        if(GetTickCount() - m_nNoErrTime > 10000)
            theErr = ENOTCONN;
    }
    if ((theErr == EINPROGRESS) || (theErr == EAGAIN))
    {
        //
        // Request an async event
        fSocket->GetSocket()->SetTask(this);
        fSocket->GetSocket()->RequestEvent(fSocket->GetEventMask());
        qtss_printf("ClientSession::Run(). EAGAIN...fState: %d, this:%d\n", fState, this);

    }
    else if (theErr != OS_NoErr)
    {
        //
        // We encountered some fatal error with the socket. Record this as a connection failure   
        qtss_printf("ClientSession:Run(). theErr:%d,  this:%d, fState:%d", theErr, this, fState);
        m_InitLocker.Reset();
        if (fState == kSendingTeardown)
            fDeathReason = kTeardownFailed;
        else if (fState == kPlaying)
            fDeathReason = kDiedWhilePlaying;
        else if (fClient->GetStatus() != RTSP_OK)
            fDeathReason = kRequestFailed;
        else
            fDeathReason = kConnectionFailed;

   		if(fState != kSendingTeardown && m_bReConnect)
  		{
			// Add by Ding Wei Cheng 2009-07-10 --- Start
			if(m_bHistorical && m_tmStart.GetTime() == LONG_MAX)
			{
				m_bReConnect = false;
				fState = kDone;
				goto aa;
			}			

			long len = static_cast<long>(m_tmEnd.GetTime() - m_tmStart.GetTime());
			if(m_bHistorical && len <= 0)
			{
				m_bReConnect = false;
				fState = kDone;
				goto aa;
			}
			// Add by Ding Wei Cheng 2009-07-10 --- End
			
			qtss_printf("ClientSession:: this=%ld,Client connection error,will reconnect, Death reason = %u\n", this,fDeathReason);
		
			if ( fClient != NULL )
			{
            	_KernalMutex::scoped_lock lock(m_ClientLocker);
                delete fClient;
                fClient = NULL;
			}

			if ( fSocket != NULL )
			{
				delete fSocket;
				fSocket = NULL;
			}
			
			if(m_bHistorical)
			{
				UInt32 uEndTime = fStartPlayTimeInSec + fDurationInSec;
				fStartPlayTimeInSec = m_tmStart.GetTime();
				fDurationInSec = uEndTime - fStartPlayTimeInSec;
				//FormatHistoricalURL();
			}
//             {
//                 _KernalMutex::scoped_lock lock (m_mtxProData);
//                 if(fStreamingClient)
//                     fStreamingClient->SendAlarm(STREAMING_SERVER,RTSP_NETWORK_ERROR , CMD_PLAYING, NULL);
//             }
			Init();
			//dbg_printf("[query][[reconnect, error:%d]\n", theErr);
            qtss_printf("ClientSession::Run()....reconnect....Init finished..this:%d\n",this);
   			//fState = fCmdType;
   			theErr = OS_NoErr;
 			//fSocket->GetSocket()->SetTask(this);
 			//fSocket->GetSocket()->RequestEvent(fSocket->GetEventMask());
			return ((UInt32) ::rand()) % kMaxWaitTimeInMsec + 1;
   		}
   		else
			fState = kDone;
    }
aa:
	if ( fVerboseLevel >= 2)
    	if (fState == kDone)
        	qtss_printf("Client connection complete. Death reason = %"_U32BITARG_"\n", fDeathReason);

#if CLIENT_SESSION_DEBUG
        if (fState == kDone)
        {
			qtss_printf("ClientSession:: this=%ld,Client connection complete. Death reason = %u\n", this,fDeathReason);
			_KernalMutex::scoped_lock lock (m_mtxProData);
            if (fStreamingClient)
            {
                qtss_printf("ClientSession:: StreamingClient = %ld,Session kDone.\n",fStreamingClient);                

				// Modify by Ding Hai Tao 2011-5-18: fStreamingClient->clientSession == NULL is not meaningful
                fStreamingClient->SetClientSessionNull();
                fStreamingClient = NULL;
            }
            return -1;
            
        }
#endif  
		return ((UInt32) ::rand()) % kMaxWaitTimeInMsec + 1; /*(fState == kPlaying) ? 1: kMaxWaitTimeInMsec;*/
}

void    ClientSession::SetupUDPSockets()
{

    static UInt16 sCurrentRTPPortToUse = 6970;
    static const UInt16 kMinRTPPort = 6970;
    static const UInt16 kMaxRTPPort = 36970;

    OS_Error theErr = OS_NoErr;
    
    //
    // Create a UDP socket pair (RTP, RTCP) for each stream
    fUDPSocketArray = NEW UDPSocket*[fSDPParser.GetNumStreams() * 2];
    for (UInt32 x = 0; x < fSDPParser.GetNumStreams() * 2; x++)
    {
        fUDPSocketArray[x] = NEW UDPSocket(this, Socket::kNonBlockingSocketType);
        theErr = fUDPSocketArray[x]->Open();
        if (theErr != OS_NoErr)
        {
            qtss_printf("ClientSession: Failed to open a UDP socket.\n");
			return;
            //::exit(-1);
        }
    }
    
    for (UInt32 y = 0; y < fSDPParser.GetNumStreams(); y++)
    {   
        for (UInt32 portCheck = 0; true; portCheck++)
        {
            theErr = fUDPSocketArray[y * 2]->Bind(INADDR_ANY, sCurrentRTPPortToUse);
            if (theErr == OS_NoErr)
                theErr = fUDPSocketArray[(y*2)+1]->Bind(INADDR_ANY, sCurrentRTPPortToUse + 1);

            sCurrentRTPPortToUse += 2;
            if (sCurrentRTPPortToUse > 30000)
                sCurrentRTPPortToUse = 6970;
                
            if (theErr == OS_NoErr)
            {
                // This is a good pair. Set the rcv buf on the RTP socket to be really big
                fUDPSocketArray[y * 2]->SetSocketRcvBufSize(fSockRcvBufSize);
                break;
            }
                
            if (sCurrentRTPPortToUse == kMaxRTPPort)
                sCurrentRTPPortToUse = kMinRTPPort;
            if (portCheck == 5000)
            {
                // Make sure we don't loop forever trying to bind a UDP socket. If we can't
                // after a certain point, just bail...
                qtss_printf("ClientSession: Failed to bind a UDP socket.\n");
				return;
                //::exit(-1);
            }
        }
    }                       
	if ( fVerboseLevel >= 3)
    	qtss_printf("Opened UDP sockets for %"_U32BITARG_" streams\n", fSDPParser.GetNumStreams());
}

//Will keep reading until all the packets buffered up has been read.
OS_Error    ClientSession::ReadMediaData()
{
    // For iterating over the array of UDP sockets
    UInt32 theUDPSockIndex = 0;
    OS_Error theErr = OS_NoErr;
	
	while (true)
    {
        //
        // If the media data is being interleaved, get it from the control connection
        UInt32 theTrackID = 0;
        UInt32 theLength = 0;
        Bool16 isRTCP = false;
		int PacketType = 0;
        char* thePacket = NULL;

        if (fTransportType == kTCPTransportType)
        {
            thePacket = NULL;
           // theErr = fClient->GetMediaPacket(&theTrackID, &isRTCP, &thePacket, &theLength);
			theErr = fClient->GetPacket(&theTrackID, &PacketType, &thePacket, &theLength);
            if (thePacket == NULL)
                break;
        }
        else
        {
            static const UInt32 kMaxPacketSize = 2048;
            
            UInt32 theRemoteAddr = 0;
            UInt16 theRemotePort = 0;
            char thePacketBuf[kMaxPacketSize];
            
            // Get a packet from one of the UDP sockets.
            theErr = fUDPSocketArray[theUDPSockIndex]->RecvFrom(&theRemoteAddr, &theRemotePort,
                                                                &thePacketBuf[0], kMaxPacketSize,
                                                                &theLength);
            if ((theErr != OS_NoErr) || (theLength == 0))
            {
                //Finished processing all the UDP packets that have been buffered up by the lower layer.
                if ((fTransportType == kReliableUDPTransportType) && (!(theUDPSockIndex & 1)))
                {
					UInt32 trackIndex = TrackID2TrackIndex(fSDPParser.GetStreamInfo(theUDPSockIndex / 2)->fTrackID);
					SendAckPackets(trackIndex);
					/*
                    for (UInt32 trackIndex = 0; trackIndex < fSDPParser.GetNumStreams(); trackIndex++)
                    {
                        if (fSDPParser.GetStreamInfo(trackIndex)->fTrackID == fSDPParser.GetStreamInfo(theUDPSockIndex / 2)->fTrackID)
                        {
                            SendAckPackets(trackIndex);

                            //if (fStats[trackCount].fHighestSeqNumValid)
                                // If we are supposed to be sending acks, and we just finished
                                // receiving all packets for this track that are available at this time,
                                // send an ACK packet
                                //this->AckPackets(trackCount, 0, false);
                        }
                    }
					*/
                }
                
                theUDPSockIndex++;
                if (theUDPSockIndex == fSDPParser.GetNumStreams() * 2)
                    break;
                continue;
            }
            
            theTrackID = fSDPParser.GetStreamInfo(theUDPSockIndex / 2)->fTrackID;
            isRTCP = (theUDPSockIndex & 1) > 0 ? true : false;
            thePacket = &thePacketBuf[0];
        }
        //
        // We have a valid packet. Invoke the packet handler function
        if (PacketType == 1)
			this->ProcessRTCPPacket(thePacket, theLength, theTrackID);
		else if(PacketType == 0)
            this->ProcessRTPPacket(thePacket, theLength, theTrackID);
		else if(PacketType == RTSP_PACKET)
		{
			if(fClient->SetRTSPPacket(thePacket, theLength) == false)
                theErr = ENOTCONN;
		}

		if(PacketType == 0 )
		{
			GetPacketTime(thePacket,theLength,m_tmStart);
		}
		m_nNoErrTime = GetTickCount();
    }
    return theErr;
}

void    ClientSession::ProcessRTPPacket(char* inPacket, UInt32 inLength, UInt32 inTrackID)
{
// 	if (inPacket == NULL || inLength <= RTP_HERDER_LENGTH)
// 	{
// 		return;
// 	}

//////////////////////////////////////////////////////////////////////////
//	The following code is added by Henry CHEN
	
	//////////////////////////////////////////////////////////////////////////
	// Add by Hai Tao 2011-11-23
	// For Playback blocked
	long nTimeSecond = 0;
	long nTimeMilliSecond = 0;
	if (false && m_bHistorical)
	{
		long nTimeSecond = ntohl(*((long*)(void*)(inPacket + 10)));
		long nTimeMilliSecond = ntohl(*((long*)(void*)(inPacket + 14)));
		if (m_nLastMilliSecond == 0)
		{
			m_nLastMilliSecond = nTimeSecond* 1000 + nTimeMilliSecond;
		}
		
		if (m_nTimeMilliSecond ==0)
		{
			m_nTimeMilliSecond = GetTickCount();
		}

		long nDelta = (nTimeSecond* 1000 + nTimeMilliSecond - m_nLastMilliSecond);
		nDelta -= (GetTickCount() - m_nTimeMilliSecond);
		nDelta -= 20;
		if (nDelta > 0)
		{
			WaitForSingleObject(NULL,nDelta);
		}
	}

	_KernalMutex::scoped_lock lock (m_mtxProData);
	if (fState != kSendingTeardown && fState != kDone && fStreamingClient)
    {
#if 0
		long nTime = ntohl(*(u_long*)(inPacket+14));
		CTime tmS((time_t)(nTime));
		long nTime2 = ntohl(*(u_long*)(inPacket+18));
		u_long nType1= ntohl(*(u_long*)(inPacket+6));
		u_long nType2= ntohl(*(u_long*)(inPacket+10));
		__int64 lType = nType1;__int64 lType2 = nType2;
		
		__int64 type = lType<<32 | lType2;

		my_printf("Time:%s %d\nType:%I64d\nLength:%d\n\n",
			(char*)(_bstr_t)(((CString)(tmS.Format(_T("%Y-%m-%d %H-%M-%S")))).AllocSysString()),nTime2,type,inLength-RTP_HERDER_LENGTH);
#endif
        // Get the Sequence to check packet lost
        ULONG ulSeq = ntohl( *reinterpret_cast<ULONG*>(inPacket + 2) );
        
        if(fPacketSeq == 1) 
            qtss_printf("New Connection Started: Dev ID = %s, This:%d\n", fStreamingClient->GetDevID(), this);

        if((ulSeq - fPacketSeq) != 1 && ulSeq != 1)
            qtss_printf("Some Packets had been lost...fPacketSeq = %d, the PacketSeq = %d, Current Dev ID = %s, This:%d\n", fPacketSeq, ulSeq, fStreamingClient->GetDevID(), this);

        fPacketSeq = ulSeq;		
        fStreamingClient->ProcessPacket(inPacket, inLength, inTrackID);	
    }   
	
	if (false && m_bHistorical)
	{
		m_nLastMilliSecond = nTimeSecond* 1000 + nTimeMilliSecond;
		m_nTimeMilliSecond = GetTickCount();
	}

    return; //Gavin return here, no need to do the following staff
//  Henry code end
//////////////////////////////////////////////////////////////////////////

	UInt32 trackIndex = TrackID2TrackIndex(inTrackID);
	if (trackIndex == kUInt32_Max)
	{	if(fVerboseLevel >= 3) 
			qtss_printf("ClientSession::ProcessRTPPacket fatal packet processing error. unknown track\n");
		return;
	}
	TrackStats &trackStats = fStats[trackIndex];
	
	if(fVerboseLevel >= 3)
	{
		SInt64 curTime = OS::Milliseconds();
		qtss_printf("Processing incoming packets at time %"_S64BITARG_"\n", curTime);
	}

    //first validate the header and check the SSRC
	Bool16 badPacket = false;
    RTPPacket rtpPacket = RTPPacket(inPacket, inLength);
    if (!rtpPacket.HeaderIsValid())
		badPacket = true;
	else
	{
		if (trackStats.fServerSSRC != 0)
		{
			if (rtpPacket.GetSSRC() != trackStats.fServerSSRC)
				badPacket = true;
		}
		else	//obtain the SSRC from the first packet if it's not available
			trackStats.fServerSSRC = rtpPacket.GetSSRC();
	}
		
	if(badPacket)
	{
        trackStats.fNumMalformedPackets++;
		if (fVerboseLevel >= 1)
			qtss_printf("TrackID=%"_U32BITARG_", len=%"_U32BITARG_"; malformed packet\n", inTrackID, inLength);
        return;
	}

    //Now check the sequence number
    UInt32 packetSeqNum = kUInt32_Max;
    if (trackStats.fHighestSeqNum == kUInt32_Max)     //this is the first sequence number received
        packetSeqNum = trackStats.fBaseSeqNum = trackStats.fHighestSeqNum = static_cast<UInt32>(rtpPacket.GetSeqNum());
    else
        packetSeqNum = CalcSeqNum(trackStats.fHighestSeqNum, rtpPacket.GetSeqNum());

    if (packetSeqNum == kUInt32_Max)            //sequence number is out of range
	{
        trackStats.fNumOutOfBoundPackets++;
		if (fVerboseLevel >= 2)
			qtss_printf("TrackID=%"_U32BITARG_", len=%"_U32BITARG_", seq=%u"", ref(32)=%"_U32BITARG_"; out of bound packet\n",
				inTrackID, inLength, rtpPacket.GetSeqNum(), trackStats.fHighestSeqNum);
	}
    else
    {
		//the packet is good -- update statisics
		Bool16 packetIsOutOfOrder = false;
        if (trackStats.fHighestSeqNum <= packetSeqNum)
            trackStats.fHighestSeqNum = packetSeqNum;
		else
			packetIsOutOfOrder = true;

        fNumPacketsReceived++;
		sPacketsReceived++;
        trackStats.fNumPacketsReceived++;
		fNumBytesReceived += inLength;
		sBytesReceived += inLength;
        trackStats.fNumBytesReceived += inLength;

		//record this sequence number so that it can be acked later on
        if (fTransportType == kReliableUDPTransportType)
            trackStats.fPacketsToAck.push_back(packetSeqNum);
		
		if (fVerboseLevel >= 3)
			qtss_printf("TrackID=%"_U32BITARG_", len=%"_U32BITARG_", seq(32)=%"_U32BITARG_", ref(32)=%"_U32BITARG_"; good packet\n",
				inTrackID, inLength, packetSeqNum, trackStats.fHighestSeqNum);

        //RTP-Meta-Info
        RTPMetaInfoPacket::FieldID* theMetaInfoFields = fClient->GetFieldIDArrayByTrack(inTrackID);
        if (theMetaInfoFields != NULL)
        {
            //
            // This packet is an RTP-Meta-Info packet. Parse it out and print out the results
            RTPMetaInfoPacket theMetaInfoPacket;
            Bool16 packetOK = theMetaInfoPacket.ParsePacket((UInt8*)inPacket, inLength, theMetaInfoFields);
            if (!packetOK)
            {
                if( fVerboseLevel >= 2)
                    qtss_printf("Received invalid RTP-Meta-Info packet\n");
            }
            else if( fVerboseLevel >= 2)
            {
                qtss_printf("---\n");
                qtss_printf("TrackID: %"_U32BITARG_"\n", inTrackID);
                qtss_printf("Packet transmit time: %"_64BITARG_"d\n", theMetaInfoPacket.GetTransmitTime());
                qtss_printf("Frame type: %u\n", theMetaInfoPacket.GetFrameType());
                qtss_printf("Packet number: %"_64BITARG_"u\n", theMetaInfoPacket.GetPacketNumber());
                qtss_printf("Packet position: %"_64BITARG_"u\n", theMetaInfoPacket.GetPacketPosition());
                qtss_printf("Media data length: %"_U32BITARG_"\n", theMetaInfoPacket.GetMediaDataLen());
            }
        }


        if (fEnable3GPP)
        {
            UInt32 timeStamp = rtpPacket.GetTimeStamp();
            Bool16 packetIsDuplicate = fPlayerSimulator.ProcessRTPPacket(trackIndex, inLength, rtpPacket.GetBody().Len, packetSeqNum, timeStamp);
            if(packetIsOutOfOrder && !packetIsDuplicate)
                trackStats.fNumOutOfOrderPackets++;
        }
    }
}

void    ClientSession::ProcessRTCPPacket(char* inPacket, UInt32 inLength, UInt32 inTrackID)
{
	UInt32 trackIndex = TrackID2TrackIndex(inTrackID);
	if (trackIndex == kUInt32_Max)
	{			
		if (fVerboseLevel >= 3) 
			qtss_printf("ClientSession::ProcessRTCPPacket fatal packet processing error. unknown track\n");
		return;
	}
	TrackStats &trackStats = fStats[trackIndex];
	
	SInt64 curTime = OS::Milliseconds();
	if(fVerboseLevel >= 2)
		qtss_printf("Processing incoming RTCP packets on track %"_U32BITARG_" at time %"_S64BITARG_"\n", trackIndex, curTime);

    //first validate the header and check the SSRC
	RTCPSenderReportPacket packet;
	Bool16 badPacket = !packet.ParseReport(reinterpret_cast<UInt8 *>(inPacket), inLength);
	if (!badPacket)
	{
		if (trackStats.fServerSSRC != 0)
		{
			if (packet.GetPacketSSRC() != trackStats.fServerSSRC)
				badPacket = true;
		}
		else	//obtain the SSRC from the first packet if it's not available
			trackStats.fServerSSRC = packet.GetPacketSSRC();
	}
		
	if(badPacket)
	{
		if (fVerboseLevel >= 1)
			qtss_printf("TrackID=%"_U32BITARG_", len=%"_U32BITARG_"; malformed RTCP packet\n", inTrackID, inLength);
        return;
	}

	//Now obtains the NTP timestamp and the current time -- we'll need it for the LSR field of the receiver report
	trackStats.fLastSenderReportNTPTime = packet.GetNTPTimeStamp();
	trackStats.fLastSenderReportLocalTime = curTime;
}

void ClientSession::SendAckPackets(UInt32 inTrackIndex)
{
	TrackStats &trackStats = fStats[inTrackIndex];

	if (trackStats.fPacketsToAck.empty())
		return;
    trackStats.fNumAcks++;

	if(fVerboseLevel >= 3)
	{
		SInt64 curTime = OS::Milliseconds();
		qtss_printf("Sending %"_U32BITARG_" acks at time %"_S64BITARG_" on track %"_U32BITARG_"\n",
			trackStats.fPacketsToAck.size(), curTime, inTrackIndex);
	}
		
    char sendBuffer[kMaxUDPPacketSize];
	
    //First send an empty Receivor Report
    RTCPRRPacket RRPacket = RTCPRRPacket(sendBuffer, kMaxUDPPacketSize);
    RRPacket.SetSSRC(trackStats.fClientSSRC);
    StrPtrLen buffer = RRPacket.GetBufferRemaining();

    //Now send the Ack packets
    RTCPAckPacketFmt ackPacket = RTCPAckPacketFmt(buffer);
    ackPacket.SetSSRC(trackStats.fClientSSRC);
    ackPacket.SetAcks(trackStats.fPacketsToAck, trackStats.fServerSSRC);
    trackStats.fPacketsToAck.clear();

    UInt32 packetLength = RRPacket.GetPacketLen() + ackPacket.GetPacketLen();

    // Send the packet
    Assert(trackStats.fDestRTCPPort != 0);
    fUDPSocketArray[(inTrackIndex*2)+1]->SendTo(fSocket->GetHostAddr(), trackStats.fDestRTCPPort, sendBuffer, packetLength);
}

OS_Error ClientSession::SendRTCPPackets(UInt32 trackIndex)
{
	TrackStats &trackStats = fStats[trackIndex];

	char buffer[kMaxUDPPacketSize];
	//::memset(buffer, 0, kMaxUDPPacketSize);

    //First send the RTCP Receiver Report packet
    RTCPRRPacket RRPacket = RTCPRRPacket(buffer, kMaxUDPPacketSize);
    RRPacket.SetSSRC(trackStats.fClientSSRC);

	UInt8 fracLost = 0;
	SInt32 cumLostPackets = 0;
	UInt32 lsr = 0;
	UInt32 dlsr = 0;
	SInt64 curTime = OS::Milliseconds();
	if (trackStats.fHighestSeqNum != kUInt32_Max)
	{
		CalcRTCPRRPacketsLost(trackIndex, fracLost, cumLostPackets);

		//Now get the middle 32 bits of the NTP time stamp and send it as the LSR
		lsr = static_cast<UInt32>(trackStats.fLastSenderReportNTPTime >> 16);

		//Get the time difference expressed as units of 1/65536 seconds
		if (trackStats.fLastSenderReportLocalTime != 0)
			dlsr = static_cast<UInt32>(OS::TimeMilli_To_Fixed64Secs(curTime - trackStats.fLastSenderReportLocalTime) >> 16);

		RRPacket.AddReportBlock(trackStats.fServerSSRC, fracLost, cumLostPackets, trackStats.fHighestSeqNum, lsr, dlsr);
	}

    StrPtrLen remainingBuf = RRPacket.GetBufferRemaining();
	UInt32 *theWriter = reinterpret_cast<UInt32 *>(remainingBuf.Ptr);

    // RECEIVER REPORT
	/*
    *(theWriter++) = htonl(0x81c90007);     // 1 src RR packet
    *(theWriter++) = htonl(0);
    *(theWriter++) = htonl(0);
    *(theWriter++) = htonl(0);
    *(theWriter++) = htonl(trackStats.fHighestSeqNum == kUInt32_Max ? 0 : trackStats.fHighestSeqNum);				//EHSN
    *(theWriter++) = 0;                         // don't do jitter yet.
    *(theWriter++) = 0;                         // don't do last SR timestamp
    *(theWriter++) = 0;                         // don't do delay since last SR
	*/

	//The implementation should be sending an SDES to conform to the standard...but its not done.
	
	if(fTransportType == kRTSPReliableUDPClientType)
	{
		// APP PACKET - QoS info
		*(theWriter++) = htonl(0x80CC000C); 
		//*(ia++) = htonl(trk[i].TrackSSRC);
		*(theWriter++) = htonl(trackStats.fClientSSRC);
	// this QTSS changes after beta to 'qtss'
		*(theWriter++) = htonl(FOUR_CHARS_TO_INT('Q', 'T', 'S', 'S'));
		//*(ia++) = toBigEndian_ulong(trk[i].rcvrSSRC);
		*(theWriter++) = htonl(trackStats.fServerSSRC);
		*(theWriter++) = htonl(8);          // number of 4-byte quants below
	#define RR 0x72720004
	#define PR 0x70720004
	#define PD 0x70640002
	#define OB 0x6F620004
		*(theWriter++) = htonl(RR);
		//unsigned int now, secs;
		//now = microseconds();
		//secs = now - trk[i].last_rtcp_packet_sent_us / USEC_PER_SEC;
		//if (secs)
		//  temp = trk[i].bytes_received_since_last_rtcp / secs;
		//else
		//  temp = 0;
		//*(ia++) = htonl(temp);
		*(theWriter++) = htonl(0);
		*(theWriter++) = htonl(PR);
		//*(ia++) = htonl(trk[i].rtp_num_received);
		*(theWriter++) = htonl(0);
		//*(theWriter++) = htonl(PL);
		//*(ia++) = htonl(trk[i].rtp_num_lost);
		//*(theWriter++) = htonl(0);
		*(theWriter++) = htonl(OB);
		*(theWriter++) = htonl(fOverbufferWindowSizeInK * 1024);
		*(theWriter++) = htonl(PD);
		*(theWriter++) = htonl(0);      // should be a short, but we need to pad to a long for the entire RTCP app packet
	}

	char *buf = reinterpret_cast<char *>(theWriter);
	UInt32 packetLen = buf - buffer;
	UInt32 playoutDelay = 0;

	if (fEnable3GPP && trackStats.fHighestSeqNum != kUInt32_Max)
	{
		//now add the RTCP NADU packet
		RTCPNADUPacketFmt nadu = RTCPNADUPacketFmt(buf, kMaxUDPPacketSize - packetLen);
		nadu.SetSSRC(trackStats.fClientSSRC);
		
		//If there is no packet in the buffer, use the extended highest sequence number received + 1
		UInt32 seqNum = fPlayerSimulator.GetNextSeqNumToDecode(trackIndex);
		if(seqNum == kUInt32_Max)
			seqNum = trackStats.fHighestSeqNum + 1;
			
		if (fEnableForcePlayoutDelay)
			playoutDelay = fPlayoutDelay;
		else
			playoutDelay = fPlayerSimulator.GetPlayoutDelay(trackIndex);

		nadu.AddNADUBlock(0, seqNum, 0, fPlayerSimulator.GetFreeBufferSpace(trackIndex), playoutDelay); 

		packetLen += nadu.GetPacketLen();
	}
	
	if(fVerboseLevel >= 2)
	{
		qtss_printf("Sending receiver reports.\n");
// 		qtss_printf("Sending receiver report at time %"_S64BITARG_" on track %"_U32BITARG_"; lostFrac=%u, lost=%"_S32BITARG_
// 			", FBS=%"_U32BITARG_", delay=%"_U32BITARG_", lsr=%"_U32BITARG_", dlsr=%"_U32BITARG_"\n",
// 			curTime, trackIndex, fracLost, cumLostPackets,
// 			fPlayerSimulator.GetFreeBufferSpace(trackIndex),
// 			playoutDelay,
// 			lsr, dlsr);
	}
		
    // Send the packet
    if (fUDPSocketArray != NULL)
    {
        Assert(trackStats.fDestRTCPPort != 0);
        fUDPSocketArray[(trackIndex*2)+1]->SendTo(fSocket->GetHostAddr(), trackStats.fDestRTCPPort, buffer, packetLen);
    }
    else
    {
        OS_Error theErr = fClient->PutMediaPacket(fSDPParser.GetStreamInfo(trackIndex)->fTrackID, true, buffer, packetLen);
        if (theErr != OS_NoErr)
		{
#if CLIENT_SESSION_DEBUG
			qtss_printf("Sended receiver reports,os error=%d.\n",theErr);
#endif

            return theErr;
		}

    }
#if CLIENT_SESSION_DEBUG
	qtss_printf("Sended receiver reports,no error.\n");
#endif

    return OS_NoErr;
}

//The lost packets in the RTCP RR are defined differently then the GetNumPacketsLost function. See  RFC 3550 6.4.1 and A.3
void ClientSession::CalcRTCPRRPacketsLost(UInt32 trackIndex, UInt8 &outFracLost, SInt32 &outCumLostPackets)
{
	TrackStats &trackStats = fStats[trackIndex];

	if (trackStats.fHighestSeqNum == kUInt32_Max)
	{
		outFracLost = 0;
		outCumLostPackets = 0;
		return;
	}

	UInt32 expected = trackStats.fHighestSeqNum - trackStats.fBaseSeqNum + 1;
	UInt32 expectedInterval = expected - trackStats.fExpectedPrior;
	UInt32 receivedInterval = trackStats.fNumPacketsReceived - trackStats.fReceivedPrior;

	trackStats.fExpectedPrior = expected;
	trackStats.fReceivedPrior = trackStats.fNumPacketsReceived;

	if (expectedInterval == 0 || expectedInterval < receivedInterval)
		outFracLost = 0;
	else
	{
		UInt32 lostInterval = expectedInterval - receivedInterval;
		outFracLost = static_cast<UInt8>((lostInterval << 8) / expectedInterval);
	}
	outCumLostPackets = expected - trackStats.fNumPacketsReceived;
}

//If newSeqNum is no more than kMaxMisorder behind and kMaxDropOut ahead of referenceSeqNum(modulo 2^16), returns the corresponding
//32 bit sequence number.  Otherwise returns kUInt32_Max
UInt32 ClientSession::CalcSeqNum(UInt32 referenceSeqNum, UInt16 newSeqNum)
{
		
    UInt16 refSeqNum16 = static_cast<UInt16>(referenceSeqNum);
    UInt32 refSeqNumHighBits = referenceSeqNum >> 16;

	if (static_cast<UInt16>(newSeqNum - refSeqNum16) <= kMaxDropOut)
	{
		//new sequence number is ahead and is in range
		if (newSeqNum >= refSeqNum16)
			return (refSeqNumHighBits << 16) | newSeqNum;			//no wrap-around
		else
			return ((refSeqNumHighBits + 1) << 16) | newSeqNum;		//wrapped-around
	}
	else if (static_cast<UInt16>(refSeqNum16 - newSeqNum) < kMaxMisorder)
	{
		//new sequence number is behind and is in range
		if (newSeqNum < refSeqNum16)
			return (refSeqNumHighBits << 16) | newSeqNum;			//no wrap-around
		else if (refSeqNumHighBits != 0)
			return ((refSeqNumHighBits - 1) << 16) | newSeqNum;		//wrapped-around
	}
	return kUInt32_Max;												//bad sequence number(out of range)
	/*
    if (refSeqNum16 <= newSeqNum)
    {
        UInt16 diff = newSeqNum - refSeqNum16;
        if (diff <= kMaxDropOut)                                    //new sequence number is ahead and is in range, no overflow
            return (refSeqNumHighBits << 16) | newSeqNum;

        diff = kUInt16_Max - newSeqNum;
        diff += refSeqNum16 + 1;
        if (diff <= kMaxMisorder && refSeqNumHighBits != 0)         //new sequence number is behind, underflow
            return ((refSeqNumHighBits - 1) << 16) | newSeqNum;
    }
    else
    {
        UInt16 diff = refSeqNum16 - newSeqNum;
        if (diff <= kMaxMisorder)                                   //new sequence number is behind and is in range, no underflow
            return (refSeqNumHighBits << 16) | newSeqNum;

        diff = kUInt16_Max - refSeqNum16;
        diff += newSeqNum + 1;
        if (diff <= kMaxDropOut)                                    //new sequence number is ahead and is in range, overflow
            return ((refSeqNumHighBits + 1) << 16) | newSeqNum;
    }
    return kUInt32_Max;                                             //Bad sequence number
	*/
}

OS_Error ClientSession::SendingParam(char * strParam)
{
	return OS_NoErr;
    //strcpy(m_strParam,strParam);
    //fState = kSendingParam;
//     StrPtrLen szParam(strParam);
//     char *pParam = NULL;
//     if(pParam = szParam.FindStringIgnoreCase("Pause"))
//     {
// 		Signal(kPauseEvent);
// 		return OS_NoErr;
//     }
//     if(pParam = szParam.FindStringIgnoreCase("Resume"))
//     {
//        Signal(kResumeEvent);
// 	   return OS_NoErr;
//     }
//     if(pParam = szParam.FindStringIgnoreCase("Speed"))
//     {
// 		szParam.Set(pParam);
// 		pParam = szParam.FindStringIgnoreCase("/");		
// 		if(pParam == NULL) return EEXIST;
// 
// 		if(strlen(pParam )<=1) return EEXIST;
// 
// 		fScale = ::atof((char*)(pParam+1));
// 
// 		Signal(kPlayControlEvent);
// 		return OS_NoErr;
// 
//     }
// 	if(pParam = szParam.FindStringIgnoreCase("MoveTo"))
// 	{
// 		szParam.Set(pParam);
// 		pParam = szParam.FindStringIgnoreCase("/");		
// 		if(pParam == NULL) return EEXIST;
// 
// 		if(strlen(pParam )<=1) return EEXIST;
// 
// 		fStartPlayTimeInSec = ::atoi((char*)(pParam+1));
// 
// 		Signal(kPlayControlEvent);
// 		return OS_NoErr;
// 	}
//     
//     if(pParam = szParam.FindStringIgnoreCase("Record"))
//     {
//         szParam.Set(pParam);
//         double dRecordTime = 0.0;
//         pParam = szParam.FindStringIgnoreCase("/");		
//         if(pParam)
//            dRecordTime = ::atof((char*)(pParam+1));
// 
//        return    fClient->SendRecord(0, dRecordTime);
//     }
    
    return EEXIST; // Not support this command.
}

void ClientSession::GetUrlTime(CTime& tmStart,CTime& tmEnd,std::string& strUrl)
{
	//historical play rtsp://192.168.250.50/{guid}/{guid}/004/user=ecserver/starttime$len
	int nEqualPos = strUrl.find("/004/user=");
	m_nStartTimePos = strUrl.find_last_of("/");
	int nLenPos = strUrl.find_first_of("$");
	std::string strStartTime = strUrl.substr(m_nStartTimePos+1,nLenPos - m_nStartTimePos);
	time_t tTime = std1::ext_string::integer(strStartTime);
	tmStart = CTime(tTime);
	std::string strLen = strUrl.substr(nLenPos+1);
	time_t tLen = std1::ext_string::integer(strLen);
	CTimeSpan span(tLen);
	tmEnd = tmStart + span;	
}

bool ClientSession::GetPacketTime(char* inPacket, UInt32 inLength,CTime& packetTime)
{
// 	if(inLength < sizeof(u_long) + 10)
// 		return false;
// 	u_long tmPacket = *((u_long*)(inPacket+10));
// 	tmPacket = htonl(tmPacket);
// 	packetTime = CTime(time_t(tmPacket));
	if(inLength < sizeof(u_long) + 18)
		return false;

	u_long tmPacket = *((u_long*)(inPacket + 14));
	tmPacket = htonl(tmPacket);
	packetTime = CTime(time_t(tmPacket));
	return true;
}
void ClientSession::SetURLToClient(StrPtrLen& theURL)
{
    _KernalMutex::scoped_lock lock(m_ClientLocker);
    if(fClient)
        fClient->Set(theURL);
}

char* ClientSession::GetSessionID()
{
    _KernalMutex::scoped_lock lock(m_ClientLocker);
    if(fClient)
        return fClient->GetSessionID()->Ptr;
    else
        return NULL;
}
void ClientSession::FormatHistoricalURL()
{
	//historical play rtsp://192.168.250.50/{guid}/{guid}/004/user=ecserver/starttime$len
	// Modified by Ding Wei Cheng 2009-07-10 --- Start
	long len = static_cast<long>(m_tmEnd.GetTime() - m_tmStart.GetTime());
// 	if(len <= 0)
// 	{
// 		m_bReConnect = false;
// 	}
	
	// Modified by Ding Wei Cheng 2009-07-10 --- End

	std::string strUrl = std::string(theURL.Ptr);
	std::string strsubUrl = strUrl.substr(0,m_nStartTimePos);
	char cUrl[1024];
	sprintf(cUrl,"%s/%d$%d",strsubUrl.c_str(),m_tmStart.GetTime(),len);

	theURL.Delete();
	theURL.Ptr = NEW char[strlen(cUrl)+1];
	memset(theURL.Ptr,0,strlen(cUrl) +1);
	strcpy(theURL.Ptr,cUrl);
	theURL.Len = strlen(cUrl);
}


// Marco for send cmd begin process
#define BEGIN_SEND_CMD()	\
    m_StatusLocker.Reset();\
   if(fState == kSendingTeardown) \
   {\
        m_StatusLocker.Set();\
        return RTSP_METHOD_NOT_VALID_IN_THIS_STATE;\
    }\
    if(m_InitLocker.do_lock(9000) == false) \
    {\
         qtss_printf("Cmd Initialization Failed! this=%d\n", this);\
         m_StatusLocker.Set();\
         m_InitLocker.Reset();\
         return RTSP_SERVICE_UNAVALIBLE;\
     }\
     m_InitLocker.Reset();\
     if(fStatus != RTSP_OK)\
    {\
      qtss_printf("Cmd Initializtion Failed! fStatus = %d this = %d\n", fStatus, this);\
      m_StatusLocker.Set();\
      return fStatus;\
    }\
    m_StatusLocker.Set();\
	OS_Error theErr = OS_NoErr;	\
/*    if(fState != kPlaying)   return EAGAIN;\*/\
    UInt32 theErrNum = RTSP_OK;	\
    DWORD dwTime = GetTickCount();\
    HUS::mutex::scoped_lock locker(_ClientLocker);\
    _KernalMutex::scoped_lock lockClient(m_ClientLocker);\
    if(fClient)   fClient->ResetNeedSetRtspWaitSignal();\
	do \
	{ \
		if(fClient)\
		{\
// Marco for send cmd end process
#define END_SEND_CMD(AlarmType, AlarmAppend)	\
            if(theErr == EAGAIN && (GetTickCount() - dwTime) > 5000)\
            {    theErr = EINVAL; }\
	        if(theErr == OS_NoErr)\
            {\
                  theErrNum = fClient->GetStatus();\
                  fStatus = theErrNum;\
            }\
		}\
        Sleep(50);\
	} while (theErr == EAGAIN || theErr == EINPROGRESS);	\
    if(fClient) fClient->SetNeedSetRtspWaitSignal();\
	if(theErr == OS_NoErr)	\
	{	\
	    return theErrNum;	\
	} \
	return RTSP_REQUEST_TIMEOUT;
//////////////////////////////////////////////////////////////////

OS_Error ClientSession::SendPlay(UInt32 inStartPlayTimeInSec, UInt32 inEndPlayTimeInSec /* = 0 */, Float32 inScale /* = 1.0 */, Float32 inSpeed /* = 1.0 */, UInt32 inTrackID /* = kUInt32_Max */)
{
	BEGIN_SEND_CMD() 
        fScale = inScale;
		theErr = fClient->SendPlay(inStartPlayTimeInSec, inEndPlayTimeInSec, inScale, inSpeed, inTrackID);
	END_SEND_CMD	(STREAMING_SERVER, CMD_PLAY);
}

OS_Error ClientSession::SendRecord(UInt32 inStartPlayTimeInSec, UInt32 inRecordTime)
{
    qtss_printf("Sending Record........this = %d\n", this);
	BEGIN_SEND_CMD()
		theErr = fClient->SendRecord(inStartPlayTimeInSec, inRecordTime);
	END_SEND_CMD	(STREAMING_SERVER, CMD_RECORD);
}

OS_Error ClientSession::SendQuery(UInt32 inStartTimeInSec, UInt32 inEndTimeInSec, UInt32 inPrecision, UInt64 i64Condition)
{
	qtss_printf("Sending Query........this = %d\n", this);
	BEGIN_SEND_CMD()
		theErr = fClient->SendQuery(inStartTimeInSec, inEndTimeInSec, inPrecision, i64Condition);
	END_SEND_CMD	(STREAMING_SERVER, CMD_QUERY);
}

OS_Error ClientSession::SendPause()
{
	BEGIN_SEND_CMD()
		theErr = fClient->SendPause();
	END_SEND_CMD	(STREAMING_SERVER, CMD_PAUSE);
}

OS_Error ClientSession::SendPacketRangePlay(char* inPacketRangeHeader, Float32 inSpeed /* = 1 */)
{
	BEGIN_SEND_CMD()
		theErr = fClient->SendPacketRangePlay(inPacketRangeHeader, inSpeed);
	END_SEND_CMD	(STREAMING_SERVER, CMD_PLAY);
}


OS_Error ClientSession::Ping()
{
	return fClient->SendOptions();
}
