using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Diagnostics.CodeAnalysis;
using HUS.Controls.Base;

namespace HUS.Controls.TabDock
{
	public class DockContent : Form, IDockContent
    {

        #region Fields

        private string m_StatusText;
        private string deviceTypeName;
        

        #endregion

        #region Constructors

        public DockContent()
		{
			m_dockHandler = new DockContentHandler(this, new GetPersistStringCallback(GetPersistString));
			m_dockHandler.DockStateChanged += new EventHandler(DockHandler_DockStateChanged);
            m_dockHandler.IsActivatedChanged += new EventHandler(m_dockHandler_IsActivatedChanged);

        }

        #endregion

        public string StatusText
        {
            get { return m_StatusText; }
            set
            {
                if (String.Compare(m_StatusText, value, false) != 0)
                {
                    m_StatusText = value;
                    if (StatusTextChanged != null)
                    {
                        StatusTextChanged(this, EventArgs.Empty);
                    }
                }
            }
        }

        public string DeviceTypeName
        {
            get { return deviceTypeName; }
            set { deviceTypeName = value; }
        }

        [Browsable(false)]
        public virtual Control CustomControl
        {
            get { return null; }
        }

		private DockContentHandler m_dockHandler = null;
		[Browsable(false)]
		public DockContentHandler DockHandler
		{
			get	{	return m_dockHandler;	}
		}

		[LocalizedCategory("Category_Docking")]
		[LocalizedDescription("DockContent_AllowEndUserDocking_Description")]
		[DefaultValue(true)]
		public bool AllowEndUserDocking
		{
			get	{	return DockHandler.AllowEndUserDocking;	}
			set	{	DockHandler.AllowEndUserDocking = value;	}
		}

		[LocalizedCategory("Category_Docking")]
		[LocalizedDescription("DockContent_DockAreas_Description")]
		[DefaultValue(DockAreas.DockLeft|DockAreas.DockRight|DockAreas.DockTop|DockAreas.DockBottom|DockAreas.Document|DockAreas.Float)]
		public DockAreas DockAreas
		{
			get	{	return DockHandler.DockAreas;	}
			set	{	DockHandler.DockAreas = value;	}
		}

		[LocalizedCategory("Category_Docking")]
		[LocalizedDescription("DockContent_AutoHidePortion_Description")]
		[DefaultValue(0.25)]
		public double AutoHidePortion
		{
			get	{	return DockHandler.AutoHidePortion;	}
			set	{	DockHandler.AutoHidePortion = value;	}
		}

		[Localizable(true)]
		[LocalizedCategory("Category_Docking")]
		[LocalizedDescription("DockContent_TabText_Description")]
		[DefaultValue(null)]
		public string TabText
		{
			get	{	return DockHandler.TabText;	}
			set	{	DockHandler.TabText = value;	}
		}
		private bool ShouldSerializeTabText()
		{
			return (DockHandler.TabText != null);
		}
              
		[LocalizedCategory("Category_Docking")]
		[LocalizedDescription("DockContent_CloseButton_Description")]
		[DefaultValue(true)]
		public bool CloseButton
		{
			get	{	return DockHandler.CloseButton;	}
			set	{	DockHandler.CloseButton = value;	}
		}
		
		[Browsable(false)]
		public TabDockPanel DockPanel
		{
			get {	return DockHandler.DockPanel; }
			set	{	DockHandler.DockPanel = value;	}
		}

		[Browsable(false)]
		public DockState DockState
		{
			get	{	return DockHandler.DockState;	}
			set	{	DockHandler.DockState = value;	}
		}

        [Browsable(false)]
        public DockState PreDockState
        {
            get { return DockHandler.PreDockState; }
        }

		[Browsable(false)]
		public DockPane Pane
		{
			get {	return DockHandler.Pane; }
			set	{	DockHandler.Pane = value;		}
		}

		[Browsable(false)]
		public bool IsHidden
		{
			get	{	return DockHandler.IsHidden;	}
			set	{	DockHandler.IsHidden = value;	}
		}

		[Browsable(false)]
		public DockState VisibleState
		{
			get	{	return DockHandler.VisibleState;	}
			set	{	DockHandler.VisibleState = value;	}
		}

		[Browsable(false)]
		public bool IsFloat
		{
			get	{	return DockHandler.IsFloat;	}
			set	{	DockHandler.IsFloat = value;	}
		}

		[Browsable(false)]
		public DockPane PanelPane
		{
			get	{	return DockHandler.PanelPane;	}
			set	{	DockHandler.PanelPane = value;	}
		}

		[Browsable(false)]
		public DockPane FloatPane
		{
			get	{	return DockHandler.FloatPane;	}
			set	{	DockHandler.FloatPane = value;	}
		}

        [SuppressMessage("Microsoft.Design", "CA1024:UsePropertiesWhereAppropriate")]
        protected virtual string GetPersistString()
		{
            return GetType().ToString();
		}

		[LocalizedCategory("Category_Docking")]
		[LocalizedDescription("DockContent_HideOnClose_Description")]
		[DefaultValue(false)]
		public bool HideOnClose
		{
			get	{	return DockHandler.HideOnClose;	}
			set	{	DockHandler.HideOnClose = value;	}
		}

		[LocalizedCategory("Category_Docking")]
		[LocalizedDescription("DockContent_ShowHint_Description")]
		[DefaultValue(DockState.Unknown)]
		public DockState ShowHint
		{
			get	{	return DockHandler.ShowHint;	}
			set	{	DockHandler.ShowHint = value;	}
		}

		[Browsable(false)]
		public bool IsActivated
		{
			get	{	return DockHandler.IsActivated;	}
		}

		public bool IsDockStateValid(DockState dockState)
		{
			return DockHandler.IsDockStateValid(dockState);
		}

		[LocalizedCategory("Category_Docking")]
		[LocalizedDescription("DockContent_TabPageContextMenu_Description")]
		[DefaultValue(null)]
		public ContextMenu TabPageContextMenu
		{
			get	{	return DockHandler.TabPageContextMenu;	}
			set	{	DockHandler.TabPageContextMenu = value;	}
		}

        [LocalizedCategory("Category_Docking")]
        [LocalizedDescription("DockContent_TabPageContextMenuStrip_Description")]
        [DefaultValue(null)]
        public ContextMenuStrip TabPageContextMenuStrip
        {
            get { return DockHandler.TabPageContextMenuStrip; }
            set { DockHandler.TabPageContextMenuStrip = value; }
        }

		[Localizable(true)]
		[Category("Appearance")]
		[LocalizedDescription("DockContent_ToolTipText_Description")]
		[DefaultValue(null)]
		public string ToolTipText
		{
			get	{	return DockHandler.ToolTipText;	}
			set {	DockHandler.ToolTipText = value;	}
		}

		public new void Activate()
		{
			DockHandler.Activate();
		}

		public new void Hide()
		{
			DockHandler.Hide();
		}

		public new void Show()
		{
			DockHandler.Show();
		}

		public void Show(TabDockPanel dockPanel)
		{
			DockHandler.Show(dockPanel);
		}

        public void RemoveDockPanel(TabDockPanel dockPanel)
        {
            DockHandler.Remove(dockPanel);
        }

		public void Show(TabDockPanel dockPanel, DockState dockState)
		{
			DockHandler.Show(dockPanel, dockState);
		}

        [SuppressMessage("Microsoft.Naming", "CA1720:AvoidTypeNamesInParameters")]
		public void Show(TabDockPanel dockPanel, Rectangle floatWindowBounds)
		{
			DockHandler.Show(dockPanel, floatWindowBounds);
		}

		public void Show(DockPane pane, IDockContent beforeContent)
		{
			DockHandler.Show(pane, beforeContent);
		}

		public void Show(DockPane previousPane, DockAlignment alignment, double proportion)
		{
			DockHandler.Show(previousPane, alignment, proportion);
		}

        [SuppressMessage("Microsoft.Naming", "CA1720:AvoidTypeNamesInParameters")]
        public void FloatAt(Rectangle floatWindowBounds)
        {
            DockHandler.FloatAt(floatWindowBounds);
        }

        public void DockTo(DockPane paneTo, DockStyle dockStyle, int contentIndex)
        {
            DockHandler.DockTo(paneTo, dockStyle, contentIndex);
        }

        public void DockTo(TabDockPanel panel, DockStyle dockStyle)
        {
            DockHandler.DockTo(panel, dockStyle);
        }

        public virtual void PreClose()
        {

        }

        public virtual void CloseFromDockPanel()
        { 
        
        }

        public virtual void ShowFromDockPanel()
        { 
        
        }

        //protected override void SetBoundsCore(int x, int y, int width, int height, BoundsSpecified specified)
        //{
        //    base.SetBoundsCore(x, y, width, height, specified);

        //    if (specified != BoundsSpecified.None)
        //    {
        //        if ((specified & BoundsSpecified.X) == BoundsSpecified.None)
        //        {
        //            x = Left;
        //        }
        //        if ((specified & BoundsSpecified.Y) == BoundsSpecified.None)
        //        {
        //            y = Top;
        //        }
        //        if ((specified & BoundsSpecified.Width) == BoundsSpecified.None)
        //        {
        //            width = Width;
        //        }
        //        if ((specified & BoundsSpecified.Height) == BoundsSpecified.None)
        //        {
        //            height = Height;
        //        }
        //    }

        //    if (x != Left || y != Top || width != Width || height != Height)
        //    {
        //        UpdateBounds(x, y, width, height);
        //    }
        //}

		#region Events
		private void DockHandler_DockStateChanged(object sender, EventArgs e)
		{
			OnDockStateChanged(e);
		}

		private static readonly object DockStateChangedEvent = new object();
		[LocalizedCategory("Category_PropertyChanged")]
		[LocalizedDescription("Pane_DockStateChanged_Description")]
		public event EventHandler DockStateChanged
		{
			add	{	Events.AddHandler(DockStateChangedEvent, value);	}
			remove	{	Events.RemoveHandler(DockStateChangedEvent, value);	}
		}
		protected virtual void OnDockStateChanged(EventArgs e)
		{
			EventHandler handler = (EventHandler)Events[DockStateChangedEvent];
			if (handler != null)
				handler(this, e);
		}

        void m_dockHandler_IsActivatedChanged(object sender, EventArgs e)
        {
            OnIsActivatedChanged(e);
        }

        private static readonly object IsActivatedChangedEvent = new object();
        [LocalizedCategory("Category_PropertyChanged")]
        public event EventHandler IsActivatedChanged
        {
            add { Events.AddHandler(IsActivatedChangedEvent, value); }
            remove { Events.RemoveHandler(IsActivatedChangedEvent, value); }
        }

        protected virtual void OnIsActivatedChanged(EventArgs e)
        {
            EventHandler handler = (EventHandler)Events[IsActivatedChangedEvent];
            if (handler != null)
                handler(this, e);
        }

        public event EventHandler StatusTextChanged;

        #endregion
	}
}
