﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.ComponentModel;
using System.Windows.Forms;
using System.Drawing.Drawing2D;

namespace HUS.Controls.TabDock.Customization
{
    internal class DefaultDockPaneCaption : DockPaneCaptionBase
    {
        private sealed class InertButton : InertButtonBase
        {
            private Bitmap m_image, m_imageAutoHide;

            public InertButton(DefaultDockPaneCaption dockPaneCaption, Bitmap image, Bitmap imageAutoHide)
                : base()
            {
                m_dockPaneCaption = dockPaneCaption;
                m_image = image;
                m_imageAutoHide = imageAutoHide;
                RefreshChanges();
            }

            private DefaultDockPaneCaption m_dockPaneCaption;
            private DefaultDockPaneCaption DockPaneCaption
            {
                get { return m_dockPaneCaption; }
            }

            public bool IsAutoHide
            {
                get { return DockPaneCaption.DockPane.IsAutoHide; }
            }

            public override Bitmap Image
            {
                get { return IsAutoHide ? m_imageAutoHide : m_image; }
            }

            protected override void OnRefreshChanges()
            {
                if (DockPaneCaption.TextColor != ForeColor)
                {
                    ForeColor = DockPaneCaption.TextColor;
                    Invalidate();
                }
            }
        }

        #region consts
        private const int _TextGapTop = 2;
        private const int _TextGapBottom = 0;
        private const int _TextGapLeft = 3;
        private const int _TextGapRight = 3;
        private const int _ButtonGapTop = 2;
        private const int _ButtonGapBottom = 1;
        private const int _ButtonGapBetween = 1;
        private const int _ButtonGapLeft = 1;
        private const int _ButtonGapRight = 2;
        #endregion

        private bool _showToolTip = false;
        public bool ShowToolTip
        {
            get { return _showToolTip; }
            set { _showToolTip = value; }
        }

        private static Bitmap _imageButtonClose;
        private static Bitmap ImageButtonClose
        {
            get
            {
                if (_imageButtonClose == null)
                    _imageButtonClose = Resources.DockPane_Close;

                return _imageButtonClose;
            }
        }

        private InertButton m_buttonClose;
        private InertButton ButtonClose
        {
            get
            {
                if (ShouldShowCloseButton)
                {
                    if (m_buttonClose == null)
                    {
                        m_buttonClose = new InertButton(this, ImageButtonClose, ImageButtonClose);
                        if (ShowToolTip)
                        {
                            m_toolTip.SetToolTip(m_buttonClose, ToolTipClose);
                        }
                        m_buttonClose.Click += new EventHandler(Close_Click);
                        Controls.Add(m_buttonClose);
                    }
                }
                else
                    m_buttonClose = null;

                return m_buttonClose;
            }
        }

        private bool m_buttonCloseVisible = false;
        public bool ShouldShowCloseButton
        {
            get { return m_buttonCloseVisible; }
            set { m_buttonCloseVisible = value; }
        }

        private static Bitmap _imageButtonAutoHide;
        private static Bitmap ImageButtonAutoHide
        {
            get
            {
                if (_imageButtonAutoHide == null)
                    _imageButtonAutoHide = Resources.DockPane_AutoHide;

                return _imageButtonAutoHide;
            }
        }

        private static Bitmap _imageButtonDock;
        private static Bitmap ImageButtonDock
        {
            get
            {
                if (_imageButtonDock == null)
                    _imageButtonDock = Resources.DockPane_Dock;

                return _imageButtonDock;
            }
        }

        private InertButton m_buttonAutoHide;
        private InertButton ButtonAutoHide
        {
            get
            {
                if (m_buttonAutoHide == null)
                {
                    m_buttonAutoHide = new InertButton(this, ImageButtonDock, ImageButtonAutoHide);
                    if (ShowToolTip)
                    {
                        m_toolTip.SetToolTip(m_buttonAutoHide, ToolTipAutoHide);
                    }
                    m_buttonAutoHide.Click += new EventHandler(AutoHide_Click);
                    Controls.Add(m_buttonAutoHide);
                }

                return m_buttonAutoHide;
            }
        }

        private static Bitmap _imageButtonOptions;
        private static Bitmap ImageButtonOptions
        {
            get
            {
                if (_imageButtonOptions == null)
                    _imageButtonOptions = Resources.DockPane_Option;

                return _imageButtonOptions;
            }
        }

        private InertButton m_buttonOptions;
        private InertButton ButtonOptions
        {
            get
            {
                if (m_buttonOptions == null)
                {
                    m_buttonOptions = new InertButton(this, ImageButtonOptions, ImageButtonOptions);
                    if (ShowToolTip)
                    {
                        m_toolTip.SetToolTip(m_buttonOptions, ToolTipOptions);
                    }
                    m_buttonOptions.Click += new EventHandler(Options_Click);
                    Controls.Add(m_buttonOptions);
                }
                return m_buttonOptions;
            }
        }

        private IContainer m_components;
        private IContainer Components
        {
            get { return m_components; }
        }

        private ToolTip m_toolTip;

        public DefaultDockPaneCaption(DockPane pane)
            : base(pane)
        {
            SuspendLayout();

            m_components = new Container();
            m_toolTip = new ToolTip(Components);

            ResumeLayout();
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
                Components.Dispose();
            base.Dispose(disposing);
        }

        private static int TextGapTop
        {
            get { return _TextGapTop; }
        }

        private static Font TextFont
        {
            get { return SystemInformation.MenuFont; }
        }

        private static int TextGapBottom
        {
            get { return _TextGapBottom; }
        }

        private static int TextGapLeft
        {
            get { return _TextGapLeft; }
        }

        private static int TextGapRight
        {
            get { return _TextGapRight; }
        }

        private static int ButtonGapTop
        {
            get { return _ButtonGapTop; }
        }

        private static int ButtonGapBottom
        {
            get { return _ButtonGapBottom; }
        }

        private static int ButtonGapLeft
        {
            get { return _ButtonGapLeft; }
        }

        private static int ButtonGapRight
        {
            get { return _ButtonGapRight; }
        }

        private static int ButtonGapBetween
        {
            get { return _ButtonGapBetween; }
        }

        private static string _toolTipClose;
        private static string ToolTipClose
        {
            get
            {
                if (_toolTipClose == null)
                    _toolTipClose = Strings.DockPaneCaption_ToolTipClose;
                return _toolTipClose;
            }
        }

        private static string _toolTipOptions;
        private static string ToolTipOptions
        {
            get
            {
                if (_toolTipOptions == null)
                    _toolTipOptions = Strings.DockPaneCaption_ToolTipOptions;

                return _toolTipOptions;
            }
        }

        private static string _toolTipAutoHide;
        private static string ToolTipAutoHide
        {
            get
            {
                if (_toolTipAutoHide == null)
                    _toolTipAutoHide = Strings.DockPaneCaption_ToolTipAutoHide;
                return _toolTipAutoHide;
            }
        }

        private static Blend _activeBackColorGradientBlend;
        private static Blend ActiveBackColorGradientBlend
        {
            get
            {
                if (_activeBackColorGradientBlend == null)
                {
                    Blend blend = new Blend(2);

                    blend.Factors = new float[] { 0.5F, 1.0F };
                    blend.Positions = new float[] { 0.0F, 1.0F };
                    _activeBackColorGradientBlend = blend;
                }

                return _activeBackColorGradientBlend;
            }
        }

        private static Color ActiveBackColorGradientBegin
        {
            get { return Color.Black; }//SystemColors.GradientActiveCaption; }
        }

        private static Color ActiveBackColorGradientEnd
        {
            get { return SystemColors.ControlDark; }//SystemColors.ActiveCaption; }
        }

        private static Color InactiveBackColor
        {
            get
            {
                return SystemColors.ControlDark;
            }
        }

        private static Color ActiveTextColor
        {
            get { return SystemColors.ActiveCaptionText; }
        }

        private static Color InactiveTextColor
        {
            get { return SystemColors.ControlText; }
        }

        private Color TextColor
        {
            get 
            {
                if (HUS.Controls.Helper.RendererManager.Style == HUS.Controls.Helper.StyleEnum.Default)
                {
                    return ActiveTextColor;
                }
                else
                {
                    return DockPane.IsActivated ? ActiveTextColor : InactiveTextColor;
                }
            }
        }

        private static TextFormatFlags _textFormat =
            TextFormatFlags.SingleLine |
            TextFormatFlags.EndEllipsis |
            TextFormatFlags.VerticalCenter;
        private TextFormatFlags TextFormat
        {
            get
            {
                if (RightToLeft == RightToLeft.No)
                    return _textFormat;
                else
                    return _textFormat | TextFormatFlags.RightToLeft | TextFormatFlags.Right;
            }
        }

        protected internal override int MeasureHeight()
        {
            int height = TextFont.Height + TextGapTop + TextGapBottom;

            if (ShouldShowCloseButton)
            {
                if (height < ButtonClose.Image.Height + ButtonGapTop + ButtonGapBottom)
                    height = ButtonClose.Image.Height + ButtonGapTop + ButtonGapBottom;
            }
            else
            {
                if (height < ButtonAutoHide.Image.Height + ButtonGapTop + ButtonGapBottom)
                    height = ButtonAutoHide.Image.Height + ButtonGapTop + ButtonGapBottom;
            }

            return height;
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            DrawCaption(e.Graphics);
        }

        private void DrawCaption(Graphics g)
        {
            if (ClientRectangle.Width == 0 || ClientRectangle.Height == 0)
                return;

            if (DockPane.IsActivated)
            {
                if (HUS.Controls.Helper.RendererManager.Style == HUS.Controls.Helper.StyleEnum.Default)
                {
                    g.DrawImage(Properties.Resources.caption_activebk, ClientRectangle);
                }
                else
                {
                    using (LinearGradientBrush brush = new LinearGradientBrush(ClientRectangle, ActiveBackColorGradientBegin, ActiveBackColorGradientEnd, LinearGradientMode.Vertical))
                    {
                        brush.Blend = ActiveBackColorGradientBlend;
                        g.FillRectangle(brush, ClientRectangle);
                    }
                }
            }
            else
            {
                if (HUS.Controls.Helper.RendererManager.Style == HUS.Controls.Helper.StyleEnum.Default)
                {
                    g.DrawImage(Properties.Resources.captionbk, ClientRectangle);
                }
                else
                {
                    using (SolidBrush brush = new SolidBrush(InactiveBackColor))
                    {
                        g.FillRectangle(brush, ClientRectangle);
                    }
                    g.DrawRectangle(SystemPens.ControlDark, new Rectangle(0, 0, ClientSize.Width - 1, ClientSize.Height - 1));
                }
            }

            Rectangle rectCaption = ClientRectangle;

            Rectangle rectCaptionText = rectCaption;
            rectCaptionText.X += TextGapLeft;
            rectCaptionText.Width -= TextGapLeft + TextGapRight;
            if(ShouldShowCloseButton)
                rectCaptionText.Width -= ButtonGapLeft + ButtonClose.Width + ButtonGapRight;
            else
                rectCaptionText.Width -= ButtonGapLeft + ButtonGapRight;
            if (ShouldShowAutoHideButton)
                rectCaptionText.Width -= ButtonAutoHide.Width + ButtonGapBetween;
            if (HasTabPageContextMenu)
                rectCaptionText.Width -= ButtonOptions.Width + ButtonGapBetween;
            rectCaptionText.Y += TextGapTop;
            rectCaptionText.Height -= TextGapTop + TextGapBottom;
            TextRenderer.DrawText(g, DockPane.CaptionText, TextFont, DrawHelper.RtlTransform(this, rectCaptionText), Color.White, TextFormat);
        }

        protected override void OnLayout(LayoutEventArgs levent)
        {
            SetButtonsPosition();
            base.OnLayout(levent);
        }

        protected override void OnRefreshChanges()
        {
            SetButtons();
            Invalidate();
        }

        private bool CloseButtonEnabled
        {
            get { return (DockPane.ActiveContent != null) ? DockPane.ActiveContent.DockHandler.CloseButton : false; }
        }

        private bool ShouldShowAutoHideButton
        {
            get { return !DockPane.IsFloat; }
        }

        private void SetButtons()
        {
            if (ShouldShowCloseButton)
                ButtonClose.Enabled = CloseButtonEnabled;
            ButtonAutoHide.Visible = ShouldShowAutoHideButton;
            ButtonOptions.Visible = HasTabPageContextMenu;
            if(ShouldShowCloseButton)
                ButtonClose.RefreshChanges();
            ButtonAutoHide.RefreshChanges();
            ButtonOptions.RefreshChanges();

            SetButtonsPosition();
        }

        private void SetButtonsPosition()
        {
            // set the size and location for close and auto-hide buttons
            Point point = new Point();
            Rectangle rectCaption = ClientRectangle;
            point.X = rectCaption.X + rectCaption.Width - 1 - ButtonGapRight;
            point.Y = rectCaption.Y + ButtonGapTop;

            if (ShouldShowCloseButton)
            {
                int buttonWidth = ButtonClose.ButtonWidth;//.Image.Width;
                int buttonHeight = ButtonClose.ButtonHeight;//.Image.Height;
                int height = rectCaption.Height - ButtonGapTop - ButtonGapBottom;
                if (buttonHeight < height)
                {
                    buttonWidth = buttonWidth * (height / buttonHeight);
                    buttonHeight = height;
                }
                Size buttonSize = new Size(buttonWidth, buttonHeight);

                point.X -= buttonWidth;
                ButtonClose.Bounds = DrawHelper.RtlTransform(this, new Rectangle(point, buttonSize));
                
            }

            if (ShouldShowAutoHideButton)
            {
                int buttonWidth = ButtonAutoHide.ButtonWidth;//.Image.Width;
                int buttonHeight = ButtonAutoHide.ButtonHeight;//.Image.Height;
                int height = rectCaption.Height - ButtonGapTop - ButtonGapBottom;
                if (buttonHeight < height)
                {
                    buttonWidth = buttonWidth * (height / buttonHeight);
                    buttonHeight = height;
                }
                Size buttonSize = new Size(buttonWidth, buttonHeight);

                if(ShouldShowCloseButton)
                    point.Offset(-(buttonWidth + ButtonGapBetween), 0);
                else
                    point.X -= buttonWidth;
                ButtonAutoHide.Bounds = DrawHelper.RtlTransform(this, new Rectangle(point, buttonSize));
            }
            if (HasTabPageContextMenu)
            {
                int buttonWidth = ButtonOptions.ButtonWidth;//.Image.Width;
                int buttonHeight = ButtonOptions.ButtonHeight;//.Image.Height;
                int height = rectCaption.Height - ButtonGapTop - ButtonGapBottom;
                if (buttonHeight < height)
                {
                    buttonWidth = buttonWidth * (height / buttonHeight);
                    buttonHeight = height;
                }
                Size optionsButtonSize = new Size(buttonWidth, buttonHeight);
                if (ShouldShowAutoHideButton)
                    point.Offset(-(buttonWidth + ButtonGapBetween), 0);
                else
                    point.X -= buttonWidth;
                ButtonOptions.Bounds = DrawHelper.RtlTransform(this, new Rectangle(point, optionsButtonSize));
            }
        }

        private void Close_Click(object sender, EventArgs e)
        {
            DockPane.CloseActiveContent();
        }

        private void AutoHide_Click(object sender, EventArgs e)
        {
            DockPane.DockState = DockHelper.ToggleAutoHideState(DockPane.DockState);
        }

        private void Options_Click(object sender, EventArgs e)
        {
            ShowTabPageContextMenu(PointToClient(Control.MousePosition));
        }

        protected override void OnRightToLeftChanged(EventArgs e)
        {
            base.OnRightToLeftChanged(e);
            PerformLayout();
        }
    }
}
