/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 * Jan 11, 2005		TimYu		add support MBM29DL640E
 * Jan 19, 2005		TimYu		add support S29GL064M (MODEL R4)
 *
 * The file include more flash driver.
 * S29GL064M's datasheet : S29GLxxxM_00_A4_E_Final.pdf
 */

#include "armboot.h"
#include <config.h>

#define	FlashReadU16(addr)					(*((volatile unsigned short *)(addr)))
#define FlashReadU8(addr)						(*((volatile unsigned char *)(addr)))
#define	FlashWriteU16(addr, value)	(*((volatile unsigned short *)(addr)) = (value))
#define FlashWriteU8(addr,value) 		(*((volatile unsigned char *)(addr)) = (value))

#if defined (AM29LV160D) || defined (HY29LV320B)
#define FLASH_BANK_SIZE		0x200000
#define MAIN_SECT_SIZE		0x10000
#define PARAM_SECT_SIZE		0x2000
#define INIT_SECT_SIZE		0x4000		/* for AMD29LV160DB	*/
#define MIDDLE_SECT_SIZE	0x8000		/* for AMD29LV160DB	*/
#endif

#ifdef K8B1616UBA
#define FLASH_BANK_SIZE		0x200000
#define MAIN_SECT_SIZE		0x10000
#define PARAM_SECT_SIZE		0x2000
#endif

#if defined(AM29LV320D) || defined(AT49BV321)
#define FLASH_BANK_SIZE		0x400000
#define MAIN_SECT_SIZE		0x10000
#define PARAM_SECT_SIZE		0x2000
#endif

#ifdef SST39VF160
#define FLASH_BANK_SIZE		0x200000
#define MAIN_SECT_SIZE		0x10000
#define PARAM_SECT_SIZE		0x1000
#endif

#ifdef MBM29DL640E
#define FLASH_BANK_SIZE		0x800000
#define MAIN_SECT_SIZE		0x10000
#define PARAM_SECT_SIZE		0x2000
#endif

#ifdef	S29GL064M
#define FLASH_BANK_SIZE		0x800000
#define MAIN_SECT_SIZE		0x10000
#define PARAM_SECT_SIZE		0x2000
#define FLASH_PAGE_MASK 	0x1F
#define FLASH_BUFFER_SIZE 32
#endif

flash_info_t flash_info[CFG_MAX_FLASH_BANKS];

void FlashDelay(unsigned int delay)
{
	unsigned int i = delay*10;

	while (i--);
}

/************************************************************************
 * DataToggle
 ************************************************************************/
int DataToggle(ulong addr, u8 data)
{
	u8 oldd, curd;
	u8 pd;
	
	pd = data & 0x80;
	
	while(1)
	{
		oldd = FlashReadU8(addr);
		if( (oldd & 0x80) == pd)
			return 0;
		
		if( (oldd & 0x20) )
		{
t1:
			oldd = FlashReadU8(addr);
			if( (oldd & 0x80) == pd )
				return 0;
			else
				return -1;
		}
		else
		{
			if( oldd & 0x02 )
				goto t1;
		}	
	}
	/*
	oldd = FlashReadU8(addr);
	if (oldd == data)
	{
		return 0;
	}

	for (;;)
	{
		FlashDelay(1);
		oldd = FlashReadU8(addr);
		curd = FlashReadU8(addr);
		if (oldd == curd)
		{
			break;
		}
	}

	return 0;*/
}

#ifdef	S29GL064M

/*
	addr : write address
*/
void FlashWriteToBufferCmd(u32 base_addr)
{       
	#if defined(CFG_FLASH_BUSSIZE8)
	
	FlashWriteU8(CFG_FLASH_BASE + 0xAAA, 0xAA);
    FlashWriteU8(CFG_FLASH_BASE + 0x555, 0x55);
  	
  	/* Write Write Buffer Load Command */
  	FlashWriteU8(base_addr, 0x25);
  	
  #endif
}

/*
	addr : write address
*/
void FlashProgramBufferToFlashCmd(u32 base_addr)
{       
  /* Transfer Buffer to Flash Command */
  #if defined(CFG_FLASH_BUSSIZE8)
  
  	/* Write Write Buffer Load Command */
  	FlashWriteU8(base_addr, 0x29);
  	
  #endif
}


/*
	addr : write address
	word_count : number of words to program
	data_buf : buffer containing data to program
*/
int FlashWriteBufferProgramPage(u32 base_addr, u32 offset,u32 word_count,u8 * data_buf)
{       
  u8 *p = data_buf;
  u32   last_loaded_addr;
  u32   current_offset;
  u32   end_offset;
  u8 wcount;

  /* Initialize variables */
  current_offset   = base_addr + offset;
  end_offset       = base_addr + offset + word_count - 1;
  last_loaded_addr = base_addr + offset;

  /* don't try with a count of zero */
  if (!word_count) 
  	return 0;

  if (((offset & FLASH_PAGE_MASK) + word_count -1) >  FLASH_PAGE_MASK)  
  	return 0;
  	
  FlashDelay(10);
  
  /* Issue Load Write Buffer Command Sequence */
  FlashWriteToBufferCmd(base_addr);

  /* Write # of locations to program */
  wcount = (u8)word_count - 1;
  FlashWriteU8(base_addr,wcount);

  /* Load Data into Buffer */
  while(current_offset <= end_offset)
  {
    /* Store last loaded address & data value (for polling) */
    last_loaded_addr = current_offset;

    /* Write Data */
    FlashWriteU8(current_offset,*p);
    current_offset++;
    p++;
  }

  /* Issue Program Buffer to Flash command */
  FlashProgramBufferToFlashCmd(base_addr);
	p--;

	if( 0 == DataToggle(last_loaded_addr,*p) )
  		return(word_count);
  	else
  		return 0;
}

#endif

/*-----------------------------------------------------------------------
 */

ulong
flash_init (bd_t * bd)
{
  int i, j;
  ulong size = 0;

  for (i = 0; i < CFG_MAX_FLASH_BANKS; i++)
    {
#if defined (AM29LV160D) || defined (HY29LV320B)
      ulong flashbase = 0;
#if defined(MBM29LV160BE)
      flash_info[i].flash_id =
	(FUJ_MANUFACT & FLASH_VENDMASK) | (FUJ_ID_MBM29LV160BE &
					   FLASH_TYPEMASK);
#elif defined(M29W160DB)
      flash_info[i].flash_id =
	(ST_MANUFACT & FLASH_VENDMASK) | (ST_ID_M29W160DB & FLASH_TYPEMASK);
#elif defined(HY29LV320B)
      flash_info[i].flash_id =
	(HYNIX_MANUFACT & FLASH_VENDMASK) | (HYNIX_ID_HY29LV320B &
					     FLASH_TYPEMASK);
#elif defined(S29GL064M)
      flash_info[i].flash_id =
	(FUJ_MANUFACT & FLASH_VENDMASK) | (SPS_ID_S29GL064M &
					   FLASH_TYPEMASK);
#else
      flash_info[i].flash_id =
	(AMD_MANUFACT & FLASH_VENDMASK) | (AMD_ID_LV160D & FLASH_TYPEMASK);
#endif
      flash_info[i].size = FLASH_BANK_SIZE;
      flash_info[i].sector_count = CFG_MAX_FLASH_SECT;
      memset (flash_info[i].protect, 0, CFG_MAX_FLASH_SECT);

      flashbase = PHYS_FLASH_1;


      for (j = 0; j < flash_info[i].sector_count; j++)
	{
	  if (j <= 1)
	    flash_info[i].start[j] = flashbase + j * INIT_SECT_SIZE;
	  else if ((2 <= j) && (j <= 3))
	    flash_info[i].start[j] = flashbase + (j + 1) * PARAM_SECT_SIZE;
	  else
	    flash_info[i].start[j] = flashbase + (j - 3) * MAIN_SECT_SIZE;
	}

      size += flash_info[i].size;
#else
      ulong flashbase = 0;

#if defined(K8B1616UBA)
      flash_info[i].flash_id =
	(SAMSUNG_MANUFACT & FLASH_VENDMASK) | (SAMSUNG_ID_K8B1616UBA &
					       FLASH_TYPEMASK);
#elif defined(AM29LV320D)
      flash_info[i].flash_id =
	(AMD_MANUFACT & FLASH_VENDMASK) | (AMD_ID_LV320DB & FLASH_TYPEMASK);
#elif defined(AT49BV321)
      flash_info[i].flash_id =
	(ATMEL_MANUFACT & FLASH_VENDMASK) | (ATMEL_ID_AT49BV321 &
					     FLASH_TYPEMASK);
#elif defined(SST39VF160)
      flash_info[i].flash_id =
	(SST_MANUFACT & FLASH_VENDMASK) | (SST_ID_xF160A & FLASH_TYPEMASK);

#elif defined(MBM29DL640E)
      flash_info[i].flash_id =
	(FUJ_MANUFACT & FLASH_VENDMASK) | (FUJ_ID_DL640E & FLASH_TYPEMASK);

#elif defined(S29GL064M)
      flash_info[i].flash_id =
	(SPS_MANUFACT & FLASH_VENDMASK) | (SPS_ID_GL064M & FLASH_TYPEMASK);
#endif

      flash_info[i].size		= FLASH_BANK_SIZE;
      flash_info[i].sector_count	= CFG_MAX_FLASH_SECT;
      memset (flash_info[i].protect, 0, CFG_MAX_FLASH_SECT);

      flashbase = PHYS_FLASH_1;

      for (j = 0; j < flash_info[i].sector_count; j++)
	{
#ifdef MBM29DL640E
	  if (j <= 7)
	    {
	      flash_info[i].start[j] = flashbase + j * PARAM_SECT_SIZE;
	    }
	  else if (j <= 133 && j >= 8)
	    {
	      flash_info[i].start[j] = flashbase + (j - 7) * MAIN_SECT_SIZE;
	    }
	  else
	    {
	      flash_info[i].start[j] = flashbase + 126 * MAIN_SECT_SIZE +
		      			8 * PARAM_SECT_SIZE +
					(j - 133) * PARAM_SECT_SIZE;

#else
#ifdef SST39VF160
	  if (j <= 15)
#else
	  if (j <= 7)
#endif
	    {
	      flash_info[i].start[j] = flashbase + j * PARAM_SECT_SIZE;
	    }
	  else
	    {
#ifdef SST39VF160
	      flash_info[i].start[j] = flashbase + (j - 15) * MAIN_SECT_SIZE;
#else
	      flash_info[i].start[j] = flashbase + (j - 7) * MAIN_SECT_SIZE;
#endif
#endif

	    }
	}
      size += flash_info[i].size;
#endif

    }

  /* Protect monitor and environment sectors
   */
  flash_protect (FLAG_PROTECT_SET,
		 CFG_FLASH_BASE,
		 CFG_FLASH_BASE + _armboot_end - _armboot_start,
		 &flash_info[0]);

  flash_protect (FLAG_PROTECT_SET,
		 CFG_ENV_ADDR,
		 CFG_ENV_ADDR + CFG_ENV_SIZE - 1,
		 &flash_info[0]);

  return size;

} /* end flash_init */

/*-----------------------------------------------------------------------
 */
void
flash_print_info (flash_info_t * info)
{
  int i;

  switch (info->flash_id & FLASH_VENDMASK)
    {
    case (SAMSUNG_MANUFACT & FLASH_VENDMASK):
      printf ("SAMSUNG : ");
      break;
#if (1)
    case (SPS_MANUFACT & FLASH_VENDMASK):
      printf ("SPANSION (AMD) : ");
      break;
#else
    case (AMD_MANUFACT & FLASH_VENDMASK):
      printf ("AMD : ");
      break;
#endif
    case (ATMEL_MANUFACT & FLASH_VENDMASK):
      printf ("ATMEL : ");
      break;
    case (FUJ_MANUFACT & FLASH_VENDMASK):
      printf ("FUJITSU : ");
      break;
    case (SST_MANUFACT & FLASH_VENDMASK):
      printf ("SST : ");
      break;
    case (HYNIX_MANUFACT & FLASH_VENDMASK):
      printf ("HYNIX : ");
      break;
    default:
      printf ("Unknown Vendor ");
      break;
    }

  switch (info->flash_id & FLASH_TYPEMASK)
    {
    case (SAMSUNG_ID_K8B1616UBA & FLASH_TYPEMASK):
      printf ("K8B1616UBA\n");
      break;
    case (AMD_ID_LV320DB & FLASH_TYPEMASK):
      printf ("AM29LV320D\n");
      break;
    case (ATMEL_ID_AT49BV321 & FLASH_TYPEMASK):
      printf ("AT49BV321\n");
      break;
#if defined(MBM29LV160BE)
    case (FUJ_ID_MBM29LV160BE & FLASH_TYPEMASK):
      printf ("MBM29LV160BE\n");
      break;
#elif defined(M29W160DB)
    case (ST_ID_M29W160DB & FLASH_TYPEMASK):
      printf ("M29W160DB\n");
      break;
#elif defined(MBM29DL640E)
    case (FUJ_ID_MBM29DL640E & FLASH_TYPEMASK):
      printf ("MBM29DL640E\n");
      break;
#elif defined(S29GL064M)
    case (SPS_ID_S29GL064M & FLASH_TYPEMASK):
      printf ("S29GL064M\n");
      break;
#else
    case (AMD_ID_LV160D & FLASH_TYPEMASK):
      printf ("AM29LV160D\n");
      break;
#endif
    case (SST_ID_xF160A & FLASH_TYPEMASK):
      printf ("SST39VF160\n");
      break;
    case (HYNIX_ID_HY29LV320B & FLASH_TYPEMASK):
      printf ("HY29LV320B\n");
      break;
    default:
      printf ("Unknown Chip Type\n");
      goto Done;
      break;
    }

  printf ("  Size: %ld MB in %d Sectors\n", info->size >> 20,
	  info->sector_count);

  printf ("  Sector Start Addresses:");
  for (i = 0; i < info->sector_count; i++)
    {
      if ((i % 5) == 0)
	{
	  printf ("\n   ");
	}
      printf (" %08lX%s", info->start[i],
	      info->protect[i] ? " (RO)" : "     ");
    }
  printf ("\n");

Done:
}

/*-----------------------------------------------------------------------
 */

int
flash_erase (flash_info_t * info, int s_first, int s_last)
{
  int flag, prot, sect;
  int rc = ERR_OK;

  if (info->flash_id == FLASH_UNKNOWN)
    return ERR_UNKNOWN_FLASH_TYPE;

  if ((s_first < 0) || (s_first > s_last))
    return ERR_INVAL;


  //if ((info->flash_id & FLASH_VENDMASK) !=(SAMSUNG_MANUFACT & FLASH_VENDMASK))
  //  return ERR_UNKNOWN_FLASH_VENDOR;

  prot = 0;

  for (sect = s_first; sect <= s_last; ++sect)
    {
      if (info->protect[sect])
	prot++;
    }

  //if (prot)
  //  {
  //    return ERR_PROTECTED;
  //  }

  /*
   * Disable interrupts which might cause a timeout
   * here. Remember that our exception vectors are
   * at address 0 in the flash, and we don't want a
   * (ticker) exception to happen while the flash
   * chip is in programming mode.
   */
  flag = disable_interrupts ();

  /* Start erase on unprotected sectors */
  for (sect = s_first; sect <= s_last && !ctrlc (); sect++)
    {
      unsigned long SectorAddr;

      printf ("Erasing sector %2d ...\n", sect);
      /* arm simple, non interrupt dependent timer */
      /*reset_timer_masked(); */

      SectorAddr = flash_info[0].start[sect];	/* total address */
      
      #if defined(CFG_FLASH_BUSSIZE8)
      	FlashWriteU8(CFG_FLASH_BASE + 0xAAA, 0xAA);
      	FlashWriteU8(CFG_FLASH_BASE + 0x555, 0x55);
      	FlashWriteU8(CFG_FLASH_BASE + 0xAAA, 0x80);
      	FlashWriteU8(CFG_FLASH_BASE + 0xAAA, 0xAA);
      	FlashWriteU8(CFG_FLASH_BASE + 0x555, 0x55); 
      	FlashWriteU8(SectorAddr, 0x30); 
      	
				FlashDelay(10);
	
	      while ((FlashReadU8(SectorAddr) & 0xFF) != 0xFF)
	      {
	        FlashDelay(1);
	      }
	      
	      /*
	       * Flash reset
	       */
	      FlashWriteU8(CFG_FLASH_BASE, 0xF0);       
      
      #else
      	FlashWriteU16(CFG_FLASH_BASE + 2*0x555, 0xAA);
      	FlashWriteU16(CFG_FLASH_BASE + 2*0x2AA, 0x55);
      	FlashWriteU16(CFG_FLASH_BASE + 2*0x555, 0x80);
      	FlashWriteU16(CFG_FLASH_BASE + 2*0x555, 0xAA);
      	FlashWriteU16(CFG_FLASH_BASE + 2*0x2AA, 0x55);
      	FlashWriteU16(SectorAddr, 0x30);
      	
      	/*
      	* Sector to be erased.
      	*/
	      FlashDelay(10);
	
	      while ((FlashReadU16(SectorAddr) & 0xFF) != 0xFF)
	      {
	        FlashDelay(1);
	        if (ctrlc ())
    				printf ("User Interrupt!\n");
	      }
	
	      /*
	       * Flash reset
	       */
	      FlashWriteU16(CFG_FLASH_BASE, 0xF0);      
      
      #endif


    }

  return rc;
}

/*-----------------------------------------------------------------------
 * Copy memory to flash
 */
static int write_word(flash_info_t * info, ulong addr, ushort data)
{
	disable_interrupts();

	#if defined(CFG_FLASH_BUSSIZE8)
			

			
	#else
			FlashWriteU16(CFG_FLASH_BASE + 2*0x555, 0xAA);
			FlashWriteU16(CFG_FLASH_BASE + 2*0x2AA, 0x55);
			FlashWriteU16(CFG_FLASH_BASE + 2*0x555, 0xA0);
		
			/*
			 * Write data.
			 */
			FlashWriteU16(addr, data);
		
			/*
			 * Data toggle.
			 */
			DataToggle(addr, data);
		
			/*
			 * Flash reset.
			 */
			FlashWriteU16(CFG_FLASH_BASE, 0xF0);
		
	
	#endif

	/*
	 * arm simple, non interrupt dependent timer
	 */	
	reset_timer_masked();
	return 0;
} /* end WriteWord */

/*-----------------------------------------------------------------------
 * Copy memory to flash.
 */
#if 0
int
write_buff (flash_info_t * info, uchar * src, ulong addr, ulong cnt)
{
  ulong cp, wp;
  int l;
  int i, rc;
  ushort data = 0;

  wp = (addr & ~3);		/* get lower word aligned address */

  /*
   * handle unaligned start bytes
   */
  if ((l = addr - wp) != 0)
    {
      data = 0;
      for (i = 0, cp = wp; i < l; ++i, ++cp)
	data = (data >> 8) | (*(uchar *) cp << 24);

      for (; i < 4 && cnt > 0; ++i)
	{
	  data = (data >> 8) | (*src++ << 24);
	  --cnt;
	  ++cp;
	}

      for (; cnt == 0 && i < 4; ++i, ++cp)
	data = (data >> 8) | (*(uchar *) cp << 24);


      if ((rc = write_word (info, wp, data)) != 0)
	return (rc);

      wp += 4;
    }

  /*
   * handle word aligned part
   */

  while (cnt >= 2)
    {
      data = *((unsigned short *) src);
      if ((rc = write_word (info, addr, data)) != 0)
	return (rc);

      src	+= 2;
      addr	+= 2;
      cnt	-= 2;

    }

  if (cnt == 0)
    return ERR_OK;


  /*
   * handle unaligned tail bytes
   */
  data = 0;
  for (i = 0, cp = wp; i < 4 && cnt > 0; ++i, ++cp)
    {
      data = (data >> 8) | (*src++ << 24);
      --cnt;
    }
  for (; i < 4; ++i, ++cp)
    data = (data >> 8) | (*(uchar *) cp << 24);

  return write_word (info, wp, data);
}

#else
/*	addr must is sector address */
int
write_buff (flash_info_t * info, uchar * src, ulong addr, ulong cnt)
{
	u32 i, lcnt;
	u8 *p = src;
	u32 offset;
	u32 b_addr;
	u32 secl, sect;
	
	i = cnt;
	offset = 0;
	b_addr = addr;
	info = &flash_info[0];
	
	while(1)
	{
		/* first get the current sector address */
		if( 0 == i )
			break;
		
		secl = 0;
		for (sect = 0; sect < info->sector_count; sect++)
		{
			if( info->start[sect] == b_addr )
			{
				sect++;
				if( info->sector_count == sect )
					secl = CFG_FLASH_BASE + PHYS_FLASH_SIZE - b_addr;
				else
					secl = info->start[sect] - b_addr;
				break;
			}
		}
		
		if( 0 == secl )
			return 0;
			
		if( secl > i )
			secl = i;
		
		i -= secl;
		
		/* write the dtat buffer to current sector */
		offset = 0;
		printf ("Write Flash : 0x%08lx -> 0x%08lx\n", b_addr, secl);
		//printf ("Source Addr : 0x%08lx\n", (u32)src );
		while(1)
		{
			if( offset == secl )
			{
				b_addr += secl;
				break;
			}
				
			if( ( secl - offset)  > FLASH_BUFFER_SIZE )
				lcnt = FLASH_BUFFER_SIZE;
			else
				lcnt = ( secl - offset);
						
			if ( 0 == FlashWriteBufferProgramPage(b_addr,offset,lcnt,p) )
				return -1;
				
			offset += lcnt;
			p += lcnt;	
			//secl -= lcnt;
		}/* end write sector */
		
	}
	
	return 0;
}



#endif

/*
	Read the ID of the flash
		ManID : Manufacturer ID
		DevID : Device ID
*/
void flash_ReadID(unsigned int *ManID, unsigned int *DevID)
{
	unsigned int devid = 0;
	disable_interrupts();

	#if defined(CFG_FLASH_BUSSIZE8)
			
			//Manufacturer ID
			FlashWriteU8(CFG_FLASH_BASE + 0xAAA, 0xAA);
    	FlashWriteU8(CFG_FLASH_BASE + 0x555, 0x55);
    	FlashWriteU8(CFG_FLASH_BASE + 0xAAA, 0x90);
			*ManID =  FlashReadU8(CFG_FLASH_BASE);
      FlashWriteU8(CFG_FLASH_BASE, 0xF0);  
      
      //Device ID
 			FlashWriteU8(CFG_FLASH_BASE + 0xAAA, 0xAA);
    	FlashWriteU8(CFG_FLASH_BASE + 0x555, 0x55);
    	FlashWriteU8(CFG_FLASH_BASE + 0xAAA, 0x90);
    	
			devid = (FlashReadU8(CFG_FLASH_BASE + 0x02)) << 16;
			devid |= (FlashReadU8(CFG_FLASH_BASE + 0x1C)) << 8;
			devid |= FlashReadU8(CFG_FLASH_BASE + 0x1E);	
      FlashWriteU8(CFG_FLASH_BASE, 0xF0); 
      *DevID = devid;  
      
	#else
			
		
	
	#endif
		
} /* end flash_ReadID */