/*
 *  armboot - Startup Code for ARM940T CPU-core
 *
 *  Copyright (c) 2001	Marius Gr?er <mag@sysgo.de>
 *  Copyright (c) 2002	Alex Z?ke <azu@sysgo.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */



#include "config.h"
#include "version.h"
#if defined(CONFIG_S3C2500)
#include <s3c2500.h>
#endif
#if defined(CONFIG_S3C2510)
#include <s3c2510.h>
#endif

#define AMD_FLASH
#undef SAMSUNG_FLASH
#undef BDI2K

/*
 *************************************************************************
 *
 * Jump vector table as in table 3.1 in [1]
 *
 *************************************************************************
 */


.globl _start
_start:	b       reset
	ldr	pc, _undefined_instruction
	ldr	pc, _software_interrupt
	ldr	pc, _prefetch_abort
	ldr	pc, _data_abort
	ldr	pc, _not_used
	ldr	pc, _irq
	ldr	pc, _fiq

#ifdef CONFIG_UCBOOTSTRAP
		.space	28, 0			@
		.fill	1, 4, 0x485a4f00	@
		.word	reset

_undefined_instruction:	.word UndInstr
#else
_undefined_instruction:	.word undefined_instruction
#endif

_software_interrupt:	.word software_interrupt
_prefetch_abort:	.word prefetch_abort
_data_abort:		.word data_abort
_not_used:		.word not_used
_irq:			.word irq
_fiq:			.word fiq

	.balignl 16,0xdeadbeef


/*
 *************************************************************************
 *
 * Startup Code (reset vector)
 *
 * do important init only if we don't start from memory!
 * relocate armboot to ram
 * setup stack
 * jump to second stage
 *
 *************************************************************************
 */

/*
 * CFG_MEM_END is in the board dependent config-file (configs/config_BOARD.h)
 */
_TEXT_BASE:
	.word	TEXT_BASE

.globl _armboot_start
_armboot_start:
	.word _start

/*
 * Note: armboot_end is defined by the (board-dependent) linker script
 */
.globl _armboot_end
_armboot_end:
	.word armboot_end

/*
 * _armboot_real_end is the first usable RAM address behind armboot
 * and the various stacks
 */
.globl _armboot_real_end
_armboot_real_end:
	.word 0x0badc0de

#ifdef CONFIG_USE_IRQ
/* IRQ stack memory (calculated at run-time) */
.globl IRQ_STACK_START
IRQ_STACK_START:
	.word	0x0badc0de

/* IRQ stack memory (calculated at run-time) */
.globl FIQ_STACK_START
FIQ_STACK_START:
	.word 0x0badc0de
#endif


/*
 * the actual reset code
 */

#ifndef BDI2K
reset:
	/* wait for I/O device ready  */

	ldr		r0, =fMCLK*200
delay:
	subs		r0, r0,#1
	bne	 	delay

	/* turn on the LED connected with GPIO0 */

	ldr		r0, =IOPMODE1
	ldr		r1, =0xf003ff00
	str		r1, [r0]

/* 
 * 1. SYSTEM CONFIGURATION & INTERRUPT DISABLE
*/
	ldr		r0, =CLKCON
	ldr		r1, =0x0			/* System Clock Dividing by 0 - non-divided clock is used */
	str		r1, [r0]

	ldr		r0, =SYSCFG			/* Clear all SYSCFG except REMAP, BIG(Read-only) */
	ldr		r1, [r0]
	and		r1, r1, #REMAP
	str		r1, [r0]

	ldr		r0, =HPRIR
	ldr		r1, =0x330			/* Ethernet controllers have highest arbitor priority */
	str		r1, [r0]

	ldr		r0, =PCLKDIS		/* All Clock Enable */
	ldr		r1, =ALLCLKENABLE
	str		r1, [r0]

	ldr		r0, =INTMASK 			
	ldr   	r1, =0xffffffff		/* All Interrupt Disable */
	str     r1, [r0]	
	ldr		r0, =EXTMASK 			
   	ldr   	r1, =0x8000007f		/* All Interrupt Disable */
	str     r1, [r0]    				

	ldr		r0, =INTMOD			/* All IRQ Mode */
	ldr		r1, =0
	str		r1, [r0]
	ldr		r0, =EXTMOD			/* All IRQ Mode */
	str		r1, [r0]
	
/*	bl		Clear_AllPendingReg  */ /* this is s3c2500 specific, and not required in armboot */

/*
 * 2. SDRAM  & OTHER MEMORY CONFIGURATION
 *	reference User's Manual, MEMORY CONTROLLER chapter, SDRAM initialization sequence
 */
	/*
	 * <1 > Wait 200us to allow SDRAM power and clocks to stabilize.
	 *		Wait Time  =(fMCLK/MHz)*200
	 */
	ldr		r0, =fMCLK*200
wait1:
	subs	r0, r0,#1
	bne	 	wait1
 
	/*
	 * <2> PALL COMMAND  SET:
	 *		Program the INIT[1:0] to 01(PALL).
	 */
	ldr     r0, =CMDREG
	ldr     r1, =SDR_PALL
	str     r1, [r0]

	/*
	 * <3> Every 15 CLOCK CYCLE REFRESH
	 *	Write 0xF into the refresh timer register.
	 */
	ldr		r0, =REFREG
	ldr		r1, =0xF     
	str		r1, [r0]     

	/* 
	 * <4> WAIT  FUNCTION:
	 *	Wait for a time period equivalent to 120 clock cycles (8 refresh cycles).
	 */
	ldr		r1, =0x78
wait2:
	subs	r1, r1,#1
	bne		wait2

	/*
	 * <5> NORMAL  REFRESH  CYCLE
	 *	Normal operational value into the refresh timer.
	 *	Common refresh time 15.6 usec     
	 *  SDRAM refresh time = fMCLK*15 + (6*fMCLK)/10     
	 */
	ldr		r1, =(fMCLK*15 + (6*fMCLK)/10)
	str		r1, [r0]

	/*
	 * <6> Program CFGREG to their normal operation values
	 */
	ldr		r0, =CFGREG
	ldr		r1, =rSDRAM64MCFGREG
	str		r1, [r0]

	/*
	 *<7> MRS  COMMAND  SET
	 *	Program the INIT[1:0] to 10. This automatically issues a MRS command to the SDRAM.
	 */
	ldr     r0, =CMDREG
	ldr     r1, =SDR_MRS
	str     r1, [r0]

	/*
	 * <8> SyncDRAM Configration Register Setting
	 *	Program the INIT[1:0] to 00. The controller enters the normal mode.    
	 */
	ldr     r0, =CMDREG
	ldr     r1, =SDR_NORMAL
	str     r1, [r0]

	/*
	 * <9> Program CMDREG and WBTOREG to their normal operation values
	 */
	ldr		r0, =CMDREG
	ldr		r1, =rSDRAMCMDREG
	str		r1, [r0]
	ldr		r0, =WBTOREG
	ldr		r1, =rSDRAMWBTO
	ldr		r1, [r0]

	/*
	 * Initialize Memory Controller for EEPROM/FLASH/SRAM
	 */
	adrl    r0, MEMCON_INITTBL
	ldmia   r0, {r1-r10}
	ldr	r0, =ASIC_BASE + 0x10000 	/* B0CON offset : 0x10000 */
	stmia   r0, {r1-r10}   

/*
 * 4. GOTO SUPERVISOR MODE & IRQ/FIQ LOCKOUT
 */
	/* set the cpu to SVC32 mode*/
	mrs	r0,cpsr
	bic	r0,r0,#0x1f
	orr	r0,r0,#0x13
	msr	cpsr,r0

/*
 * 5. PROTECTION OFF/ SETTING ENDIAN/ CLOCKING MODE
 */
	mrc 	p15, 0, r0, c1, c0, 0
	bic		r0, r0, #PROTECTION_ON		/* Protection Off */
	mcr 	p15, 0, r0, c1, c0, 0

	mcr 	p15, 0, r1, c1, c0, 0		/* Read CP15 Reg, C1(Control Register) */
	ldr	 	r0, =(nFASTBUS | ASYNC_CLK | 0x80)
	bic		r1, r1, r0		/* Clear Clocking Mode bit, Endian bit */
	
	ldr		r0, =CLKST
	ldr		r0, [r0]		/* CLCOK MODE from MCU input setting */
	and		r0, r0, #(nFASTBUS | ASYNC_CLK)
	orr		r1, r1, r0
	ldr		r0, =SYSCFG
	ldr		r0, [r0]		/* ENDIAN from MCU input setting */
	and     r0, r0, #BIG				
	mov		r0, r0, LSR #9
	orr		r1, r1, r0
	mcr 	p15, 0, r1, c1, c0, 0		/* Set CP15, C1 with CLOCK MODE, ENDIAN MODE */

relocate:
	 /* relocate armboot to RAM*/
 	adr	r0, _start			/* r0 <- current position of code */
 	ldr	r2, _armboot_start
 	ldr	r3, _armboot_end
 	sub	r2, r3, r2			/* r2 <- size of armboot */
 	ldr	r1, =0x40000000			/* r1 <- destination address */
 	add	r2, r0, r2			/* r2 <- source end address */

	 /* r0 = source address
	 * r1 = target address
	 * r2 = source end address*/
copy_loop:
 	ldmia	r0!, {r3-r10}
 	stmia	r1!, {r3-r10}
 	cmp	r0, r2
 	ble	copy_loop

	/* rom to ram address swapping */
REMAP_START_FOR_ROM:
    ldr 	r0, =SYSCFG
    ldr		r1, =REMAP
    
	ldr		r2, [r0]
	orr		r2, r2, r1
	str		r2, [r0]
	
	nop
	nop
	nop
	nop
	nop

	/*********************************************************
	 * auto detect 32_64M SDRAM by TimYu 20051013
	 *********************************************************/
	ldr		r0, =0x00800000
	ldr		r1, =0x55AA55AA
	str		r1, [r0]
	ldr		r2, [r0]

	ldr		r0, =0x01800000
	ldr		r1, [r0]

	cmp		r2, r1 
	BEQ		reset_again_start
	b		reset_again_end
	
reset_again_start:
	/* turn on the LED connected with GPIO0 */

	ldr		r0, =IOPMODE1
	ldr		r1, =0xf003ff00
	str		r1, [r0]

/* 
 * 1. SYSTEM CONFIGURATION & INTERRUPT DISABLE
*/
	ldr		r0, =CLKCON
	ldr		r1, =0x0			/* System Clock Dividing by 0 - non-divided clock is used */
	str		r1, [r0]

	ldr		r0, =SYSCFG			/* Clear all SYSCFG except REMAP, BIG(Read-only) */
	ldr		r1, [r0]
	and		r1, r1, #REMAP
	str		r1, [r0]

	ldr		r0, =HPRIR
	ldr		r1, =0x330			/* Ethernet controllers have highest arbitor priority */
	str		r1, [r0]

	ldr		r0, =PCLKDIS		/* All Clock Enable */
	ldr		r1, =ALLCLKENABLE
	str		r1, [r0]

	ldr		r0, =INTMASK 			
	ldr   	r1, =0xffffffff		/* All Interrupt Disable */
	str     r1, [r0]	
	ldr		r0, =EXTMASK 			
   	ldr   	r1, =0x8000007f		/* All Interrupt Disable */
	str     r1, [r0]    				

	ldr		r0, =INTMOD			/* All IRQ Mode */
	ldr		r1, =0
	str		r1, [r0]
	ldr		r0, =EXTMOD			/* All IRQ Mode */
	str		r1, [r0]
	
/*	bl		Clear_AllPendingReg  */ /* this is s3c2500 specific, and not required in armboot */

/*
 * 2. SDRAM  & OTHER MEMORY CONFIGURATION
 *	reference User's Manual, MEMORY CONTROLLER chapter, SDRAM initialization sequence
 */
	/*
	 * <1 > Wait 200us to allow SDRAM power and clocks to stabilize.
	 *		Wait Time  =(fMCLK/MHz)*200
	 */
	ldr		r0, =fMCLK*200
wait3:
	subs	r0, r0,#1
	bne	 	wait3
 
	/*
	 * <2> PALL COMMAND  SET:
	 *		Program the INIT[1:0] to 01(PALL).
	 */
	ldr     r0, =CMDREG
	ldr     r1, =SDR_PALL
	str     r1, [r0]

	/*
	 * <3> Every 15 CLOCK CYCLE REFRESH
	 *	Write 0xF into the refresh timer register.
	 */
	ldr		r0, =REFREG
	ldr		r1, =0xF     
	str		r1, [r0]     

	/* 
	 * <4> WAIT  FUNCTION:
	 *	Wait for a time period equivalent to 120 clock cycles (8 refresh cycles).
	 */
	ldr		r1, =0x78
wait4:
	subs	r1, r1,#1
	bne		wait4

	/*
	 * <5> NORMAL  REFRESH  CYCLE
	 *	Normal operational value into the refresh timer.
	 *	Common refresh time 15.6 usec     
	 *  SDRAM refresh time = fMCLK*15 + (6*fMCLK)/10     
	 */
	ldr		r1, =(fMCLK*15 + (6*fMCLK)/10)
	str		r1, [r0]

	/*
	 * <6> Program CFGREG to their normal operation values
	 */
	ldr		r0, =CFGREG
	ldr		r1, =rSDRAM32MCFGREG
	str		r1, [r0]

	/*
	 *<7> MRS  COMMAND  SET
	 *	Program the INIT[1:0] to 10. This automatically issues a MRS command to the SDRAM.
	 */
	ldr     r0, =CMDREG
	ldr     r1, =SDR_MRS
	str     r1, [r0]

	/*
	 * <8> SyncDRAM Configration Register Setting
	 *	Program the INIT[1:0] to 00. The controller enters the normal mode.    
	 */
	ldr     r0, =CMDREG
	ldr     r1, =SDR_NORMAL
	str     r1, [r0]

	/*
	 * <9> Program CMDREG and WBTOREG to their normal operation values
	 */
	ldr		r0, =CMDREG
	ldr		r1, =rSDRAMCMDREG
	str		r1, [r0]
	ldr		r0, =WBTOREG
	ldr		r1, =rSDRAMWBTO
	ldr		r1, [r0]

	/*
	 * Initialize Memory Controller for EEPROM/FLASH/SRAM
	 */
	adrl    r0, MEMCON_INITTBL
	ldmia   r0, {r1-r10}
	ldr	r0, =ASIC_BASE + 0x10000 	/* B0CON offset : 0x10000 */
	stmia   r0, {r1-r10}   

/*
 * 4. GOTO SUPERVISOR MODE & IRQ/FIQ LOCKOUT
 */
	/* set the cpu to SVC32 mode*/
	mrs	r0,cpsr
	bic	r0,r0,#0x1f
	orr	r0,r0,#0x13
	msr	cpsr,r0

/*
 * 5. PROTECTION OFF/ SETTING ENDIAN/ CLOCKING MODE
 */
	mrc 	p15, 0, r0, c1, c0, 0
	bic		r0, r0, #PROTECTION_ON		/* Protection Off */
	mcr 	p15, 0, r0, c1, c0, 0

	mcr 	p15, 0, r1, c1, c0, 0		/* Read CP15 Reg, C1(Control Register) */
	ldr	 	r0, =(nFASTBUS | ASYNC_CLK | 0x80)
	bic		r1, r1, r0					/* Clear Clocking Mode bit, Endian bit */
	
	ldr		r0, =CLKST
	ldr		r0, [r0]					/* CLCOK MODE from MCU input setting */
	and		r0, r0, #(nFASTBUS | ASYNC_CLK)
	orr		r1, r1, r0
	ldr		r0, =SYSCFG
	ldr		r0, [r0]					/* ENDIAN from MCU input setting */
	and     r0, r0, #BIG				
	mov		r0, r0, LSR #9
	orr		r1, r1, r0
	mcr 	p15, 0, r1, c1, c0, 0		/* Set CP15, C1 with CLOCK MODE, ENDIAN MODE */

relocate_32m:
	 /* relocate armboot to RAM*/
 	adr	r0, _start					/* r0 <- current position of code */
 	ldr	r2, _armboot_start
 	ldr	r3, _armboot_end
 	sub	r2, r3, r2					/* r2 <- size of armboot */
 	ldr	r1, =0x40000000				/* r1 <- destination address */
 	add	r2, r0, r2					/* r2 <- source end address */

	 /* r0 = source address
	 * r1 = target address
	 * r2 = source end address*/

copy_loop_32m:
 	ldmia	r0!, {r3-r10}
 	stmia	r1!, {r3-r10}
 	cmp	r0, r2
 	ble	copy_loop_32m

	/* rom to ram address swapping */
REMAP_START_FOR_ROM_32M:
    ldr 	r0, =SYSCFG
    ldr		r1, =REMAP
    
	ldr		r2, [r0]
	orr		r2, r2, r1
	str		r2, [r0]
	
	nop
	nop
	nop
	nop
	nop

#else 

reset:
	/* set system parameters */
	ldr     r0, =SYSCFG
	ldr     r1, =rSYSCFG
	str     r1, [r0]

	/* rom to ram address swapping */
	adr     r0, SDRAM_SYSINIT_BOOT
	ldmia   r0, {r1-r10} 
	ldr     r0, =SYS_INIT_BASE
	stmia   r0, {r1-r10}
		
	/* set the cpu to SVC32 mode*/
	mrs		r0,	cpsr
	bic		r0,	r0,	#0x1f
	orr		r0,	r0,	#0x13
	msr		cpsr, r0

	/* cache flushing */
	ldr	r0, =0x11000000
	mov	r1, #0
	mov	r2, #256

Cache_Flush_Loop:
	str     r1, [r0], #4
	subs    r2, r2, #1
	bne     Cache_Flush_Loop
#endif

reset_again_end:
		
	/* set up the stack */
	ldr	r0, _armboot_end
	add	r0, r0, #CONFIG_STACKSIZE
	sub	sp, r0, #12		/* leave 3 words for abort-stack */
	
	ldr	pc, _start_armboot
		
_start_armboot:	.word start_armboot

/*
 *************************************************************************
 *
 * Clear_AllPendingReg
 *
 * Clear All pending register
 *
 *************************************************************************
 */
Clear_AllPendingReg:

	/*
	 * nEXT0_INT (0), nEXT1_INT (1), nEXT2_INT (2), nEXT3_INT (3)
	 * nEXT4_INT (4), nEXT5_INT (5)
	 */
	 
	ldr		r0, =IOPEXTINTPND
	ldr		r1, =0x3f
	str		r1, [r0, #0]

	/*
	 * nIOM2_INT (6)
	 */
	
	ldr		r0, =IOM2STAT
	ldr		r1, =0x1ff5
	str		r1, [r0, #0]

	/*
	 * nIICC_INT (7)
	 */

	ldr		r0, =IICPND
	ldr		r1, =0x1
	str		r1, [r0, #0]

	/*
	 * nHUART0_TX_INT (8), nHUART0_RX_INT (9)
	 * nHUART1_TX_INT (10), nHUART1_RX_INT (11)
	 */

	ldr		r0, =HUSTAT0
	ldr		r1, =0x714ff
	str		r1, [r0, #0]
	ldr		r0, =HUSTAT1
	str		r1, [r0, #0]

	/*
	 * nCUART_TX_INT (12), nCUART_RX_INT (13)
	 */

	ldr		r0, =CUSTAT
	ldr		r1, =0x6003f
	str		r1, [r0, #0]
	
	/*
	 * nUSB_INT (14)
	 */

	ldr		r0, =USBINTR
	ldr		r1, =0xf1f
	str		r1, [r0, #0]

	/*
	 * nHDLC_TX0_INT (15), nHDLC_RX0_INT (16), nHDLC_TX1_INT (17), nHDLC_RX1_INT (18)
	 * nHDLC_TX2_INT (19), nHDLC_RX2_INT (20)
	 */

	ldr		r0, =HSTAT0
	ldr		r1, =0xedcffff0
	str		r1, [r0, #0]
	ldr		r0, =HSTAT1
	str		r1, [r0, #0]
	ldr		r0, =HSTAT2
	str		r1, [r0, #0]

	/*
	 * nMACA_TX_INT (21), nMACA_RX_INT (22), nMACB_TX_INT (23), nMACB_RX_INT (24)
	 */

	ldr		r0, =BMTXSTATA			/* For S3C2500 MAC, writing '0' will erase status register */
	ldr		r1, =0x0
	str		r1, [r0, #0]
	ldr		r0, =BMTXSTATB
	str		r1, [r0, #0]
	
	ldr		r0, =BMRXSTATA
	ldr		r1, =0x0
	str		r1, [r0, #0]
	ldr		r0, =BMRXSTATB
	str		r1, [r0, #0]
	
	/*
	 * nDES_INT (25)				; DES Interrupt disable
	 */

	ldr		r0, =DESINT
	ldr		r1, =0x0
	str		r1, [r0, #0]

	/* 
	 * nGDMA0_INT (26), nGDMA1_INT (27), nGDMA2_INT (28), nGDMA3_INT (29),
	 * nGDMA4_INT (30), nGDMA5_INT (31)
	 */

	ldr		r0, =DIPR0
	ldr		r1, =0x1
	str		r1, [r0, #0]
	ldr		r0, =DIPR1
	str		r1, [r0, #0]
	ldr		r0, =DIPR2
	str		r1, [r0, #0]
	ldr		r0, =DIPR3
	str		r1, [r0, #0]
	ldr		r0, =DIPR4
	str		r1, [r0, #0]
	ldr		r0, =DIPR5
	str		r1, [r0, #0]
	
	/*
	 * nTIMER0_INT (32), nTIMER1_INT (33), nTIMER2_INT (34), nTIMER3_INT (35),
	 * nTIMER4_INT (36), nTIMER5_INT (37), nWATCHDOG_INT (38)
	 */

	ldr		r0, =TIMER_TIC
	ldr		r1, =0x7f
	str		r1, [r0, #0]

	mov		pc, lr

/*
 *************************************************************************
 *
 * CPU_init_critical registers
 *
 * setup important registers
 * setup memory timing
 *
 *************************************************************************
 */
MEMCON_INITTBL:
	.word 	rB0CON			/* EXT I/O Bank #0 Control Register */
	.word 	rB1CON			/* EXT I/O Bank #1 Control Register */		
	.word 	rB2CON			/* EXT I/O Bank #2 Control Register */
	.word 	rB3CON			/* EXT I/O Bank #3 Control Register */
	.word 	rB4CON			/* EXT I/O Bank #4 Control Register */
	.word 	rB5CON			/* EXT I/O Bank #5 Control Register */
	.word 	rB6CON			/* EXT I/O Bank #6 Control Register */
	.word 	rB7CON			/* EXT I/O Bank #7 Control Register */
	.word 	rMUXBCON		/* Muxed bus Control Register */
	.word 	rWAITCON		/* Wait Control Register */
/*
 *************************************************************************
 *
 * Interrupt handling
 *
 *************************************************************************
 */

@
@ IRQ stack frame.
@
#define S_FRAME_SIZE	72

#define S_OLD_R0	68
#define S_PSR		64
#define S_PC		60
#define S_LR		56
#define S_SP		52

#define S_IP		48
#define S_FP		44
#define S_R10		40
#define S_R9		36
#define S_R8		32
#define S_R7		28
#define S_R6		24
#define S_R5		20
#define S_R4		16
#define S_R3		12
#define S_R2		8
#define S_R1		4
#define S_R0		0

#define MODE_SVC 0x13
#define I_BIT	 0x80

/*
 * use bad_save_user_regs for abort/prefetch/undef/swi ...
 * use irq_save_user_regs / irq_restore_user_regs for IRQ/FIQ handling
 */

	.macro	bad_save_user_regs
	sub	sp, sp, #S_FRAME_SIZE
	stmia	sp, {r0 - r12}			@ Calling r0-r12
	add     r8, sp, #S_PC

	ldr	r2, _armboot_end
	add	r2, r2, #CONFIG_STACKSIZE
	sub	r2, r2, #8
	ldmia	r2, {r2 - r4}                   @ get pc, cpsr, old_r0
	add	r0, sp, #S_FRAME_SIZE		@ restore sp_SVC

	add	r5, sp, #S_SP
	mov	r1, lr
	stmia	r5, {r0 - r4}                   @ save sp_SVC, lr_SVC, pc, cpsr, old_r
	mov	r0, sp
	.endm

	.macro	irq_save_user_regs
	sub	sp, sp, #S_FRAME_SIZE
	stmia	sp, {r0 - r12}			@ Calling r0-r12
	add     r8, sp, #S_PC
	stmdb   r8, {sp, lr}^                   @ Calling SP, LR
	str     lr, [r8, #0]                    @ Save calling PC
	mrs     r6, spsr
	str     r6, [r8, #4]                    @ Save CPSR
	str     r0, [r8, #8]                    @ Save OLD_R0
	mov	r0, sp
	.endm

	.macro	irq_restore_user_regs
	ldmia	sp, {r0 - lr}^			@ Calling r0 - lr
	mov	r0, r0
	ldr	lr, [sp, #S_PC]			@ Get PC
	add	sp, sp, #S_FRAME_SIZE
	subs	pc, lr, #4			@ return & move spsr_svc into cpsr
	.endm

	.macro get_bad_stack
	ldr	r13, _armboot_end		@ setup our mode stack
	add	r13, r13, #CONFIG_STACKSIZE	@ resides at top of normal stack
	sub	r13, r13, #8

	str	lr, [r13]			@ save caller lr / spsr
	mrs	lr, spsr
	str     lr, [r13, #4]

	mov	r13, #MODE_SVC			@ prepare SVC-Mode
	msr	spsr_c, r13
	mov	lr, pc
	movs	pc, lr
	.endm

	.macro get_irq_stack			@ setup IRQ stack
	ldr	sp, IRQ_STACK_START
	.endm

	.macro get_fiq_stack			@ setup FIQ stack
	ldr	sp, FIQ_STACK_START
	.endm

/*
 * exception handlers
 */
	.align  5
undefined_instruction:
	get_bad_stack
	bad_save_user_regs
	bl 	do_undefined_instruction

	.align	5
software_interrupt:
	get_bad_stack
	bad_save_user_regs
	bl 	do_software_interrupt

	.align	5
prefetch_abort:
	get_bad_stack
	bad_save_user_regs
	bl 	do_prefetch_abort

	.align	5
data_abort:
	get_bad_stack
	bad_save_user_regs
	bl 	do_data_abort

	.align	5
not_used:
	get_bad_stack
	bad_save_user_regs
	bl 	do_not_used

#ifdef CONFIG_USE_IRQ

	.align	5
irq:
	get_irq_stack
	irq_save_user_regs
	bl 	do_irq
	irq_restore_user_regs

	.align	5
fiq:
	get_fiq_stack
	/* someone ought to write a more effiction fiq_save_user_regs */
	irq_save_user_regs
	bl 	do_fiq
	irq_restore_user_regs

#else

	.align	5
irq:
	get_bad_stack
	bad_save_user_regs
	bl 	do_irq

	.align	5
fiq:
	get_bad_stack
	bad_save_user_regs
	bl 	do_fiq

#endif

	.align	5


@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@ CACHE Setting    choish 20020827
.global _ICache_Enable
_ICache_Enable:
		mrc	15, 0, r0, c1, c0, 0
		orr	r0, r0, #0x1000
		mcr	15, 0, r0, c1, c0, 0
		mov	pc, lr


		.global _ICache_Disable
_ICache_Disable:
		mrc	15, 0, r0, c1, c0, 0
		bic	r0, r0, #0x1000
		mcr	15, 0, r0, c1, c0, 0
		mov	pc, lr

		.global _ICache_Flush
_ICache_Flush:
		mov	r0, #0
		mcr	15, 0, r0, c7, c5, 0
		mov	pc, lr

		.global _ICache_Flush_Single
_ICache_Flush_Single:
		mov	r0, r0, lsl #4	@ Segment
		mov	r1, r1, lsl #26	@ Line
		orr	r0, r0, r1
		mcr	15, 0, r0, c7, c5, 2
		mov	pc, lr
		
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ DCache
		.global _DCache_Enable
_DCache_Enable:
		mrc	15, 0, r0, c1, c0, 0
		orr	r0, r0, #0x4
		mcr	15, 0, r0, c1, c0, 0
		mov	pc, lr

		.global _DCache_Disable
_DCache_Disable:
		mrc	15, 0, r0, c1, c0, 0
		bic	r0, r0, #0x4
		mcr	15, 0, r0, c1, c0, 0
		mov	pc, lr

		.global _DCache_Flush
_DCache_Flush:
		mov	r0, #0
		mcr	15, 0, r0, c7, c6, 0
		mov	pc, lr

		.global _DCache_Flush_Single
_DCache_Flush_Single:
		mov	r0, r0, lsl #4	@ Segment
		mov	r1, r1, lsl #26	@ Line
		orr	r0, r0, r1
		mcr	15, 0, r0, c7, c6, 2
		mov	pc, lr

		.global _DCache_Clean_Single
_DCache_Clean_Single:
		mov	r0, r0, lsl #4	@ Segment
		mov	r1, r1, lsl #26	@ Line
		orr	r0, r0, r1
		mcr	15, 0, r0, c7, c10, 2
		mov	pc, lr

		.global _DCache_CFlush_Single	@ Clean & flush
_DCache_CFlush_Single:
		mov	r0, r0, lsl #4	@ Segment
		mov	r1, r1, lsl #26	@ Line
		orr	r0, r0, r1
		mcr	15, 0, r0, c7, c14, 2
		mov	pc, lr

		.global _DCache_CFlush
_DCache_CFlush:
		stmfd	sp!, {r1,r2}
		mov	r1, #0
1:		mov	r0, #0
2:		orr	r2, r1, r0
		mcr	15, 0, r0, c7, c14, 2
		add	r0, r0, #0x10
		cmp	r0, #0x40
		bne	2b
		add	r1, r1, #0x04000000
		cmp	r1, #0
		bne	1b
		ldmfd	sp!, {r1,r2}
		mov	pc, lr
		
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ Write Buffer WB
@ 		void	_WB_Enable (int region);

		.global _WB_Enable
_WB_Enable:	stmfd	sp!, {r1,r2}
		mrc	15, 0, r1, cr3, cr0, 0
		mov	r2, #1
		orr	r1, r1, r2, lsl r0
		mcr	15, 0, r1, cr3, cr0, 0
		ldmfd	sp!, {r1,r2}
		mov	pc, lr

@ 		void	_WB_Disable (int region);

		.global _WB_Disable
_WB_Disable:	stmfd	sp!, {r1,r2}
		mrc	15, 0, r1, cr3, cr0, 0
		mov	r2, #1
		bic	r1, r1, r2, lsl r0
		mcr	15, 0, r1, cr3, cr0, 0
		ldmfd	sp!, {r1,r2}
		mov	pc, lr

		.global _WB_Drain
_WB_Drain:	mov	r0, #0
		mcr	15, 0, r0, cr7, cr10, 4
		mov	pc, lr
		

@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@ Protection Unit setting  choish 20020827

.globl reset_cpu
reset_cpu:
@@@@@@@@@@@@@@@@@@@@@@@
@check the sdram is 32M or 64M

	ldr		r0, =0xf0020000
	ldr		r1, =rSDRAM64MCFGREG
	ldr		r2, [r0]
	cmp		r2, r1 
	BEQ		sdram_config_64m_start
	b		sdram_config_32m_start
	
@@@@@@@@@@@@@@@@@@@@@@@
@ instruction region set
sdram_config_64m_start:
	ldr	R1, =0x0000003F	 	@ code  region 0 4Gbyte - background
	MCR	p15, 0, R1, c6, c0, 1
	ldr	R1, =0x00000033		@ code	region 1 64MB - SDRAM
	MCR p15, 0, R1, c6, c1, 1	


@@@@@@@@@@@@@@@@@@@@@@@
@ data region set
	ldr	R1, =0x0000003F	 	@ data  region 0 4Gbyte - background
	MCR	p15, 0, R1, c6, c0, 0
	ldr	R1, =0x00000033		@ data	region 1 64MB - SDRAM
	MCR p15, 0, R1, c6, c1, 0	
	ldr	R1, =0x0000002D		@ data	region 2 8MB - SDRAM dma
	MCR p15, 0, R1, c6, c2, 0
@	TimYu add 
	ldr	R1, =0x0180002D		@ data	region 3 24 - 32MB - no cache
	MCR p15, 0, R1, c6, c3, 0
@	TimYu end 

	
	mov	R1, #0x2		@ code Region Cachable bit setting
	MCR	p15, 0, R1, c2, c0, 1
	@ 1 cachable
	mov	R1, #0x2		@ data Region Cachable bit setting
	MCR	p15, 0, R1, c2, c0, 0
	@  1 cachable

	mov	R1, #0x4		@ Write Buffer Region setting
	MCR	p15, 0, R1, c3, c0, 0	@ DMA access region select
	
	ldr	R1,=0xfff
	MCR     p15,0,R1,c5,c0,1   	@ enable full access on Inst Region 0-5
	MCR     p15,0,R1,c5,c0,0   	@ enable full access on Data Region 0-5

	MRC	p15, 0, r0, c1, c0	@ get control register
	orr	r0, r0, #1		@ Enable PU
	ldr	r1, =0x3fffcf02		@ all "SB0"
	bic	r0, r0, r1
	orr	r0, r0, #0x0078		@ all "SB1"
	bic	r0, r0, #0x2000		@ No alternate vectors
	bic	r0, r0, #0x0080		@ Little-endian
	orr	r0, r0, #0x0004		@ disable DCache
	orr	r0, r0, #0x1000		@ Enable ICache
@	and	r0, r0, #0x3fffffff	@ Set Fast bus mode	@ Impossible to set bus mode !?
	orr	r0, r0, #0xC0000000	@ Set Async bus mode
	
	MCR     p15,0,R0,c1,c0,0
	
	nop;nop;nop
	mov	pc, lr

@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@ Protection Unit setting  choish 20020827
@ Protection Unit setting  TimYu for support 32 - 64M auto detect 20051013

@@@@@@@@@@@@@@@@@@@@@@@
@ instruction region set
sdram_config_32m_start:
	ldr	R1, =0x0000003F	 	@ code  region 0 4Gbyte - background
	MCR	p15, 0, R1, c6, c0, 1
	ldr	R1, =0x00000031		@ code	region 1 32MB - SDRAM
	MCR	p15, 0, R1, c6, c1, 1	

@@@@@@@@@@@@@@@@@@@@@@@
@ data region set
	ldr	R1, =0x0000003F	 	@ data  region 0 4Gbyte - background
	MCR	p15, 0, R1, c6, c0, 0
	ldr	R1, =0x00000031		@ data	region 1 32MB - SDRAM
	MCR	p15, 0, R1, c6, c1, 0	
	ldr	R1, =0x0000002D		@ data	region 2 8MB - SDRAM
	MCR	p15, 0, R1, c6, c2, 0
@	TimYu add 
	ldr	R1, =0x0080002D		@ data	region 2 16MB - SDRAM
	MCR	p15, 0, R1, c6, c3, 0
@	TimYu end 

	mov	R1, #0x2		@ code Region Cachable bit setting
	MCR	p15, 0, R1, c2, c0, 1
	@ 1 cachable
	mov	R1, #0x2		@ data Region Cachable bit setting
	MCR	p15, 0, R1, c2, c0, 0
	@  1 cachable

	mov	R1, #0x4		@ Write Buffer Region setting
	MCR	p15, 0, R1, c3, c0, 0	@ DMA access region select
	
	ldr	R1,=0xfff
	MCR     p15,0,R1,c5,c0,1   	@ enable full access on Inst Region 0-5
	MCR     p15,0,R1,c5,c0,0   	@ enable full access on Data Region 0-5

	MRC	p15, 0, r0, c1, c0	@ get control register
	orr	r0, r0, #1		@ Enable PU
	ldr	r1, =0x3fffcf02		@ all "SB0"
	bic	r0, r0, r1
	orr	r0, r0, #0x0078		@ all "SB1"
	bic	r0, r0, #0x2000		@ No alternate vectors
	bic	r0, r0, #0x0080		@ Little-endian
	orr	r0, r0, #0x0004		@ disable DCache
	orr	r0, r0, #0x1000		@ Enable ICache
@	and	r0, r0, #0x3fffffff	@ Set Fast bus mode	@ Impossible to set bus mode !?
	orr	r0, r0, #0xC0000000	@ Set Async bus mode
	
	MCR     p15,0,R0,c1,c0,0
	
	nop;nop;nop
	mov	pc, lr

/*
 *************************************************************************
 *
 * Below is for connection with uClinux environment.
 * And, 
 *
 *************************************************************************
 */
#ifdef CONFIG_UCBOOTSTRAP

	    .global	UndInstr
UndInstr:
	    str		r6, PtTmp	@ save r6 & r7
	    str		r7, PtTmp+4
	    ldr		r6, [lr, #-4]	@ get code of undef instr
	    ldr		r7, =0xe65a4f10	@ bootstrap magic number
	    cmp		r6, r7			@ is it bootstrap syscall
	    ldral	r6, PtTmp		@ restore r6 & r7
	    ldral	r7, PtTmp+4
	    beq		.+12

	    @---------------------------< Kernel undefined instr >
	    mov		pc, $0xe4	@ 0xe4 is a kernel-2.0 vector_undefinstr addres.
	    nop				@ For compability it is still here for kernel-2.4.
	    nop				@ But to make it work the following was done :
	    nop				@ 1. from this address to the 0x200 (kernel __stub_start)
					@    memory was fillout by "nop" instructions.
					@ 2. in the kernel-2.4 the vector_undefinstr was moved to
					@    the __stub_start.
	    
	    @---------------------------< Bootstrap system call >
@ hans, changed stack area to 0x4000 ( empty area in .text, see system.map file )
@<--
@	    ldr		sp, =0x00800000	@ Bootstrap stack pointer
@-->
	    ldr		sp, =0x4000	@ Bootstrap stack pointer
@-->

	    stmfd	sp, {lr}	@ save return register
	    sub		sp, sp, #4
	    stmfd	sp, {r0-r12}	@ save work registers
	    sub		sp, sp, #52
	    ldr		r6, PtCallTable @ r6 <- function table address
	    str		r4, [sp, #-4]!	@ push r4->sp 5th argument
	    mov		lr, pc		@ save return address	
	    ldr		pc, [r6, r10, lsl #2] @ bootstrap syscall
	    add		sp, sp, #8	
	    ldmfd	sp, {r1-r12}	@ restore work reg without R0
	    add		sp, sp, #48
	    ldmfd	sp, {pc}^	@ return 

@--------------------------------------------------------------	    
	    .global	bsc_nullsyscall
bsc_nullsyscall:
	    mov		pc, lr	    

@--------------------------------------------------------------
		.global go2
go2:		mov	pc, r0
		nop

@--------------------------------------------------------------
PtCallTable:
                                /* not implemented */
        .word   .+4
        .word   bsc_reset       @ bsc_reset         @ 0
        .word   bsc_test                            @ 1
        .word   bsc_nullsyscall                     @ 2
        .word   bsc_nullsyscall                     @ 3
        .word   bsc_nullsyscall @ bsc_program           @ 4
        .word   bsc_nullsyscall                     @ 5
        .word   bsc_nullsyscall                     @ 6
        .word   bsc_nullsyscall                     @ 7
        .word   bsc_nullsyscall                     @ 8
        .word   bsc_nullsyscall                     @ 9
        .word   bsc_nullsyscall                     @ 10
        .word   bsc_nullsyscall                     @ 11
        .word   bsc_gethwaddr                       @ 12
        .word   bsc_getserialnum                    @ 13
        .word   bsc_getenv                      @ 14
        .word   bsc_setenv                      @ 15
        .word   bsc_setpmask                        @ 16
        .word   bsc_readenv @ bsc_readenv           @ 17
        .word   flash_chattr_range @ flash_chattr_range @ 18
        .word   flash_erase_range @ flash_erase_range   @ 19
        .word   flash_write_range@ flash_write_range        @ 20
        .word   bsc_ramload @ bsc_ramload           @ 21    
@--------------------------------------------------------------	    
	    .global	bsp
bsp:
	    nop
@-------------------------------------------------------------

PtTmp:	    .word	6
	    .word	7
#endif

	.ltorg
#if defined (AM29LV160D) || defined (HY29LV320B)
	.org 0x6000
#elif defined (SST39VF160)
	.org 0x1000
#else
	.org 0x4000
#endif
