/*
 * CAST  MAC Core
 *
 * (C) Copyright 2008
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * Copyright (C) 2008 ZhengXingjian <zheng_xingjian@dahutech.com>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is loaded into SRAM in bootstrap mode, where it waits
 * for commands on UART1 to read and write memory, jump to code etc.
 * A design goal for this program is to be entirely independent of the
 * target board.  Anything with a CL-PS7111 or EP7211 should be able to run
 * this code in bootstrap mode.  All the board specifics can be handled on
 * the host.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "armboot.h"
#include "command.h"
#include "immac.h"
#include "net.h"

#ifdef CONFIG_DRIVER_IMMAC

#if (CONFIG_COMMANDS & CFG_CMD_NET)

#define	MAC_CACHE
//#undef MAC_CACHE

const  unsigned long CSKYMACADDR[2] = { 0x00800000, 0x00000041 };


#define mdio_delay() (temp = CKMAC_REG_BASEADDR[CSR9])


#define MDIO_SHIFT_CLK		0x10000
#define MDIO_DATA_WRITE0	0x00000
#define MDIO_DATA_WRITE1	0x20000
#define MDIO_ENB		0x00000 /* Ignore the 0x02000 databook setting. */
#define MDIO_ENB_IN		0x40000
#define MDIO_DATA_READ		0x80000

#define IMMAC_CUR_SETUPF	0
#define IMMAC_CUR_TX		1
#define IMMAC_CUR_RX		0

struct immac_descriptor immac_des;

/* packet page register access functions */

/*
unsigned int get_reg(int regno)
{
	return (CKMAC_REG_BASEADDR[regno]);
}*/

/*
void put_reg(int regno, unsigned short val)
{
	//
}
*/

void eth_reset(void)
{
	CKMAC_REG_BASEADDR[CSR0] =  0x100181;
	udelay(2000);
	CKMAC_REG_BASEADDR[CSR0] =  0x100180; //set bus mode
}

unsigned long immac_phyr_read(unsigned char phyr)
{

	
	int i;
	int read_cmd = (0xf6 << 10) | (( CKMAC_PHY_ADDR & 0x1f) << 5) | phyr;
	int retval = 0;

	int temp; //for mdio_delay

	unsigned long flags;

	if (phyr & ~0x1f)
		return 0xffff;

	/* Establish sync by sending at least 32 logic ones. */
	for (i = 32; i >= 0; i--) {
		//outl(MDIO_ENB | MDIO_DATA_WRITE1, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | MDIO_DATA_WRITE1;
		mdio_delay();
		//outl(MDIO_ENB | MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK;
		mdio_delay();
	}
	/* Shift the read command bits out. */
	for (i = 15; i >= 0; i--) {
		int dataval = (read_cmd & (1 << i)) ? MDIO_DATA_WRITE1 : 0;

		//outl(MDIO_ENB | dataval, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9]  = MDIO_ENB | dataval;
		mdio_delay();
		//outl(MDIO_ENB | dataval | MDIO_SHIFT_CLK, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | dataval | MDIO_SHIFT_CLK;
		mdio_delay();
	}
	/* Read the two transition, 16 data, and wire-idle bits. */
	for (i = 19; i > 0; i--) {
		//outl(MDIO_ENB_IN, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9]  = MDIO_ENB_IN;
		mdio_delay();
		retval = (retval << 1) | ((CKMAC_REG_BASEADDR[CSR9]  & MDIO_DATA_READ) ? 1 : 0);
		//outl(MDIO_ENB_IN | MDIO_SHIFT_CLK, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB_IN | MDIO_SHIFT_CLK;
		mdio_delay();
	}

	return (retval>>1) & 0xffff;
}

void immac_phyr_write(unsigned char phyr, unsigned long w_value)
{
	
	int i;
	int cmd = (0x5002 << 16) | ((CKMAC_PHY_ADDR & 0x1f) << 23) | (phyr<<18) | (w_value& 0xffff);
	
	unsigned long flags;
	int temp; //for mdio_delay

	if (phyr& ~0x1f)
		return;	

	/* Establish sync by sending 32 logic ones. */
	for (i = 32; i >= 0; i--) 
	{
		//outl(MDIO_ENB | MDIO_DATA_WRITE1, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | MDIO_DATA_WRITE1;
		mdio_delay();
		//outl(MDIO_ENB | MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK;
		mdio_delay();
	}
	/* Shift the command bits out. */
	for (i = 31; i >= 0; i--) 
	{
		int dataval = (cmd & (1 << i)) ? MDIO_DATA_WRITE1 : 0;
		//outl(MDIO_ENB | dataval, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | dataval;
		mdio_delay();
		//outl(MDIO_ENB | dataval | MDIO_SHIFT_CLK, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | dataval | MDIO_SHIFT_CLK;
		mdio_delay();
	}
	/* Clear out extra bits. */
	for (i = 2; i > 0; i--) 
	{
		//outl(MDIO_ENB_IN, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB_IN;
		mdio_delay();
		//outl(MDIO_ENB_IN | MDIO_SHIFT_CLK, mdio_addr);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB_IN | MDIO_SHIFT_CLK;
		mdio_delay();
	}
}

void immac_get_macaddr (unsigned char macaddr[])
{
	unsigned long  temp[2];
	
	temp[1] = CSKYMACADDR[1];
	temp[0] = CSKYMACADDR[0];
	
	//temp[1] = CKMAC_REG_BASEADDR[CSR13];
	//temp[0] = CKMAC_REG_BASEADDR[CSR12];

	memcpy(macaddr, (unsigned char *)&temp[1]+2, 0x02);
	memcpy(&macaddr[2], (unsigned char *)&temp[0], 0x04);
	
}

void immac_set_macaddr ( unsigned char macaddr[] )
{
	int i;
	unsigned long  temp[2];
	
	memcpy((unsigned char *)&temp[1]+2,macaddr, 0x02);
	memcpy((unsigned char *)&temp[0], &macaddr[2],0x04);
	
	CSKYMACADDR[1] = temp[1];
	CSKYMACADDR[0] = temp[0];
}

inline void immac_start_rxtx(void)
{
	CKMAC_REG_BASEADDR[CSR6] = CKMAC_REG_BASEADDR[CSR6] | RxTx;
	udelay(5);
}

inline void immac_stop_rxtx(void)
{
	if (CKMAC_REG_BASEADDR[CSR6] & RxTx) 
	{
		unsigned i=1300/100;
		
		CKMAC_REG_BASEADDR[CSR6] = CKMAC_REG_BASEADDR[CSR6] & ~RxTx;
		
		/* wait until in-flight frame completes.
		 * Max time @ 10BT: 1500*8b/10Mbps == 1200us (+ 100us margin)
		 * Typically expect this loop to end in < 50 us on 100BT.
		 */
		while (--i && (CKMAC_REG_BASEADDR[CSR5]& (CSR5_TS|CSR5_RS)))
			udelay(10);

		if (!i)
			printf("Debug Message: immac_stop_rxtx() failed\n");
	}
}

/* Well, really we just force the chip into 100baseT then
 * 10baseT, each time checking for a link status.
 */

int immac_begin_auto_negotiation(void)
{
	unsigned long	tmp;
	int 		timeout;

	// Reset the PHY. 
	tmp = RTL8201_MODECTRL_RESET;
	immac_phyr_write(RTL8201_MODECTRL, tmp);

	timeout = 64;
	while (--timeout) 
	{
		tmp = immac_phyr_read(RTL8201_MODECTRL);
		if ((tmp & RTL8201_MODECTRL_RESET) == 0)
		{
			break;
		}
		udelay(20);
	}
	
	if (timeout == 0)
	{
		printf ("CKCore Error %s : PHY reset failed.\n", IMMAC_DEVICE_NAME);
		return -1;
	}

	tmp = immac_phyr_read(RTL8201_MODECTRL);

	// First we try auto-negotiation (not 100baseT). 
	//tmp = RTL8201_MODECTRL_SPD100 | RTL8201_MODECTRL_DUPLEX | RGL8201_MODECTRL_LOOPBACK;
	tmp = RTL8201_MODECTRL_ANE ;
	immac_phyr_write(RTL8201_MODECTRL, tmp);

	tmp = immac_phyr_read(RTL8201_MODESTS);

	if (tmp & RTL8201_MODESTS_LINKSTS)
	{
		timeout = 1000;
		while(--timeout)
		{
			tmp = immac_phyr_read(RTL8201_MODESTS);
			if(tmp&RTL8201_MODESTS_ANC)
				break;
			udelay(2000);
		}
		
		//printf("tmp = %x %x \n", tmp, timeout);
		printf("\n");
		
		if(timeout==0)
		{
			printf("auto negotiation is time out \n");
			return -1;
			//dp->csr6 |= CSR6_TTM;
			//dp->csr6 &= ~CSR6_FD;
		}
		
		
		if(tmp&RTL8201_MODESTS_100BTXFD)
			{
				printf("immac phy 100 fd\n");
				//dp->csr6 &= ~CSR6_TTM;
				//dp->csr6 |= CSR6_FD;
			}
		else if(tmp&RTL8201_MODESTS_100BTXHD)
			{
				printf("immac phy 100 hd\n");
				//dp->csr6 &= ~CSR6_TTM;
				//dp->csr6 &= ~CSR6_FD;
			}
		else if(tmp&RTL8201_MODESTS_10BTFD)
			{
				printf("immac phy 10 fd\n");
				//dp->csr6 |= CSR6_TTM;
				//dp->csr6 |= CSR6_FD;
			}
		else if(tmp&RTL8201_MODESTS_10BTHD)
			{
				printf("immac phy 10 hd\n");
				//dp->csr6 |= CSR6_TTM;
				//dp->csr6 &= ~CSR6_FD;
			}
		
	}
	else
	{
		printf("not link \n");
		return -1;
		//dp->csr6 |= CSR6_TTM;
		//dp->csr6 &= ~CSR6_FD;
	}
	
	return 0;
}

int immac_rings_init(struct immac_descriptor *ring)
{
	int i;
	
	//recevie
	for( i=0; i<3; i++ )
	{
		ring->rev_des[i].DES0 |= DescOwned;
		ring->rev_des[i].DES1 = PKTSIZE_ALIGN;
		ring->rev_des[i].DES2 = &(ring->rbuf[ring->rx_cur][0]);
		//ring->rev_des[i].DES2 = NetRxPackets[1];
		
		if( 0 == i )
			ring->rev_des[i].DES0 |= 0x200;
		else if( 2 == i )
			ring->rev_des[i].DES0 |= 0x100;
	}
	
	for( i=0; i<3; i++ )
	{
		ring->mit_des[i].DES0 = DescOwned;
		//if( 0 == i )
			//ring->mit_des[i].DES2 = &(ring->tbuf[0]);	
	}

	CKMAC_REG_BASEADDR[CSR3] = &(ring->rev_des);
	CKMAC_REG_BASEADDR[CSR4] = &(ring->mit_des);	
	
	//ring->rx_cur = 0;
	
	//printf("ring addr : %#x  immac_des addr : %#x\n", ring,&immac_des);
	//printf("CKMAC_REG_BASEADDR[CSR3] %#x\n", CKMAC_REG_BASEADDR[CSR3]);
	//printf("CKMAC_REG_BASEADDR[CSR4] %#x\n", CKMAC_REG_BASEADDR[CSR4]);
}

u16 setup_frm[96];
int immac_ether_setupframe(u32 ringindex, bd_t *bd)
{
	u16 *eaddrs = (u16 *)bd->bi_enetaddr;

	/* 21140 bug: you must add the broadcast address. */
	memset(setup_frm, 0xff, sizeof(setup_frm));
	
	/* Fill the final entry of the table with our physical address. */
	setup_frm[0] = eaddrs[0]; 
	setup_frm[1] = eaddrs[0];
	
	setup_frm[2] = eaddrs[1]; 
	setup_frm[3] = eaddrs[1];
	
	setup_frm[4] = eaddrs[2]; 
	setup_frm[5] = eaddrs[2];

	/* Put the setup frame on the Tx list. */
	immac_des.mit_des[ringindex].DES0 = DescOwned;
	immac_des.mit_des[ringindex].DES1 = (0x08000000 | 192);
	immac_des.mit_des[ringindex].DES2 = &setup_frm;
	
}

int immac_ether_init(bd_t *bd)
{
	int i;
	printf("CKCore immac_ether_init \n");
	
	CKMAC_REG_BASEADDR[CSR0] =   0x100181;

	printf("%s: IM MAC 10/100baseT Ethernet **********************\n ", IMMAC_DEVICE_NAME);
	
	udelay(50);

	CKMAC_REG_BASEADDR[CSR0] =  0x100180; //set bus mode
	
	/* Clear the missed-packet counter. */
	i = CKMAC_REG_BASEADDR[CSR8] ;

	//printf("CSR0 status %x \n", CKMAC_REG_BASEADDR[CSR0]);
	
	immac_get_macaddr( bd->bi_enetaddr);
	printf("Phy macaddr = ");
	for (i = 0; i < 6; i++)
	{
		printf("%2.2x%c",  bd->bi_enetaddr[i], i == 5 ? ' ' : ':');
	}
	printf("\n");
	
	immac_des.rx_cur = 0;
	immac_rings_init(&immac_des);

	
	immac_ether_setupframe(IMMAC_CUR_SETUPF,bd);

	immac_stop_rxtx();
	udelay(5);

	CKMAC_REG_BASEADDR[CSR6] = CKMAC_REG_BASEADDR[CSR6]|TxOn; 
	CKMAC_REG_BASEADDR[CSR1] = 0;
	
	udelay(50);
	immac_stop_rxtx();
	
	CKMAC_REG_BASEADDR[CSR1] = 0;
	CKMAC_REG_BASEADDR[CSR2] = 0; 
	CKMAC_REG_BASEADDR[CSR5] = 0x0001c9e7;
	CKMAC_REG_BASEADDR[CSR7] = 0xF3FE0000;	//disable the all interrupt

	i = immac_phyr_read(0);
	printf("phys read 0 %x \n", i);

	i = immac_phyr_read(3);
	printf("phys read 3 %x \n", i);
	
	//CKMAC_REG_BASEADDR[CSR6] |= RxOn;
	/*
	CKMAC_REG_BASEADDR[CSR16] = 0x55667700;
	CKMAC_REG_BASEADDR[CSR17] = 0x1122;
	
	printf( "MACL : 0x%08lx\n", CKMAC_REG_BASEADDR[CSR16] );
	printf( "MACH : 0x%08lx\n", CKMAC_REG_BASEADDR[CSR17] );
	*/
	return 0;
}

void eth_halt( void )
{
	printf("eth halt\n");
	immac_stop_rxtx();
}

int eth_init( bd_t *bd )
{
  	printf (IMMAC_DRIVER_VERSION);
  	
  	//RT8201 Negotiation
	if (  immac_begin_auto_negotiation() )
	{
		printf("RT8201 Negotiation Error. \n");
		return -1;
	}
  	
  	immac_ether_init( bd );
  	
  	//while(1);
    return 0;
}

/* Get a data block via Ethernet */

extern int eth_rx(void)
{
	int i, s;
    u32 rxlen;
    u32 *addr, *p;
    unsigned short status;
    int err;
    
    #ifdef MAC_CACHE
   		volatile register tmp;
   	#endif
    
	err = 0;
	/*
    i = get_timer(0) + 5 * CFG_HZ;
	while((s = get_reg(CSR5) & 0x40) == 0) 
    {
		if (get_timer(0) >= i)
		{
	  		err = 1;
	  		break;
	  	}
    }*/
    
    i = 0;
    while((s = CKMAC_REG_BASEADDR[CSR5] & 0x40) == 0) 
    {
    	i++;
    	if( i > 0x3fffff )
    	{
    		err = 1;
    		break;
    	}
    }
    
    #ifdef MAC_CACHE
    	//flush D Cache
		__asm__ __volatile__
	  	(
	  	  	"mfcr %0, cr17\n"
	  	  	"bclri %0, 0\n"
	  	  	"bseti %0, 1\n"
	  	  	"bclri %0, 2\n"
	  	  	"bseti %0, 4\n"
			"mtcr %0, cr17\n"
			: "=r" (tmp)
			: "0"(tmp)
		);
	#endif
    
    CKMAC_REG_BASEADDR[CSR6] &= ~RxOn;
	immac_rings_init(&immac_des);
	s = immac_des.rx_cur;
    if( 0 == s  ) 
    	immac_des.rx_cur = 1;
    else
    	immac_des.rx_cur = 0;
	immac_des.rev_des[IMMAC_CUR_RX].DES2 = &(immac_des.rbuf[immac_des.rx_cur][0]);
	
	immac_des.rev_des[IMMAC_CUR_RX].DES0 |= 0x80000200; 
	CKMAC_REG_BASEADDR[CSR5] |= 0x40;
	
	rxlen = ( immac_des.rev_des[IMMAC_CUR_RX].DES0 ) & 0x3FFF0000;   /* len */
    rxlen = rxlen >> 16;
	CKMAC_REG_BASEADDR[CSR6] |= RxOn;
	
	if( err )
		return 0;
    
	if(rxlen > PKTSIZE_ALIGN + PKTALIGN)
	{
      printf("packet too big!\n");
      return 0;
	}  
    
	for( i=0; i<6; i++ )
	{
		if (0xff == immac_des.rbuf[s][i])
		{
			if( 5 == i )
				return 0;
		}
		else
			break;
	}

	//CKMAC_REG_BASEADDR[CSR6] |= RxOn;
	p = &(immac_des.rbuf[s][0]);
    //for(addr = (u32 *)NetRxPackets[1], i = rxlen >> 2; i > 0; i-- )
      //*addr++ = *p++;



    /* Pass the packet up to the protocol layers. */
    NetReceive( p, rxlen);

    return rxlen;
}

/* Send a data block via Ethernet. */
extern int eth_send(volatile void *packet, int length)
{
    //int tmo, s;
#if 0
	unsigned int i, j;
	unsigned int *p = packet;
	printf("p = %x\n",p);
	
	for(i=0;i<10;i++)
	{
		printf("%08lx ",*p++);
		if( (0 == (i % 4) ) && (i != 0) )
			printf("\n");
	}
	
	while(1);
	
#endif
    immac_stop_rxtx();
    //CKMAC_REG_BASEADDR[CSR6] &= ~TxOn;
    
	immac_rings_init(&immac_des);
	CKMAC_REG_BASEADDR[CSR5] = 0x0001c9e7 + 0x400;
	//CKMAC_REG_BASEADDR[CSR7] = 0xF3FE0000;	//disable the all interrupt	
	immac_des.mit_des[IMMAC_CUR_TX].DES0 = 0x80000000;	
	immac_des.rev_des[IMMAC_CUR_RX].DES0 = 0x80000000;	

    /* Write the contents of the packet */
    /* assume even number of bytes */
    
    immac_des.mit_des[IMMAC_CUR_TX].DES1 = length | 0x60000000;
    immac_des.mit_des[IMMAC_CUR_TX].DES2 = packet;
	
	CKMAC_REG_BASEADDR[CSR1] = 0x55;
	CKMAC_REG_BASEADDR[CSR2] = 0x55;
	
	immac_start_rxtx();
	//CKMAC_REG_BASEADDR[CSR6] = CKMAC_REG_BASEADDR[CSR6] | TxOn;
	
    /* don't wait for transfer to succeed because our to wait for receive*/
    /*
    tmo = get_timer(0) + 5 * CFG_HZ;
    while((s = get_reg(CSR5) & 0x01) == 0) 
    {
		if (get_timer(0) >= tmo)
	  		break;
    }*/
    return 0;
}


#endif /* COMMANDS & CFG_NET */

#endif /* CONFIG_DRIVER_IMMAC */
