/*
 * Cirrus Logic CS8900A Ethernet
 *
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * Copyright (C) 1999 Ben Williamson <benw@pobox.com>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is loaded into SRAM in bootstrap mode, where it waits
 * for commands on UART1 to read and write memory, jump to code etc.
 * A design goal for this program is to be entirely independent of the
 * target board.  Anything with a CL-PS7111 or EP7211 should be able to run
 * this code in bootstrap mode.  All the board specifics can be handled on
 * the host.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "armboot.h"
#include "command.h"
#include "net.h"
#include "s3c2500.h"
#include "mac2500.h"
#include "log.h"


#ifdef CONFIG_S3C2500

#if (CONFIG_COMMANDS & CFG_CMD_NET)

API_DEVICE			api_Mac_dev;

#if 0	//hans, FIXME, below buffers are allocated in C-data structure, but should be allocated in Non-cachable area
RECEIVE_FRAME_DESC	rx_buff_desc[RX_FD_NUM];
TRANSMIT_FRAME_DESC	tx_buff_desc[TX_FD_NUM];
char rx_framebuff_addr[RX_FD_NUM][END_BUFSIZ];
#else

//<--hans, for DMA area(Upper 1MBytes) access
typedef struct End_Buffer {
	char end_buffer[END_BUFSIZ];
} END_BUFFER;

RECEIVE_FRAME_DESC	*rx_buff_desc;
TRANSMIT_FRAME_DESC	*tx_buff_desc;
END_BUFFER *rx_framebuff_addr;
//-->hans end

#endif

void InitEtherApi(bd_t *bd)
{
	api_init(bd, &api_Mac_dev);
}

void eth_halt( void )
{
	api_MacIntDisable(&api_Mac_dev);
	api_MacHWStop(&api_Mac_dev);
	api_MacHWReset(&api_Mac_dev);
	api_MacBuffFree(&api_Mac_dev);
}

int eth_init( bd_t *bd )
{
	api_MacBuffInit(&api_Mac_dev);
	api_MacHWInit(&api_Mac_dev);
	api_MacHWStart(&api_Mac_dev);		
    return 0;
}

/* Get a data block via Ethernet */
extern int eth_rx(void)
{
	api_MacRcv(&api_Mac_dev);

    /* Pass the packet up to the protocol layers. */
    /*NetReceive(NetRxPackets[0], rxlen);*/
    return 0;
}

/* Send a data block via Ethernet. */
extern int eth_send(volatile void *packet, int length)
{
	api_MacSend(&api_Mac_dev, (ulong)packet, length);
    return 0;
}

/***************************/
/* Sub Function for driver */
/***************************/

/*******************************************************************************
*
* api_MacHWReset - .
*
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*/
int api_MacHWReset(API_DEVICE *api_Mac_dev)
{
	uBDMARXCON bdmaRxCon;
	uBDMATXCON bdmaTxCon;
	uMACCON macCon;

	ulong	pEtherAddr=api_Mac_dev->EtherAddr;

	*(ulong *)(&bdmaRxCon) = 0;
	*(ulong *)(&bdmaTxCon) = 0;
	*(ulong *)(&macCon) = 0;

	/*Init mac phy */
	MAC_PHY_INIT();
		
	bdmaRxCon.rxCon_reg.bRxRS= 1;
	REG_OUT_LONG (pEtherAddr, BDMARXCON, bdmaRxCon.rxCon_resetval);
	
	bdmaTxCon.txCon_reg.bTxRS= 1;
	REG_OUT_LONG (pEtherAddr, BDMATXCON, bdmaTxCon.txCon_resetval);
	
	macCon.macCon_reg.mReset = 1;
	macCon.macCon_reg.mII_OFF = ETH_PHY_SELECT;		/* MII ON or 7-Wire ON */	
	REG_OUT_LONG (pEtherAddr, MACCON, macCon.macCon_resetval);

	api_Mac_dev->status = HW_RESET_DONE;
		
	return OK;
}

/*******************************************************************************
*
*    api_MacBuffInit - Initialize TX and RX FD lists
*    Make a circular list of Rx and TX frame descriptors and buffers.
*    Two global variables gpReceiveFrameDescStart and gpTransmitFrameDescStart 
*    stores the pointer to the start of the list.  BDMA TX/RX PTR registers are 
*    also initialized with the start of the appropriate list.
*
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*/
int api_MacBuffInit(API_DEVICE *api_Mac_dev)
{
	STATUS status = OK;	
    int count;

    RECEIVE_FRAME_DESC* 	pReceiveFrameDesc;
    TRANSMIT_FRAME_DESC* 	pTransmitFrameDesc;
    
    ulong	pEtherAddr=api_Mac_dev->EtherAddr;

    char *pTransmitFramebuff;

//<--hans, for DMA area(Upper 1MBytes) access
	rx_buff_desc
		= (RECEIVE_FRAME_DESC	*)(DRAM_LIMIT - (sizeof(RECEIVE_FRAME_DESC)*RX_FD_NUM));
	tx_buff_desc
		=(TRANSMIT_FRAME_DESC	*)(((ulong)rx_buff_desc & ~(0x3)) - (sizeof(TRANSMIT_FRAME_DESC)*TX_FD_NUM));
	rx_framebuff_addr
		=(END_BUFFER *)(((ulong)tx_buff_desc & ~(0x3)) - (sizeof(END_BUFFER)*RX_FD_NUM));
//-->hans end

    /*Init Rx Fd */
	pReceiveFrameDesc = (RECEIVE_FRAME_DESC *)&(rx_buff_desc[0]);
	if(pReceiveFrameDesc == NULL)
		return ERROR;
	
	api_Mac_dev->gpReceiveFrameDescStart = pReceiveFrameDesc;
	REG_OUT_LONG(pEtherAddr, BDMARXDPTR, (ulong)pReceiveFrameDesc); 		
	REG_OUT_LONG(pEtherAddr, BRXBDCNT, (ulong)NULL); 		
	api_Mac_dev->RXDONEOFFSET =0;

	for(count = 0; count < RX_FD_NUM; count++)
		{
//<--hans, for DMA area(Upper 1MBytes) access
		pReceiveFrameDesc->rxFrameData = (ulong)&rx_framebuff_addr[count];
//-->hans end		
		*(ulong *) (&pReceiveFrameDesc->rxStatusLength) = 0;
		pReceiveFrameDesc->rxStatusLength.owner = 1;		/* owner: BDMA */

		pReceiveFrameDesc ++;
		}/* end of for loop*/

    /*Init Tx Fd */
	pTransmitFrameDesc = (TRANSMIT_FRAME_DESC *)&(tx_buff_desc[0]);
	if(pTransmitFrameDesc == NULL)
		return ERROR;

	api_Mac_dev->gpTransmitFrameDescStart = pTransmitFrameDesc;
	REG_OUT_LONG(pEtherAddr, BDMATXDPTR, (ulong)pTransmitFrameDesc); 		
	REG_OUT_LONG(pEtherAddr, BTXBDCNT, (ulong)NULL); 		
	api_Mac_dev->TXOFFSET =0;
	api_Mac_dev->TXDONEOFFSET =0;
	
	for(count = 0; count < TX_FD_NUM; count++)
		{
		pTransmitFrameDesc->txFrameData = (ulong)NULL;
		*(ulong *) (&pTransmitFrameDesc->txStatusLength) = 0;	/* owner: CPU */

		pTransmitFrameDesc ++;
		pTransmitFramebuff += END_BUFSIZ;
		}/* end of for loop */
		
	api_Mac_dev->buffInitialized = INIT_DONE;
	api_Mac_dev->status = BUFF_INIT_DONE;
		
	return status;
}

/*******************************************************************************
*
*	api_MacBuffFree - Free the allocated TX and RX FD lists and buffers
*	This function frees all the allocated TX and RX FDs and the associated
*	buffers
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*/
int api_MacBuffFree(API_DEVICE *api_Mac_dev)
{
	STATUS status = OK;
	
	api_Mac_dev->buffInitialized = NOT_INIT_YET;
	api_Mac_dev->status = BUFF_FREE_DONE;
		
	return status;
}

/*******************************************************************************
*
* initCAMReg - .
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*
*/
int initCAMReg
	(
	API_DEVICE *api_Mac_dev
	)
	{
	uCAMEN camEn;
	uCAMCON camCon;
	ulong count;
	uchar *pAddr;

	STATUS status = OK;

    ulong	pEtherAddr=api_Mac_dev->EtherAddr;

	/* Copy our MAC address to the first location in address array */
	pAddr = (uchar *)(&(api_Mac_dev->addrList[0]));
	*(ulong *)pAddr = htonl (*(ulong *)api_Mac_dev->enetAddr);

	#if	BIG_ENDIAN
	pAddr += 4;
	#else	/*BIG_ENDIAN*/
	pAddr += 6;
	#endif	/*BIG_ENDIAN*/

	*(ushort *)pAddr = htons (*(ushort *)(api_Mac_dev->enetAddr+4));

	/* Copy the address array into CAM registers */
	for (count = 0; count < 32; count++)
		REG_OUT_LONG(pEtherAddr, (CAM_BASE + (count*4)), api_Mac_dev->addrList[count]); 
		

	/* init CAMCON register */
	*(ulong *) (&camCon) = 0;
	
	camCon.camCon_reg.mBroad= 1;
	camCon.camCon_reg.mCompEn= 1;
	REG_OUT_LONG(pEtherAddr, CAMCON, camCon.camCon_resetval); 

	/* init CAMEN register.  Set bits for our MAC address and all
	 * multicast addresses
	 */
	api_Mac_dev->mcastAddrCount = 21;
	*(ulong *)(&camEn) = 0;
	for (count = 0; count <= api_Mac_dev->mcastAddrCount; count++)
		camEn.camen_reg.camEnable |= (0x1<<count);
	REG_OUT_LONG(pEtherAddr, CAMEN, camEn.camen_resetval); 
	
	return status;
	
	}

/*******************************************************************************
*
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO:
*/
int api_MacHWInit(API_DEVICE *api_Mac_dev)
{
	STATUS status = OK;

	uBDMARXCON bdmaRxCon;
	uBDMATXCON bdmaTxCon;
	uBDMARXLEN  bdmaRxLen;
	
	uMACCON macCon;
	uMACTXCON macTxCon;
	uMACRXCON macRxCon;

    ulong	pEtherAddr=api_Mac_dev->EtherAddr;
	
	/* init BDMATXCON register */
	*(ulong *)(&bdmaTxCon) = 0;
	bdmaTxCon.txCon_reg.bTxNBD =TX_FD_NUM_PWR2;
	bdmaTxCon.txCon_reg.bTxMSL = 100;			/* BDMA transmit to MAC tx start level :  4/8 */
	REG_OUT_LONG(pEtherAddr, BDMATXCON, bdmaTxCon.txCon_resetval); 

	/* init BDMARXCON register */
	*(ulong *) (&bdmaRxCon) = 0;
	bdmaRxCon.rxCon_reg.bRxNBD =RX_FD_NUM_PWR2;
	bdmaRxCon.rxCon_reg.bRxWA = 0;				/* 0 bytes alignment */
	REG_OUT_LONG(pEtherAddr, BDMARXCON, bdmaRxCon.rxCon_resetval); 

	/* init BDMARXLEN */
	*(ulong *)(&bdmaRxLen) = 0;
	bdmaRxLen.rxLen_reg.bRxMFS = END_BUFSIZ;
	bdmaRxLen.rxLen_reg.bRxBS = END_BUFSIZ - 16; /* Rx Buffer Size */
	REG_OUT_LONG(pEtherAddr, BDMARXLEN, bdmaRxLen.rxLen_resetval); 
	
	/* init MACCON register */	
	*(ulong *)(&macCon) = 0;
	macCon.macCon_reg.mII_OFF = ETH_PHY_SELECT; 	
	if(api_Mac_dev->duplexMode == FULL_DUPLEX)
		macCon.macCon_reg.full_duplex = 1; 	
	REG_OUT_LONG(pEtherAddr, MACCON, macCon.macCon_resetval); 

	
	/* init MACTXCON register */
	*(ulong *)(&macTxCon) = 0;
	REG_OUT_LONG(pEtherAddr, MACTXCON, macTxCon.macTxCon_resetval); 

	/* init MACRXCON register */
	*(ulong *)(&macRxCon) = 0;
	macRxCon.macRxCon_reg.mStripCRC = 1;
	REG_OUT_LONG(pEtherAddr, MACRXCON, macRxCon.macRxCon_resetval); 

	/* Init CAM regiser and CAM Enable register */
	initCAMReg(api_Mac_dev);
		
	api_Mac_dev->status = HW_INIT_DONE;
		
	return status;
}


/*******************************************************************************
*
* api_MacHWStart - .
*
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*/
int api_MacHWStart(API_DEVICE *api_Mac_dev)
{
	STATUS status = OK;
		
	uBDMARXCON bdmaRxCon;
	uMACRXCON macRxCon;
    ulong	pEtherAddr=api_Mac_dev->EtherAddr;

	/* init BDMARXCON register */
	REG_IN_LONG(pEtherAddr, BDMARXCON, *(ulong *) (&bdmaRxCon)); 
	bdmaRxCon.rxCon_reg.bRxEn = 1;
	REG_OUT_LONG(pEtherAddr, BDMARXCON, bdmaRxCon.rxCon_resetval); 
	
	/* init MACRXCON register */
	REG_IN_LONG(pEtherAddr, MACRXCON, *(ulong *) (&macRxCon)); 
	macRxCon.macRxCon_reg.mRxEn = 1;
	REG_OUT_LONG(pEtherAddr, MACRXCON, macRxCon.macRxCon_resetval); 

	api_Mac_dev->status = HW_START_STATUS;
		
	return status;
}

/*******************************************************************************
*
* api_MacHWStop - .
*
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*/
int api_MacHWStop (API_DEVICE *api_Mac_dev)
{
	STATUS status = OK;
		
	uBDMARXCON bdmaRxCon;
	uBDMATXCON bdmaTxCon;
	uMACCON macCon;
    ulong	pEtherAddr=api_Mac_dev->EtherAddr;

	REG_IN_LONG(pEtherAddr, BDMARXCON, *(ulong *)(&bdmaRxCon)); 
	bdmaRxCon.rxCon_reg.bRxEn = 0;
	REG_OUT_LONG(pEtherAddr, BDMARXCON, *(ulong *)(&bdmaRxCon)); 

	REG_IN_LONG(pEtherAddr, BDMATXCON, *(ulong *)(&bdmaTxCon)); 
	bdmaTxCon.txCon_reg.bTxEn= 0;
	REG_OUT_LONG(pEtherAddr, BDMATXCON, *(ulong *)(&bdmaTxCon)); 
	
	REG_IN_LONG(pEtherAddr, MACCON, *(ulong *)(&macCon)); 
	macCon.macCon_reg.mHaltImm = 1;
	REG_OUT_LONG(pEtherAddr, MACCON, *(ulong *)(&macCon)); 

	api_Mac_dev->status = STOP_STATUS;
		
	return status;
}

/*******************************************************************************
*
* api_MacRcv - .
*
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*/

void api_MacRcv(API_DEVICE *api_Mac_dev)
{
	RECEIVE_FRAME_DESC* pFd;
	ulong			RXFDCNT;
	uBMRXSTAT		bmRxStat;
	uBDMARXCON		bmRxCon;
	ushort*			pFrameData;
	ulong 			frameLength;

    ulong	pEtherAddr=api_Mac_dev->EtherAddr;

	/* Read & Clear BDMA Status Reg and Clear interrupt pending */
	REG_IN_LONG(pEtherAddr, BMRXSTAT, *(ulong *)(&bmRxStat));
	REG_OUT_LONG(pEtherAddr, BMRXSTAT, 0);			// Only for s3c2500

	pFd = api_Mac_dev->gpReceiveFrameDescStart + api_Mac_dev->RXDONEOFFSET;
	if((*(ulong *)(&(pFd->rxStatusLength)) & 0xffff)==0)
		return;

	REG_IN_LONG(pEtherAddr, BRXBDCNT, RXFDCNT);

    /* Process all packets received. */
	while (api_Mac_dev->RXDONEOFFSET != RXFDCNT)
 	{
 		pFd = api_Mac_dev->gpReceiveFrameDescStart + api_Mac_dev->RXDONEOFFSET;

	    /* Check BDMA Receive Ownership     */
		if( ! pFd->rxStatusLength.owner )
	    {
			if(*(ulong *)(&(pFd->rxStatusLength)) & MACRXFRAMEERRMASK ){
				/* Rx Error packet */
				api_Mac_dev->statistics.rxBad++;
				
				if (api_Mac_dev->gpReceiveFrameDescStart->rxStatusLength.mso)
					api_Mac_dev->statistics.rxOvMaxSize++;
				/*
				 * This state is not in descriptor.
				 * if (api_Mac_dev->gpReceiveFrameDescStart->rxStatusLength.ctlRcv)
				 *	api_Mac_dev->statistics.rxCtlRecd++;
				 */
				if (api_Mac_dev->gpReceiveFrameDescStart->rxStatusLength.mRx10Stat)
					api_Mac_dev->statistics.rx10Stat++;
				if (api_Mac_dev->gpReceiveFrameDescStart->rxStatusLength.alignErr)
					api_Mac_dev->statistics.rxAlignErr++;
				if (api_Mac_dev->gpReceiveFrameDescStart->rxStatusLength.crcErr)
					api_Mac_dev->statistics.rxCRCErr++;
				if (api_Mac_dev->gpReceiveFrameDescStart->rxStatusLength.overFlow)
					api_Mac_dev->statistics.rxOverflowErr++;
				if (api_Mac_dev->gpReceiveFrameDescStart->rxStatusLength.mufs)
					api_Mac_dev->statistics.rxLongErr++;
				if (api_Mac_dev->gpReceiveFrameDescStart->rxStatusLength.rxParErr)
					api_Mac_dev->statistics.rxParErr++;
				if (api_Mac_dev->gpReceiveFrameDescStart->rxStatusLength.halted)
					api_Mac_dev->statistics.rxHalted++;
			}else if( pFd->rxStatusLength.bRxDone ){
				/* Rx O.K. */
				api_Mac_dev->statistics.rxGood++;
				
				frameLength = (*(ulong *)(&(pFd->rxStatusLength)) & 0xffff);
				pFrameData = (ushort *)pFd->rxFrameData;	

			    /* Pass the packet up to the protocol layers. */
			    NetRxPackets[api_Mac_dev->statistics.rxGood%RX_FD_NUM] = (uchar *)(pFrameData);
	    		NetReceive(NetRxPackets[api_Mac_dev->statistics.rxGood%RX_FD_NUM], frameLength);

				/*SYS_MAC_RECV(api_Mac_dev, (char *)pFrameData, frameLength);			*/
    		}
    	}
		else
        	break;

	    /* Clear Reserved & Status/Length Field     */
	    *(ulong *)(&(pFd->rxStatusLength)) = 0;

	    /* Clear Framedata pointer already used   */
	    pFd->rxStatusLength.owner = 1;

		api_Mac_dev->RXDONEOFFSET ++;
		if(api_Mac_dev->RXDONEOFFSET  == RX_FD_NUM)
			api_Mac_dev->RXDONEOFFSET =0;
	}

	/*If not ownerstat than re_enable rx  */
	if(bmRxStat.rxStat_reg.bRxNO)
    {
    	REG_IN_LONG(pEtherAddr, BDMARXCON, *(ulong *)(&bmRxCon));
    	bmRxCon.rxCon_reg.bRxEn =1;
    	REG_OUT_LONG(pEtherAddr, BDMARXCON, *(ulong *)(&bmRxCon));
    }
}

/*******************************************************************************
*
* api_MacSend - .
*
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*/
int api_MacSend(API_DEVICE *api_Mac_dev, ulong pData, int len)
{	
	TRANSMIT_FRAME_DESC *pTxFd;
	uBDMATXCON bdmaTxCon;
	uMACTXCON macTxCon;

    ulong	pEtherAddr=api_Mac_dev->EtherAddr;

	pTxFd = api_Mac_dev->gpTransmitFrameDescStart + api_Mac_dev->TXOFFSET;
	if (pTxFd->txStatusLength.owner)	/* Ownership with BDMA? */
	{
		printf("Error: Ownership with BDMA?\n");
		return ERROR;
	}	
	
	*(ulong *)(&bdmaTxCon) = 0;
	*(ulong *)(&macTxCon) = 0;
    
	/* <<Attention>> FIXUP Please for RTOS and Application */
	/* FIXUP please not Use memory copy */
	/*rtn(pData, (void *)pTxFd->txFrameData.frameDataPtr, len); */
	
	pTxFd->txFrameData = pData;

	pTxFd->txStatusLength.TxWidget =0;
	pTxFd->txStatusLength.TxLength =len;
	pTxFd->txStatusLength.owner =1;

    /* place a transmit request */

    /* oldLevel = SYS_INT_STOP(MAX_ILEV); */  /* now sndsEndInt won't get confused */

    /* initiate device transmit */
	REG_IN_LONG (pEtherAddr, BDMATXCON, bdmaTxCon.txCon_resetval);
	bdmaTxCon.txCon_reg.bTxEn= 1;
	REG_OUT_LONG (pEtherAddr, BDMATXCON, bdmaTxCon.txCon_resetval);
	
	REG_IN_LONG (pEtherAddr, MACTXCON, macTxCon.macTxCon_resetval);
	macTxCon.macTxCon_reg.mTxEn = 1;
	REG_OUT_LONG (pEtherAddr, MACTXCON, macTxCon.macTxCon_resetval);
    
    /* Advance our management index */
    if(api_Mac_dev->TXOFFSET >= (TX_FD_NUM -1))
    	api_Mac_dev->TXOFFSET =0;
    else
		api_Mac_dev->TXOFFSET ++;
    
	/* SYS_INT_START (oldLevel); */   /* now sndsEndInt won't get confused */
	
	return OK;
}


/*******************************************************************************
*
* api_MacIntEnable - .
*
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*/
int api_MacIntEnable(API_DEVICE *api_Mac_dev)
{
	STATUS status = OK;

	/* Not use system interrupt handler */
	/*
	SYS_INT_CONNECT(api_Mac_dev->ivecRx, api_Mac_dev->ihandleFuncRx);
	SYS_INT_CONNECT(api_Mac_dev->ivecTx, api_Mac_dev->ihandleFuncTx);
	*/
	intEnable(api_Mac_dev->ivecRx);
	intEnable(api_Mac_dev->ivecTx);

	api_Mac_dev->status = INT_ENABLE_STATUS;
		
	return status;
}

/*******************************************************************************
*
* api_MacIntDisable - .
*
*
* RETURNS: OK or ERROR return status why..
*
* SEE ALSO: 
*/
int api_MacIntDisable(API_DEVICE *api_Mac_dev)
{
	STATUS status = OK;

	
	/* Not use system interrupt handler */
	/*
	SYS_INT_CONNECT(api_Mac_dev->ivecRx, dummyIsr);
	SYS_INT_CONNECT(api_Mac_dev->ivecTx, dummyIsr);
	*/
	
	intDisable(api_Mac_dev->ivecRx);
	intDisable(api_Mac_dev->ivecTx);
	
	api_Mac_dev->status = INT_DISABLE_STATUS;
		
	return status;
}

/******************************************************************************
* sndsEndPhyRead - Read PHY device
* This function is used to read a byte from the PHY device
*/
ulong miiRead(API_DEVICE *api_Mac_dev, 
			  ulong phyAddr,			/* Address of the PHY chip (usually 0 for single PHY) */
			  ulong phyRegAddr, 		/* Address of PHY register to be read */
			  ulong *phyData)			/* Data to be Read */
{
	uSTACON 	staCon;
	ulong		pEtherAddr=api_Mac_dev->EtherAddr;

	*(ulong *) (&staCon) = 0;
	staCon.staCon_reg.mPHYRegAddr= phyRegAddr;
	staCon.staCon_reg.mPHYaddr= phyAddr;
	staCon.staCon_reg.mPHYbusy= 1;
	REG_OUT_LONG (pEtherAddr, STACON, staCon.staCon_resetval);

	do 
	{
		REG_IN_LONG(pEtherAddr, STACON, staCon.staCon_resetval);	
	} while (staCon.staCon_resetval & 0x800);	/***** Busy bit ***/
	
	REG_IN_LONG(pEtherAddr, STADATA, *phyData);

	return 0;
}

/******************************************************************************
* sndsEndPhyWrite	- Wrire into PHY device
* This function is used to write a byte to the PHY device
*/
void miiWrite(API_DEVICE *api_Mac_dev,
			  ulong phyAddr,		/* Address of the PHY chip (usually 0 for single PHY) */
			  ulong phyRegAddr, 	/* Address of PHY register to be written */
			  ulong phyData)		/* Data to be written */
{
	uSTACON staCon;
	ulong count = 1000;
	ulong	pEtherAddr=api_Mac_dev->EtherAddr;


	REG_OUT_LONG (pEtherAddr, STADATA, phyData);
	*(ulong *) (&staCon) = 0;
	staCon.staCon_reg.mPHYRegAddr = phyRegAddr;
	staCon.staCon_reg.mPHYaddr = phyAddr;
	staCon.staCon_reg.mPHYbusy = 1;
	staCon.staCon_reg.mPHYwrite= 1;
	REG_OUT_LONG (pEtherAddr, STACON, staCon.staCon_resetval);

	do
	{
		REG_IN_LONG (pEtherAddr, STACON, staCon.staCon_resetval);	
	} while (staCon.staCon_resetval & 0x800)	;	/***** Busy bit ***/

	while (count--)
		;	/* Dummy delay after PHY write */
}

/*****************************************************************************
*
*   FUNCTION
*
*       MII_AutoNeg
*
*   DESCRIPTION
*
*       Using the mii_ReadMII and mii_WriteMII routines supplied by the driver
*       to access the MII registers, this routine directs the auto-negotiation
*       process of an MII PHY.
*
*   INPUTS
*
*       *api_Mac_dev       
*       phyAddr
*       retries        
*       *isFullDuplexP 
*       *is100Mbps     
*       miiRead
*       miiWrite
*
*   OUTPUTS
*
*       SUCCESS           
*                           
*                           
*       TIMEOUT           
*                            
*       NOT_PRESENT       
*       INVALID_OPERATION 
*                           
*       INVALID_FUNCTION  
*       INVALID_POINTER
*    
*****************************************************************************/
/*
STATUS MII_AutoNeg(DV_DEVICE_ENTRY* deviceP, int phyAddr,
                   unsigned long retries, int* isFullDuplexP, int* is100MbpsP,
                   mii_ReadMII miiRead, mii_WriteMII miiWrite)

*/

#if 0
STATUS MII_AutoNeg(API_DEVICE *api_Mac_dev, unsigned long retries, int *isFullDuplexP, int *is100MbpsP)
{
	unsigned long 		miiControl;
	unsigned long 		miiStatus = 0;
	unsigned long 		miiAutoNeg = MII_ADVR_802_3;
	unsigned long 		tries = ((unsigned long)(0xFFFFFFFF));
	STATUS 				status = OK;
	ulong 				phyAddr = 0;

	if (api_Mac_dev->channel == 0)
	{
		phyAddr = PHY_ADDR_A;
	}
	else
	{
		phyAddr = PHY_ADDR_B;
	}

	/* Reset PHY via software */
	miiRead(api_Mac_dev, phyAddr, MII_CONTROL, &miiControl);
	miiControl |= MII_CTRL_RESET;
	miiWrite(api_Mac_dev, phyAddr, MII_CONTROL, miiControl);

	/* Wait for PHY to complete reset */
	do
	{
		miiRead(api_Mac_dev, phyAddr, MII_CONTROL, &miiControl);
	} while((miiControl & MII_CTRL_RESET) != 0);
	
	/* Power up PHY and set options */
	miiControl &= ~(MII_CTRL_LOOPBACK | MII_CTRL_POWER_DOWN |
	MII_CTRL_ISOLATE | MII_CTRL_COLL_TEST);

	miiControl |= (MII_CTRL_AUTO_NEG);
	miiWrite(api_Mac_dev, phyAddr, MII_CONTROL, miiControl);
	
	/* Read PHY status and capabilities */
	miiRead(api_Mac_dev, phyAddr, MII_STATUS, &miiStatus);
	
	/* This driver relies on PHY being capable of auto-negotiation */
	if((miiStatus & MII_STAT_AUTO_NEG) == 0)
	{
		status = ERROR;
	}

	/* Determine PHY capabilities to be advertised during auto-negotiation */
	if((miiStatus & MII_STAT_TX) && (*is100MbpsP == TRUE))
	{
		miiAutoNeg |= MII_ADVR_TX;

		if((miiStatus & MII_STAT_TX_FULL_DUPLEX) && (*isFullDuplexP == TRUE))
			miiAutoNeg |= MII_ADVR_TX_FULL_DUPLEX;
	}

	if(miiStatus & MII_STAT_10)
	{
		miiAutoNeg |= MII_ADVR_10;

		if((miiStatus & MII_STAT_10_FULL_DUPLEX) && (*isFullDuplexP == TRUE))
			miiAutoNeg |= MII_ADVR_10_FULL_DUPLEX;
	}
	
	/* Indicate advertised capabilities */
	miiWrite(api_Mac_dev, phyAddr, MII_ADVERTISEMENT, miiAutoNeg);
	
	/* Restart auto-negotiation process */
	miiRead(api_Mac_dev, phyAddr, MII_CONTROL, &miiControl);
	miiControl |= MII_CTRL_RESTART; 
	miiWrite(api_Mac_dev, phyAddr, MII_CONTROL, miiControl);
	
	/* Wait for PHY to auto-negotiate */
	if(retries != 0)                     /* If not infinite retries... */
		tries = retries;                 /* Initialize attempts remaining */

	do
	{
		miiRead(api_Mac_dev, phyAddr, MII_STATUS, &miiStatus);

		if((miiStatus & MII_STAT_REMOTE_FAULT) != 0)
		{
			status = ERROR;
		}

		if(retries != 0)                  /* If not infinite retries... */
			tries -= 1;                   /* Decrement attempts remaining */
	} while((tries != 0) && ((miiStatus & MII_STAT_AUTO_NEG_DONE) == 0));
	
	/* Obtain link partner response */
	miiRead(api_Mac_dev, phyAddr, MII_LINK_PARTNER, &miiAutoNeg);

//	if((miiAutoNeg & MII_LINK_REMOTE_FAULT) != 0)
	if(miiAutoNeg & MII_LINK_REMOTE_FAULT)
	{
		status = ERROR;    
	}
	
	/* Configure PHY for link partner */
	miiRead(api_Mac_dev, phyAddr, MII_CONTROL, &miiControl);
	
	/* Select 100 Mbps if this PHY is 100Base-TX capable, the driver asked for
	100Mbps and not only 10Mbps, and the link partner responded with
	100Base-TX or did not respond at all. The "no response" behavior is
	done to allow a wire to be plugged in later and have the PHY negotiate
	the best speed. */
	if((miiStatus & MII_STAT_TX) && (*is100MbpsP == TRUE) &&
	   ((miiAutoNeg & MII_LINK_TX) || (!(miiAutoNeg & MII_LINK_ACK))))
	{
		miiControl |= MII_CTRL_100MBPS;
		*is100MbpsP = TRUE;
	}
	else            /* Use only 10Mbps, because of options or link partner */
	{
		miiControl &= (~(MII_CTRL_100MBPS));
		*is100MbpsP = FALSE;
	}

	if((miiAutoNeg & MII_LINK_ACK) && (*isFullDuplexP == TRUE) &&
	   (((*is100MbpsP == TRUE) && (miiStatus & MII_STAT_TX_FULL_DUPLEX) &&
	   (miiAutoNeg & MII_LINK_TX_FULL_DUPLEX)) ||
	   ((*is100MbpsP == FALSE) && (miiStatus & MII_STAT_10_FULL_DUPLEX) &&
	   (miiAutoNeg & MII_LINK_10_FULL_DUPLEX))))
	{
		/* Select full duplex if link partner responded and both the link partner
		and this PHY are full duplex capable */
		miiControl |= MII_CTRL_FULL_DUPLEX;
		*isFullDuplexP = TRUE;
	}
	else        /* Use only half duplex, because of options or link partner */
	{
		miiControl &= (~(MII_CTRL_FULL_DUPLEX));
		*isFullDuplexP = FALSE;
	}

	miiWrite(api_Mac_dev, phyAddr, MII_CONTROL, miiControl);

	if(tries == 0)
	{
		status = ERROR;
	}
  
  	return status;
}
#endif

int MAC_PHY_INIT(void)
{
//	int 		isFullDuplexP=0; 
//	int			is100MbpsP=0;
	STATUS 		status = OK;	

#if 0
	if((status = MII_AutoNeg(&api_Mac_dev, 0, &isFullDuplexP, &is100MbpsP)) != OK)
	{
		printf("ERROR: MII AutoNeg \n");
		return status;
	}
	printf("MII Status Full Duplex=%d, 100Mbps=%d\n", isFullDuplexP, is100MbpsP);
#endif

	return status;
}

void api_init(bd_t *bd, API_DEVICE *api_Mac_dev)
{
	api_Mac_dev->EtherAddr = ETHER_REG_ADDR_A;
	api_Mac_dev->ivecTx = INT_LVL_ETHTxA;
	api_Mac_dev->ivecRx = INT_LVL_ETHRxA;

	/* Set MAC PHY Register */
	api_Mac_dev->autoNeg = 0;
	api_Mac_dev->netSpeed = 0;
	api_Mac_dev->duplexMode = 0;

	/* Set Ethernet address */
//	SYS_ENET_ADDR_GET(api_Mac_dev, EnetAddrA);
	SYS_ENET_ADDR_GET(api_Mac_dev, bd->bi_enetaddr);
	api_Mac_dev->channel = 0;
}
#endif /* COMMANDS & CFG_NET */

#endif /* CONFIG_S3C2500 */
