#ifndef __CRAMFS_H__
#define __CRAMFS_H__

#include <byteswap.h>
#include <endian.h>

#define CRAMFS_BLOCKSIZE 4096		// *** 4096 == 0x1000 ***
#define CRAMFS_BLOCKMASK 4095		// *** 4095 == 0x0FFF ***
#define CRAMFS_BLOCKSHIFT  12		// *** 4096 == 2 ^ 12 ***

#define CRAMFS_MAGIC     0x28cd3d45
#define CRAMFS_SIGNATURE "Compressed ROMFS"

#if 0
typedef unsigned char u8;
typedef unsigned short u16;
typedef unsigned int u32;
#endif
/*
 * Width of various bitfields in struct cramfs_inode.
 */ 
#define CRAMFS_MODE_WIDTH 16
#define CRAMFS_UID_WIDTH 16
#define CRAMFS_SIZE_WIDTH 24
#define CRAMFS_GID_WIDTH 8
#define CRAMFS_NAMELEN_WIDTH 6
#define CRAMFS_OFFSET_WIDTH 26

#define C_DIR_MASK  0x4000
#define C_RD_MASK 0x120
#define C_WR_MASK 0x092
#define C_EX_MASK 0x049

#define ISDIR(X) (X & C_DIR_MASK ? 1 : 0)
#define ISRD(X) (X & C_RD_MASK ? 1 : 0)
#define ISWR(X) (X & C_WR_MASK ? 1 : 0)
#define ISEX(X) (X & C_EX_MASK ? 1 : 0)

struct cramfs_inode {
  unsigned long mode   : CRAMFS_MODE_WIDTH,
                uid    : CRAMFS_UID_WIDTH;
  unsigned long size   : CRAMFS_SIZE_WIDTH,
                gid    : CRAMFS_GID_WIDTH;
  unsigned long namelen: CRAMFS_NAMELEN_WIDTH,
                offset : CRAMFS_OFFSET_WIDTH;
};

struct cramfs_info {
  unsigned long crc;
  unsigned long version;
  unsigned long blocks;
  unsigned long files;
};

/*
 * Superblock information at the beginning of the FS.
 */
struct cramfs_super {
  unsigned long magic;
  unsigned long size;
  unsigned long flags;
  unsigned long reserved;
  unsigned char signature[16];
  struct cramfs_info info;
  unsigned char name[16];
  struct cramfs_inode root;
};

#define cramfs_get_block(ptr,inode,blknum,destination) \
	cramfs_inflate(ptr,inode,destination,blknum*CRAMFS_BLOCKSIZE,CRAMFS_BLOCKSIZE)

struct cramfs_super * cramfs_get_super(void * ptr);
struct cramfs_inode * cramfs_get_root(void * ptr);
struct cramfs_inode * cramfs_readdir_next(void * ptr, struct cramfs_inode * dir, struct cramfs_inode * current);
char * cramfs_get_name(void * ptr, struct cramfs_inode * inode);
int cramfs_get_size(void * ptr, struct cramfs_inode * inode);
struct cramfs_inode * cramfs_lookup(void * ptr, struct cramfs_inode * cwd, char * name);
int cramfs_inflate (void * ptr, struct cramfs_inode * inode, void * destination,
        int offset, int size);

/*
 * Since cramfs is little-endian, provide macros to swab the bitfields.
 */

#define swap16_(x) \
	((((x) >> 8) & 0xff) | (((x) & 0xff) << 8))

#define swap32_(x) \
	((((x) & 0xff000000) >> 24) | (((x) & 0x00ff0000) >>  8) | \
	(((x) & 0x0000ff00) <<  8) | (((x) & 0x000000ff) << 24))

#ifndef __BYTE_ORDER
#if defined(__LITTLE_ENDIAN) && !defined(__BIG_ENDIAN)
#define __BYTE_ORDER __LITTLE_ENDIAN
#elif defined(__BIG_ENDIAN) && !defined(__LITTLE_ENDIAN)
#define __BYTE_ORDER __BIG_ENDIAN
#else
#error "unable to define __BYTE_ORDER"
#endif
#endif /* not __BYTE_ORDER */

#if __BYTE_ORDER == __LITTLE_ENDIAN
#warning  "__BYTE_ORDER == __LITTLE_ENDIAN"
#define CRAMFS_16(x)   (x)
#define CRAMFS_24(x)   (x)
#define CRAMFS_32(x)   (x)
#define CRAMFS_GET_NAMELEN(x)  ((x)->namelen)
#define CRAMFS_GET_OFFSET(x)   ((x)->offset)
#define CRAMFS_SET_OFFSET(x,y) ((x)->offset = (y))
#define CRAMFS_SET_NAMELEN(x,y)        ((x)->namelen = (y))

#elif __BYTE_ORDER == __BIG_ENDIAN
#warning "__BYTE_ORDER == __BIG_ENDIAN"

#ifdef	CONFIG_M68328
/* Use M68K DragonBall specific swap instruction to speed up */
#define	CRAMFS_16(x)	bswap_16(x)
#define	CRAMFS_24(x)	((bswap_32(x)) >> 8)
#define	CRAMFS_32(x)	bswap_32(x)
#else
/* For other case, including M68K ColdFire.... */
#define CRAMFS_16(x)   swap16_(x)
#define CRAMFS_24(x)   ((swap32_(x)) >> 8)
#define CRAMFS_32(x)   swap32_(x)
#endif

#define CRAMFS_GET_NAMELEN(x)  (((u8*)(x))[8] & 0x3f)
#define CRAMFS_GET_OFFSET(x)   ((CRAMFS_24(((u32*)(x))[2] & 0xffffff) << 2) |\
                                ((((u32*)(x))[2] & 0xc0000000) >> 30))
#define CRAMFS_SET_NAMELEN(x,y)        (((u8*)(x))[8] = (((0x3f & (y))) | \
                                                 (0xc0 & ((u8*)(x))[8])))
#define CRAMFS_SET_OFFSET(x,y) (((u32*)(x))[2] = (((y) & 3) << 30) | \
                                CRAMFS_24((((y) & 0x03ffffff) >> 2)) | \
                                (((u32)(((u8*)(x))[8] & 0x3f)) << 24))
#else

#error "__BYTE_ORDER must be __LITTLE_ENDIAN or __BIG_ENDIAN"
#endif


#endif /* __CRAMFS_H__ */
