/* s5n8947.h - SAMSUNG DSL MCU S5N8947 header file */

/*
This file contains I/O address and related constants for the ARM PID board.
*/
#ifndef INCs5n8947h
#define	 INCs5n8947h

#include "config.h"

#ifdef INCLUDE_FLASH
#define FLASH_ADRS		0x01000000 	/* Base address of Flash part */
#define FLASH_WIDTH	2	  		/* 1 byte-wide part */
/*
 * If the assummptions below are not met, the definitions will
 * probably need to be changed appropriately (see FlashMem.c) and then
 * calibrated using timexN.
 */

#define SYS_FLASH_BOARD_DELAY
#define SYS_FLASH_DELAY_SHIFT 0
#define SYS_FLASH_DELAY_ADJ 0
#define SYS_FLASH_DELAY_INCR 7

#endif /* INCLUDE_FLASH */

#define CPU_SPEED	72000000	/* CPU clocked at 72 MHz. The timer */

/*************************************************************************
* changes made from here
*/

/*************************************************************************
* S5N8947 SPECIAL REGISTERS 
*
*/
#define ASIC_BASE  			0x3ff0000

/* Interrupt Control */
#define INT_CNTRL_BASE 		(ASIC_BASE+0x4000) /*Define base of all interrupt */

/*SYSTEM MANAGER REGISTERS */

#define SYSCFG			(ASIC_BASE+0x0000)
#define CLKCON  			(ASIC_BASE+0x3000)
#define EXTACON0  		(ASIC_BASE+0x3008)
#define EXTACON1  		(ASIC_BASE+0x300c)
#define EXTDBWTH  		(ASIC_BASE+0x3010)
#define ROMCON0  			(ASIC_BASE+0x3014)
#define ROMCON1  			(ASIC_BASE+0x3018)
#define ROMCON2  			(ASIC_BASE+0x301c)

#define PCMOFFSET		(ASIC_BASE+0x3020)
#define DRAMCON0  		(ASIC_BASE+0x3024)
#define DRAMCON1  		(ASIC_BASE+0x3028)
#define DRAMCON2  		(ASIC_BASE+0x302c)
#define DRAMCON3  		(ASIC_BASE+0x3030)
#define REFEXTCON 		(ASIC_BASE+0x3034)

/* controller registers */
#define INTMODE 			(ASIC_BASE+0x4000)
#define INTPEND 			(ASIC_BASE+0x4004)
#define INTMASK 			(ASIC_BASE+0x4008)
#define INTOFFSET 		(ASIC_BASE+0x4024)
#define INTPENDTST 		(ASIC_BASE+0x402c)
#define INT_DISABLE            0x007fffff

#define INTPRI0 			(ASIC_BASE+0x400C)
#define INTPRI1 			(ASIC_BASE+0x4010)
#define INTPRI2 			(ASIC_BASE+0x4014)
#define INTPRI3 			(ASIC_BASE+0x4018)
#define INTPRI4 			(ASIC_BASE+0x401C)
#define INTPRI5 			(ASIC_BASE+0x4020)

#define INTOSET_FIQ 		(ASIC_BASE+0x4030)
#define INTOSET_IRQ 		(ASIC_BASE+0x4034)

 /* I/O Port Interface  */

#define IOPMOD 			(ASIC_BASE+0x5000)
#define IOPCON0 			(ASIC_BASE+0x5004)
#define IOPCON1 			(ASIC_BASE+0x5008)
#define IOPDATA 			(ASIC_BASE+0x500C)

/* IIC Registers */
#define  IICCON   		(ASIC_BASE+0xf000)
#define  IICBUF    		(ASIC_BASE+0xf004)
#define  IICPS   		(ASIC_BASE+0xf008)
#define  IICCNT  		(ASIC_BASE+0xf00c)

/* Timer Registers */
#define TIMER_TMOD		(ASIC_BASE + 0x6000) /* TMOD    (R/W) */
#define TIMER_TDATA_0	(ASIC_BASE + 0x6004) /* TDATA0  (R/W) */
#define TIMER_TDATA_1	(ASIC_BASE + 0x6008) /* TDATA1  (R/W) */
#define TIMER_TDATA_2	(ASIC_BASE + 0x600C) /* TDATA1  (R/W) */
#define TIMER_TCNT_0	(ASIC_BASE + 0x6010) /* TCNT0   (R/W) */
#define TIMER_TCNT_1	(ASIC_BASE + 0x6014) /* TCNT1   (R/W) */
#define TIMER_TCNT_2	(ASIC_BASE + 0x6018) /* TCNT1   (R/W) */

/* UART Register */
#define ULCON		(ASIC_BASE + 0xD000)		/*UART Line Control Registers*/
#define UCON		(ASIC_BASE + 0xD004)		/*UART Control Register */
#define USTAT		(ASIC_BASE + 0xD008)	/*UART Status Register */
#define UTXBUF		(ASIC_BASE + 0xD00c)	/*UART Transmit Buffer Register*/
#define URXBUF		(ASIC_BASE + 0xD010)	/*UART Receive Buffer Register*/
#define UBRDIV		(ASIC_BASE + 0xD014)	/*UART Baud Rate Divisor Register*/
#define BRDCNT		(ASIC_BASE + 0xD018)	/*UART Baud Rate Count Register */
#define BRDCLK		(ASIC_BASE + 0xD01c)	/*UART Baud Rate Clock Monitor*/

/* Bit definitions within ULCON0/1 Line Control Register*/
#define PARITY_NONE 		0x00			/* Set No Parity*/
#define PARITY_ODD 		0x20			/* Set Odd Parity*/
#define PARITY_EVEN		0x28			/* Set Even Parity*/
#define ONE_STOP			0x00			/* One Stop Bit*/
#define WORD_LEN			0x03			/* Set Word Length 8*/
#define INT_CLK			0x00			/* Internal Clock Mode */
#define	EXT_CLK			0x40			/* External Clock Mode */

/* Bit definitions within UCON0/1 Control Register*/
#define UCON_RX			0x01			/* Receive Mode -Interrupt*/
#define UCON_STAT_EN		0x04			/* Status Interrrupt -Enable*/
#define UCON_TX			0x08			/* Transmit Mode-Interrupt*/
#define UCON_TX_DIS		0x01			/* Transmit Interrupt -Disable*/
#define UCON_DSR			0x20			/* Data Set Ready -Enable*/
#define UCON_BREAK			0x40			/* Set Break*/
#define UCON_RX_TX_RESET 	0xe4		/* Rx and Tx Reset */

/* Bit definitions within USTAT0/1 Status Register*/

#define USTAT_DTR_LOW		0x10			/* DTR Enable */
#define USTAT_DTR_HIGH	0x00			/* DTR Disable*/
#define USTAT_TX_READY	0x40	 		/* Transmitter Ready for another char */
#define USTAT_RX_AVAIL	0x20	 		/* Character has arrived*/ 
#define USTAT_OVER_ERR  	0x01			/* Over Run Error*/
#define USTAT_PAR_ERR   	0x02			/* Parity Error*/
#define USTAT_FRAME_ERR 	0x04			/* Frame Error*/ 
#define USTAT_RX_READY  	0x20			/* Receive Data Buffer*/

#ifndef REG_READ
#define REG_READ(reg, result) \
	((result) = *(volatile ulong *)(reg))
#endif /*READ REG*/

#ifndef REG_WRITE
#define REG_WRITE(reg, data) \
	(*((volatile ulong *)(reg)) = (data))
#endif /*WRITE REG*/

/*************************************************************************
*
* DRAM Memory Bank 0 area MAP for Exception vector table 
* and Stack, User code area. 
*
*/

#define DRAM_BASE  		0x0	/* Final start address of DRAM */
#define DRAM_LIMIT 		0x1000000			/* 16MByte */
//#define DRAM_LIMIT 		0x800000			/* 8MByte */
#define	RESET_DRAM_START	0x1000000	/* Start of DRAM on power-up */
#define	RESET_ROM_START	0x0	/* Start od ROM on power-up */

/****************************************************************************
*
* Format of the Program Status Register 
*/

#define FBit 	 	0x40
#define IBit  	 	0x80
#define LOCKOUT  	0xC0 	/* Interrupt lockout value */
#define LOCK_MSK 	0xC0 	/* Interrupt lockout mask value */
#define MOD_MASK 	0x1F 	/* Processor Mode Mask */
#define UDF_MODE 	0x1B 	/* Undefine Mode(UDF) */
#define ABT_MODE 	0x17 	/* Abort Mode(ABT) */
#define SUP_MODE 	0x13 	/* Supervisor Mode (SVC) */
#define IRQ_MODE 	0x12 	/* Interrupt Mode (IRQ) */
#define FIQ_MODE 	0x11 	/* Fast Interrupt Mode (FIQ) */
#define USR_MODE 	0x10 	/* User Mode(USR) */

/*************************************************************************
* SYSTEM CLOCK 
*/

#define MHz 		 1000000

#define fMCLK_MHz 	 72000000 	/* 50MHz, KS32C50100*/
#define fMCLK 		 fMCLK_MHz/MHz

/*************************************************************************
* SYSTEM MEMORY CONTROL REGISTER EQU TABLES 
*/

/* SYSCFG Register Value */
#define SYSCONFIG_VAL_EDRAM	0x03ff0000	/* System Configuration Value, EDO RAM */
#define SYSCONFIG_VAL_SDRAM	0xA3ff0000  /* System Configuration Value, SDRAM */

/*External I/O access timing register 0,1*/
#define rEXTACON0	(0x0E780E78)
#define rEXTACON1	(0x0E780E78)

/* -> EXTDBWTH : Memory Bus Width register */
#define DSR0  	(1<<0) 	/* ROM0, 0 : Disable, 1 : Byte etc.*/
#define DSR1  	(2<<2) 	/* ROM1  */
#define DSR2  	(3<<4)	/* ROM2  */
#define DSP  	(3<<6) 	/* DSP  */
#define DSRD0	(0<<8) 	/* Dummy  */
#define DSRD1  	(0<<10) /* Dummy  */
#define DSD0  	(3<<12) 	/* DRAM0 */
#define DSD1  	(3<<14) 	/* DRAM1 */
#define DSD2  	(3<<16) 	/* DRAM2 */
#define DSD3  	(3<<18) 	/* DRAM3 */
#define DSX0  	(2<<20) 	/* EXTIO0*/
#define DSX1  	(2<<22) 	/* EXTIO1*/
#define DSX2  	(3<<24) 	/* EXTIO2*/
#define DSX3  	(3<<26) 	/* EXTIO3*/

/* 20020612 by drsohn
#define rEXTDBWTH 	(DSR0+DSR1+DSR2+DSP+DSD0+DSD1+DSD2+DSD3+DSX0+DSX1+DSX2+DSX3)
*/
/***********************************************************
*
* -> ROMCON0 : ROM Bank0 Control register 
*/
#ifdef SD16BIT
#define rEXTDBWTH		0x0AAAA0BA				// for 16bit sdram access - all banks
#else
#define rEXTDBWTH		0x0FFFF0BA				// all SDRAM banks : 32bit
#endif

	
#if defined(AM29LV320D) || defined(AT49BV321)
#define rROMCON0_R		0x040003FF
#define rROMCON0_B		0x140403FF
#else
#define rROMCON0_R		0x020003FF
#define rROMCON0_B		0x120403FF
#endif
/***************************************************************************
* -> ROMCON1 : ROM Bank1 Control register 
*/

#if defined(AM29LV320D) || defined(AT49BV321)
#define rROMCON1_R		0x060103FF
#define rROMCON1_B		0x160503FF
#else
#define rROMCON1_R		0x040083FF
#define rROMCON1_B		0x140483FF
#endif

#define rROMCON2 	 	0x00000000
#define rROMCON3		0x24080000		// 0x2000000 ~ 0x2400000 : PCMCIA address map

//-----------------------------------------------------------------------------
// Configuration of DRAM Bank
//-----------------------------------------------------------------------------
#if defined(SD16MB) && defined(SD2BANK) && defined(SD32BIT)
#define rSDRAMCON0_R		0x18040380
#define rSDRAMCON0_B		0x08000380
#define rSDRAMCON1_R		0x20060380
#define rSDRAMCON1_B		0x10020380
#endif

#if defined(SD16MB) && defined(SD1BANK) && defined(SD32BIT)
#define rSDRAMCON0_R		0x20040380
#define rSDRAMCON0_B		0x10000380
#define rSDRAMCON1_R		0x0
#define rSDRAMCON1_B		0x0
#endif

#if defined(SD16MB) && defined(SD2BANK) && defined(SD16BIT)
#define rSDRAMCON0_R		0x58040380
#define rSDRAMCON0_B		0x48000380
#define rSDRAMCON1_R		0x60060380
#define rSDRAMCON1_B		0x50020380
#endif

#if defined(SD16MB) && defined(SD1BANK) && defined(SD16BIT)
#define rSDRAMCON0_R		0x60040380
#define rSDRAMCON0_B		0x50000380
#define rSDRAMCON1_R		0x0
#define rSDRAMCON1_B		0x0
#endif

#if defined(SD8MB) && defined(SD32BIT)
#define rSDRAMCON0_R		0x18040380
#define rSDRAMCON0_B		0x08000380
#define rSDRAMCON1_R		0x0
#define rSDRAMCON1_B		0x0
#endif

#if defined(SD8MB) && defined(SD16BIT)
#define rSDRAMCON0_R		0x58040380
#define rSDRAMCON0_B		0x48000380
#define rSDRAMCON1_R		0x0
#define rSDRAMCON1_B		0x0
#endif

#define rSDRAMCON2			0x00
#define rSDRAMCON3 			0x00

#define rREFEXTCON				0xCE27E360

#define INT_NUM_LEVELS	23
#define INT_CSR_MASK		0x007fffff	
#define INT_MODE_IRQ		0x00

#define INT_LVL_EXTINT0		0	/* External Interrupt0 */
#define INT_LVL_EXTINT1		1	/* External Interrupt1 */
#define INT_LVL_EXTINT2		2	/* External Interrupt2 */
#define INT_LVL_EXTINT3		3	/* External Interrupt3 */
#define INT_LVL_EXTINT4		4	/* External Interrupt4 */
#define INT_LVL_EXTINT5		5	/* External Interrupt5 */
#define INT_LVL_EXTINT6		6	/* External Interrupt6 */
#define INT_LVL_UARTTX		7	/* UART 0 Transmit Interrupt */
#define INT_LVL_UARTRX		8	/* UART 0 Receive & Error Interrupt */
#define INT_LVL_TIMER0		9	/* Timer 0 Interrupt */
#define INT_LVL_TIMER1		10	/* Timer 1 Interrupt  */
#define INT_LVL_TIMER2		11	/* Timer 2 Interrupt */
#define INT_LVL_GDMA0			12	/* GDMA channel 0 interrupt*/
#define INT_LVL_GDMA1			13	/* GDMA channel 1 interrupt */
#define INT_LVL_USB			14	/* USB interrupt */
#define INT_LVL_SAR_ERR		15	/* SAR TX/RX Done interrupt */
#define INT_LVL_SAR_DONE		16	/* SAR TX/RX Error interrupt */
#define INT_LVL_ETHTxA		17	/* Ethernet controller0 Tx Interrupt*/
#define INT_LVL_ETHRxA		18	/* Ethernet controller0 Rx Interrupt */
#define INT_LVL_ETHTxB		19	/* Ethernet controller1 Tx Interrupt*/
#define INT_LVL_ETHRxB		20	/* Ethernet controller1 Rx Interrupt */
#define INT_LVL_IIC			21	/* IIC -Bus Interrupt */
#define INT_LVL_SPI			22	/* SPI Interrupt */

/**********************************************************************************************************
* Cache Definitions
*
*/
#define NON_CACHE_REGION		0x4000000
#define CACHE_ENABLE		0x02
#define CACHE_4K			0x00
#define WRITE_BUFF		0x04
#define TAGRAM			0x11000000

/*Use UART CLOCK - use UART clk = 1, Internal clk = 0 */
#define	USE_UART_CLK		0


/* 20020612 by drsohn */
#define rSYSCFG		0x83FF0000
#define SYS_INIT_BASE		0x3FF3010
#define VPint	*(volatile unsigned int *)
#define UARTSTAT0           (VPint(ASIC_BASE+0xd008))
#define	UART_STAT_TXB_EMPTY     0x40
#define UARTTXH0            (VPint(ASIC_BASE+0xd00c))
#define UARTRXB0            (VPint(ASIC_BASE+0xd010))
/* drsohn end */


//-----------------------------------------------------------------------------
// Declare function protype
//-----------------------------------------------------------------------------
/*
extern void reset_cpu(ulong addr);
extern void intDisable(int vect);
*/

#endif	
