/* vi: set sw=4 ts=4: */
/*
 * $Id: ping.c,v 1.56 2004/03/15 08:28:48 andersen Exp $
 * Mini ping implementation for busybox
 *
 * Copyright (C) 1999 by Randolph Chung <tausq@debian.org>
 *
 * Adapted from the ping in netkit-base 0.10:
 * Copyright (c) 1989 The Regents of the University of California.
 * Derived from software contributed to Berkeley by Mike Muuss.
 *
 * Licensed under GPLv2 or later, see file LICENSE in this tarball for details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>
#include <asm/cksim.h>
#include "busybox.h"
#include "usage.h"

extern int avi_CommStr2Int(const unsigned char *str);

void I2CRead(int dev_addr, int a, char *pbuf, int count);
void I2CWrite(int dev_addr, int addr, int dat);

#define I2cWrite(a, b, c) I2CWrite(a, b, c)
#define I2cRead(a, b, c) I2CRead(a, b, c)

FILE * i2c_handle = NULL;

#define BB_IIC_UNKNOW       0
#define BB_IIC_READ         1
#define BB_IIC_WRITE        2
#define BB_IIC_MSGREAD      3

#define BB_IIC_READ_ARGC    4
#define BB_IIC_WRITE_ARGC   5

/* iic opration mode */

#define BB_IIC_READ_MAX     16



FILE * Open_I2CDevice(void)
{
	FILE * fd;
	int res;

	if ((fd = open("/dev/i2c/0", 2)) >= 0) 
	{
		res = ioctl(fd,I2C_TENBIT,0);
	}
	else
	{
		printf("file open error %x\n", fd);
		return NULL;
	}
	return fd;
}

void I2CRead(int dev_addr, int addr, char *pbuf, int count)
{
	char buf[ BB_IIC_READ_MAX ] = {0};
	int res;
	
	if ( count > BB_IIC_READ_MAX )
	{
		count = BB_IIC_READ_MAX;
	}
	else if ( 0 == count )
	{
		count = 1;
	}

	res = ioctl(i2c_handle, I2C_SLAVE_FORCE, dev_addr>>1);

	buf[0] = addr&0xff;
	res = write(i2c_handle, buf, 1);
	res = read(i2c_handle, buf, count);
	
	memcpy( pbuf, buf, count);

}

void I2CWrite(int dev_addr, int addr, int dat)
{
	char buf[12] = {0};
	int res;

	res = ioctl(i2c_handle, I2C_SLAVE_FORCE, dev_addr>>1);
	
	buf[0] = addr&0xff;
	buf[1] = dat;
	res = write(i2c_handle, buf, 2);
}


static void i2c_msg_read( int dev_addr, int addr, char *pbuf, int count )
{
    int	i = 0;
	unsigned char i2cbuf[ BB_IIC_READ_MAX ], ad[ 1 ];
	struct i2c_msg msgs[ 2 ];
	struct i2c_rdwr_ioctl_data rw_msg;
	
	if ( count > BB_IIC_READ_MAX )
	{
		count = BB_IIC_READ_MAX;
	}
	else if ( 0 == count )
	{
		count = 1;
	}
	
    /* packet the message0 */
    msgs[ 0 ].addr = (unsigned short)( dev_addr ) >> 1;
    msgs[ 0 ].flags = 0;
    msgs[ 0 ].len = 1;
    msgs[ 0 ].buf = ad;
    
    //set reg address
    ad[ 0 ] = addr;
    
    /* packet the message1 */
    msgs[ 1 ].addr = (unsigned short)( dev_addr ) >> 1;
    msgs[ 1 ].flags = I2C_M_RD;
    msgs[ 1 ].len = count;
    msgs[ 1 ].buf = i2cbuf;
    
    rw_msg.msgs = msgs;
    rw_msg.nmsgs = 2;
    
    ioctl( i2c_handle, I2C_RDWR, &rw_msg );
    
    memcpy( pbuf, i2cbuf, count);
}


int i2c_main(int argc, char **argv)
{
    unsigned int ops;
    int dev_addr, addr,cnt;
    char tmp;
    char rbuf[16];
    
    /* check the argv */
    ops = BB_IIC_UNKNOW;
    i2c_handle = NULL;
    if ( argc > 1 )
    {
        if ( 'r' == argv[1][0] )
        {
            ops = BB_IIC_READ;
        }
        else if ( 'w' == argv[1][0] )
        {
            ops = BB_IIC_WRITE;
        }
        else if ( 'm' == argv[1][0] )
        {
            ops = BB_IIC_MSGREAD;
        }
        else
        {
            i2c_show_mode();
            goto _i2c_out;
        }
    }
    
    /* do for read or write */
    switch ( ops )
    {
        case BB_IIC_READ:
        case BB_IIC_MSGREAD:
            {
                if ( argc < BB_IIC_READ_ARGC  )
                {
                    printf(i2c_trivial_usage);
                }
                else
                {
                    i2c_handle = Open_I2CDevice();
                    if ( NULL == i2c_handle )
                    {
                        printf("i2c device open failed!\n" );
                        goto _i2c_out;
                    }
                    
                    dev_addr = avi_CommStr2Int( argv[2] );
                    addr = avi_CommStr2Int( argv[3] );
                    cnt = 0;
                    
                    if ( argc > 4 )
                    {
                    	cnt = avi_CommStr2Int( argv[4] );
                    }
                    
					if ( cnt > 16 )
                	{
                		cnt = 16;
                	}
                	else if ( 0 == cnt )
                	{
                		cnt = 1;
                	}
                
                    if ( BB_IIC_READ == ops )
                    {
                        I2CRead( dev_addr, addr, rbuf, cnt );
                    }
                    else
                    {
                        i2c_msg_read( dev_addr, addr, rbuf, cnt );
                    }
                    
                    for ( tmp = 0; tmp < cnt; tmp ++ )
                    {
                    	printf( "I2CR[0x%02x]->[0x%02x] = %#x\n", dev_addr, addr + tmp, rbuf[tmp] );
                    }
                    
                }
                break;
            }
        case BB_IIC_WRITE:
            {
                if ( BB_IIC_WRITE_ARGC != argc )
                {
                    printf(i2c_trivial_usage);
                }
                else
                {
                    i2c_handle = Open_I2CDevice();
                    if ( NULL == i2c_handle )
                    {
                        printf("i2c device open failed!\n" );
                        goto _i2c_out;
                    }
                    
                    dev_addr = avi_CommStr2Int( argv[2] );
                    addr = avi_CommStr2Int( argv[3] );
                    tmp = avi_CommStr2Int( argv[4] );
                    
                    I2CWrite( dev_addr, addr, tmp );
                    printf( "I2CW[%#x]->[%#x] = %#x\n", dev_addr, addr, tmp );
                }
                break;
            }
        default:
            {
                printf("i2c unsupport this command[%s]\n", argv[1] );
                goto _i2c_out;
            }
    }
    

_i2c_out:
    if ( NULL != i2c_handle )
    {
        close(i2c_handle);
        i2c_handle = NULL;
    }
	return EXIT_SUCCESS;
}


