/*
 *  linux/arch/arm/kernel/bios32.c
 *
 *  PCI bios-type initialisation for PCI machines
 *
 *  Bits taken from various places.
 */
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/init.h>

#include <asm/page.h> /* for BUG() */
#include <asm/irq.h>
#include <asm/cksim.h>
#include <asm/pci.h>
#include <asm/pci_hardware.h>

#define __virt_to_bus(x) x

//static int debug_pci = 0;//1;

#define BIOS32_DEBUG
#undef BIOS32_DEBUG

#ifdef BIOS32_DEBUG
	#define BPRINTK(fmt, args...) printk(KERN_ERR "%s: " fmt, __FUNCTION__, ## args)
#else
	#define BPRINTK(fmt, args...)
#endif


int have_isa_bridge;

struct pci_sys_data {
	/*
	 * The hardware we are attached to
	 */
	struct hw_pci	*hw;

	unsigned long	mem_offset;

	/*
	 * These are the resources for the root bus.
	 */
	struct resource *resource[3];
};

void pcibios_report_status(u_int status_mask, int warn)
{
	struct pci_dev *dev;

	pci_for_each_dev(dev) {
		u16 status;

		/*
		 * ignore host bridge - we handle
		 * that separately
		 */
		if (dev->bus->number == 0 && dev->devfn == 0)
			continue;

		pci_read_config_word(dev, PCI_STATUS, &status);

		status &= status_mask;
		if (status == 0)
			continue;

		/* clear the status errors */
		pci_write_config_word(dev, PCI_STATUS, status);

		if (warn)
			printk("(%02x:%02x.%d: %04X) ", dev->bus->number,
				PCI_SLOT(dev->devfn), PCI_FUNC(dev->devfn),
				status);
	}
}


static void pci_fixup_unassign(struct pci_dev *dev)
{
	dev->resource[0].end -= dev->resource[0].start;
	dev->resource[0].start = 0;
}

/*
 * Prevent the PCI layer from seeing the resources allocated to this device
 * if it is the host bridge by marking it as such.  These resources are of
 * no consequence to the PCI layer (they are handled elsewhere).
 */
static void pci_fixup_isp1561(struct pci_dev *dev)
{
//	int i;

  //      printk("Do you want to fixup ISP1561?\n");
//		for (i = 0; i < PCI_NUM_RESOURCES; i++) {
//			dev->resource[i].start = 0;
//			dev->resource[i].end   = 0;
//			dev->resource[i].flags = 0;
//		}
     return;
}

unsigned long pci_bridge_check_io(struct pci_dev *bridge)
{
	u16 io;

	pci_read_config_word(bridge, PCI_IO_BASE, &io);
	if (!io) {
		pci_write_config_word(bridge, PCI_IO_BASE, 0xf0f0);
		pci_read_config_word(bridge, PCI_IO_BASE, &io);
		pci_write_config_word(bridge, PCI_IO_BASE, 0x0);
	}
	if (io)
		return IORESOURCE_IO;
	printk(KERN_WARNING "PCI: bridge %s does not support I/O forwarding!\n",
				bridge->name);
	return 0;
}

struct pci_fixup pcibios_fixups[] = {
	{
		PCI_FIXUP_HEADER,
		PCI_VENDOR_ID_PHILIPS,	0x1561,
		pci_fixup_isp1561
	}, { 0 }
};

void 
pcibios_update_resource(struct pci_dev *dev, struct resource *root,
			struct resource *res, int resource)
{
	struct pci_sys_data *sys = dev->sysdata;
	u32 val, check;
	int reg;

	BPRINTK("PCI: Assigning %3s %08lx to %s\n",
			res->flags & IORESOURCE_IO ? "IO" : "MEM",
			res->start, dev->name);

	if (resource < 6) {
		reg = PCI_BASE_ADDRESS_0 + 4*resource;
	} else if (resource == PCI_ROM_RESOURCE) {
		reg = dev->rom_base_reg;
	} else {
		/* Somebody might have asked allocation of a
		 * non-standard resource.
		 */
		return;
	}

	val = res->start;
	if (res->flags & IORESOURCE_MEM)
		val -= sys->mem_offset;
	val |= res->flags & PCI_REGION_FLAG_MASK;

	pci_write_config_dword(dev, reg, val);
	pci_read_config_dword(dev, reg, &check);
	if ((val ^ check) & ((val & PCI_BASE_ADDRESS_SPACE_IO) ?
	    PCI_BASE_ADDRESS_IO_MASK : PCI_BASE_ADDRESS_MEM_MASK)) {
		printk(KERN_ERR "PCI: Error while updating region "
			"%s/%d (%08x != %08x)\n", dev->slot_name,
			resource, val, check);
	}
}

void pcibios_update_irq(struct pci_dev *dev, int irq)
{
	BPRINTK("PCI: Assigning IRQ %02d to %s\n", irq, dev->name);
	pci_write_config_byte(dev, PCI_INTERRUPT_LINE, irq);
}

/*
 * If the bus contains any of these devices, then we must not turn on
 * parity checking of any kind.  Currently this is CyberPro 20x0 only.
 */
static inline int pdev_bad_for_parity(struct pci_dev *dev)
{
	return (dev->vendor == PCI_VENDOR_ID_INTERG &&
		(dev->device == PCI_DEVICE_ID_INTERG_2000 ||
		 dev->device == PCI_DEVICE_ID_INTERG_2010));
}

/*
 * Adjust the device resources from bus-centric to Linux-centric.
 */
static void
pdev_fixup_device_resources(struct pci_sys_data *root, struct pci_dev *dev)
{
	int i;

	for (i = 0; i < PCI_NUM_RESOURCES; i++) {
		if (dev->resource[i].start == 0)
			continue;
		if (dev->resource[i].flags & IORESOURCE_MEM) {
			dev->resource[i].start += root->mem_offset;
			dev->resource[i].end   += root->mem_offset;
		}
	}
}

static void 
pbus_assign_bus_resources(struct pci_bus *bus, struct pci_sys_data *root)
{
	struct pci_dev *dev = bus->self;
	int i;

	if (dev) {
		for (i = 0; i < 3; i++) {
			if(root->resource[i]) {
				bus->resource[i] = &dev->resource[PCI_BRIDGE_RESOURCES+i];
				bus->resource[i]->end  = root->resource[i]->end;
				bus->resource[i]->name = bus->name;
			}
		}
		bus->resource[0]->flags |= pci_bridge_check_io(dev);
		bus->resource[1]->flags |= IORESOURCE_MEM;

		if (root->resource[2])
			bus->resource[2]->flags = root->resource[2]->flags;
		else {
			/* no prefetchable memory region - disable it */
			bus->resource[2]->start = 1024*1024;
			bus->resource[2]->end   = bus->resource[2]->start - 1;
		}
	} else {
		/*
		 * Assign root bus resources.
		 */
		for (i = 0; i < 3; i++)
			bus->resource[i] = root->resource[i];
	}
}

/*
 * pcibios_fixup_bus - Called after each bus is probed,
 * but before its children are examined.
 */
void pcibios_fixup_bus(struct pci_bus *bus)
{
	struct pci_sys_data *root = bus->sysdata;
	struct list_head *walk;
	u16 features = PCI_COMMAND_SERR | PCI_COMMAND_PARITY;
	u16 all_status = -1;

	pbus_assign_bus_resources(bus, root);

	/*
	 * Walk the devices on this bus, working out what we can
	 * and can't support.
	 */
	for (walk = bus->devices.next; walk != &bus->devices; walk = walk->next) {
		struct pci_dev *dev = pci_dev_b(walk);
		u16 status;

		pdev_fixup_device_resources(root, dev);

//        printk("PCI Read devfunc %x\n", dev->devfn);
		pci_read_config_word(dev, PCI_STATUS, &status);
		all_status &= status;

		if (pdev_bad_for_parity(dev))
			features &= ~(PCI_COMMAND_SERR | PCI_COMMAND_PARITY);

		/*
		 * If this device is an ISA bridge, set the have_isa_bridge
		 * flag.  We will then go looking for things like keyboard,
		 * etc
		 */
		if (dev->class >> 8 == PCI_CLASS_BRIDGE_ISA ||
		    dev->class >> 8 == PCI_CLASS_BRIDGE_EISA)
			have_isa_bridge = !0;
	}

	/*
	 * If any device on this bus does not support fast back to back
	 * transfers, then the bus as a whole is not able to support them.
	 * Having fast back to back transfers on saves us one PCI cycle
	 * per transaction.
	 */
	if (all_status & PCI_STATUS_FAST_BACK)
		features |= PCI_COMMAND_FAST_BACK;

	/*
	 * Now walk the devices again, this time setting them up.
	 */
	for (walk = bus->devices.next; walk != &bus->devices; walk = walk->next) {
		struct pci_dev *dev = pci_dev_b(walk);
		u16 cmd;

		pci_read_config_word(dev, PCI_COMMAND, &cmd);
		cmd |= features;
		pci_write_config_word(dev, PCI_COMMAND, cmd);
	}

	/*
	 * Report what we did for this bus
	 */
	printk(KERN_INFO "PCI: bus%d: Fast back to back transfers %sabled\n",
		bus->number, (features & PCI_COMMAND_FAST_BACK) ? "en" : "dis");
}

/*
 * Convert from Linux-centric to bus-centric addresses for bridge devices.
 */
void 
pcibios_fixup_pbus_ranges(struct pci_bus *bus, struct pbus_set_ranges_data *ranges)
{
	struct pci_sys_data *root = bus->sysdata;

	ranges->mem_start -= root->mem_offset;
	ranges->mem_end -= root->mem_offset;
	ranges->prefetch_start -= root->mem_offset;
	ranges->prefetch_end -= root->mem_offset;
}

u8 no_swizzle(struct pci_dev *dev, u8 *pin)
{
	return 0;
}

static int irqmap_personal_server[] __initdata = {
	IRQ_IN0, IRQ_IN1, IRQ_IN2, IRQ_IN3, 0, 0, 0,
	IRQ_DOORBELLHOST, IRQ_DMA1, IRQ_DMA2, IRQ_PCI
};

static int __init personal_server_map_irq(struct pci_dev *dev, u8 slot, u8 pin)
{
	unsigned char line;


	pci_read_config_byte(dev, PCI_INTERRUPT_LINE, &line);
    if (line == 0) 
        return CKCORE_PCI_IRQ;
    return line;
}

#define MAX_SLOTS		21

#define PCICMD_ERROR_BITS ((PCI_STATUS_DETECTED_PARITY | \
			PCI_STATUS_REC_MASTER_ABORT | \
			PCI_STATUS_REC_TARGET_ABORT | \
			PCI_STATUS_PARITY) << 16)

//extern int setup_arm_irq(int, struct irqaction *);
//extern void pcibios_report_status(u_int status_mask, int warn);
//extern void register_isa_ports(unsigned int, unsigned int, unsigned int);

void print_istatus(void)
{
    volatile unsigned int *icrp;
 	icrp = (volatile unsigned int *) (CKSIM_PIC_BASE);

	printk("Enter CPU IStatus %x and mask bit %x\n", *PLD_CPU_ISTATUS, *PLD_CPU_IMASK);
//	(*PLD_CPU_ISTATUS) |= (cpu_to_le32(PLD_INTA_ENABLE));//needn't clear the status

	printk("CPU IStatus %x and mask bit %x\n", *PLD_CPU_ISTATUS, *PLD_CPU_IMASK);
	printk("PICMASK %x PICMODE %x COW %x\n", icrp[CKSIM_PIC_MASK], icrp[CKSIM_PIC_MODE], \
      		icrp[CKSIM_PIC_COW1]);
}

static unsigned long configure_read(struct pci_dev *dev, int where, void *value, int size)
{
	unsigned long addr = 0;
	unsigned int devfn = dev->devfn;
	unsigned long st = 0;
	
	if (dev->bus->number == 0) 
    {
	if (PCI_SLOT(devfn) == 0)
        {  
              int i;
			// For devfn 0, point at the AHB-PCI Registers 
			
			 
             if (size == 1)
              {
                  *(unsigned char *)value = (*((unsigned char *)(CKCOREPLD_BASE + where)));
              }else if (size == 2)
              {
                   #ifdef __BIG_ENDIAN
                        *(unsigned short *)value = cpu_to_le16(*((unsigned short *)(CKCOREPLD_BASE + where)));
                    #else
                        *(unsigned short *)value = (*((unsigned short *)(CKCOREPLD_BASE + where)));
                    #endif
              }else
              {
                #ifdef __BIG_ENDIAN
                  *(unsigned long *)value = cpu_to_le32(*((unsigned long *)(CKCOREPLD_BASE + where)));
                 #else
                    *(unsigned long *)value = (*((unsigned long *)(CKCOREPLD_BASE + where)));
                 #endif
              }
             for (i = 0; i < 10000; i++) ;
             for (i = 0; i < 10000; i++) ;
             
          }
		else {
              int i;
              //unsigned long value_test[5];//value_test should have DWORD align 
              unsigned int *value_test;
              

			devfn -= 1 << 3;
             for (i = 0; i < 10000; i++) ;

             //memset(value_test, 0, 20);
             
				value_test = kmalloc(20,GFP_DMA);
				if( NULL == value_test )
				{
					printk("kmalloc failed!\n");
				}
				else
				{
					//printk("value_test = 0x%08lx\n",value_test);
				}
				
				memset( value_test, 0, 20 );
             
			if (devfn < PCI_DEVFN(MAX_SLOTS, 0))
            {   
				    addr = PCICFG0_BASE  | (devfn << 8);
				    
				   #ifdef __BIG_ENDIAN
                      *PLD_RDMA_SRC_ADDR = le32_to_cpu(((unsigned long)addr + ((where) &(~0x03))) | (DEV1_IDSEL>>(devfn>>3)));
                      *PLD_RDMA_DST_ADDR = le32_to_cpu(value_test);
                      *PLD_RDMA_CTRL = le32_to_cpu((0x10 << 8) |PLD_DMA_CONFIG_READ);
                      *PLD_RDMA_CTRL |= le32_to_cpu(PLD_DMA_START);
                       while (cpu_to_le32(*PLD_RDMA_CTRL) & PLD_DMA_START)
                      {
    	                 BPRINTK("Before and control %x Read from %x to %x  dev %x \n", cpu_to_le32(*PLD_RDMA_CTRL), cpu_to_le32(*PLD_RDMA_SRC_ADDR), \
            	         cpu_to_le32(*PLD_RDMA_DST_ADDR), addr);
                      };//wait until the start bit cleared
    
                     if (cpu_to_le32(*PLD_CPU_ISTATUS) & 0x0200)
                      {
                        //I set the value for dma error cause unknow data in
                        memset(value_test, 0, 20);
    			        st = 1;
                      }
                      
                     *PLD_CPU_ISTATUS &= cpu_to_le32(0x0300);//bit 1 clear the bit, refer to datasheet
                   #else
                   
                   
                   
                       *PLD_RDMA_SRC_ADDR = (((unsigned long)addr + ((where) &(~0x03))) | (DEV1_IDSEL>>(devfn>>3)));
                      *PLD_RDMA_DST_ADDR = (value_test);
                      *PLD_RDMA_CTRL = ((0x10 << 8) |PLD_DMA_CONFIG_READ);
                      *PLD_RDMA_CTRL |= (PLD_DMA_START);
                       while ((*PLD_RDMA_CTRL) & PLD_DMA_START)
                      {
                        #ifdef __BIG_ENDIAN
    	                 BPRINTK("Before and control %x Read from %x to %x  dev %x \n", cpu_to_le32(*PLD_RDMA_CTRL), cpu_to_le32(*PLD_RDMA_SRC_ADDR), \
            	         (*PLD_RDMA_DST_ADDR), addr);
            	         #else
            	         
                        BPRINTK("Before and control %x Read from %x to %x  dev %x \n", (*PLD_RDMA_CTRL), (*PLD_RDMA_SRC_ADDR), \
            	         (*PLD_RDMA_DST_ADDR), addr);         
            	         #endif
                      };//wait until the start bit cleared
                      
    
                     if ((*PLD_CPU_ISTATUS) & 0x0200)
                      {
                        //I set the value for dma error cause unknow data in
                        memset(value_test, 0, 20);
    			        st = 1;
                      }
                      
                     *PLD_CPU_ISTATUS &= (0x0300);//bit 1 clear the bit, refer to datasheet                  
                   #endif
                 
                 
            }
            
            
		    rmb();
	        BPRINTK("read value %x offset %x\n", cpu_to_le32(value_test[0]), where);
	        
           if (size == 1)
            {
                *(unsigned char *)value = (*((unsigned char *)(((unsigned long)value_test + (where&0x03)) )))&0xff;
            }
            else if (size == 2)
            {
                #ifdef __BIG_ENDIAN
                    *(unsigned short *)value = cpu_to_le16((*((unsigned short *)(((unsigned long)value_test) + ((where)&0x03))))&0xffff);
                #else
                    *(unsigned short *)value = ((*((unsigned short *)(((unsigned long)value_test) + ((where)&0x03))))&0xffff);
                #endif
            }else
            {
                #ifdef __BIG_ENDIAN
                    *(unsigned long *)value = cpu_to_le32(*((unsigned long *)(((unsigned long)value_test) + (where&0x03))));
                
                #else
                    *(unsigned long *)value = (*((unsigned long *)(((unsigned long)value_test) + (where&0x03))));
                #endif
            }
            
             kfree(value_test);
		}
	} else
  {
        printk("PCI Bridge Cross, maybe some error encountered\n");
		addr = PCICFG1_BASE | (dev->bus->number << 16) | (devfn << 8);
   }
   

	return st;
}

static unsigned long configure_write(struct pci_dev *dev, int where, void *value, int size)
{
	unsigned long addr = 0;
	unsigned int devfn = dev->devfn;
	

	if (dev->bus->number == 0) {
		if (PCI_SLOT(devfn) == 0)
        {  
              int i;
			/* * For devfn 0, point at the AHB-PCI Registers */
			 //here I am according to a rule that adress should align to it's write value alignment
             if (size == 1)
              {
                  *((unsigned char *)(CKCOREPLD_BASE + where)) = (*(unsigned char *)value);
              }else if (size == 2)
              {
               #ifdef __BIG_ENDIAN
                  *((unsigned short *)(CKCOREPLD_BASE + where)) = cpu_to_le16(*(unsigned short *)value);
                #else
                    *((unsigned short *)(CKCOREPLD_BASE + where)) = (*(unsigned short *)value);
                #endif
              }
              else
              {
                #ifdef __BIG_ENDIAN
                  *((unsigned long *)(CKCOREPLD_BASE + where)) = cpu_to_le32(*(unsigned long *)value);
                #else
                    *((unsigned long *)(CKCOREPLD_BASE + where)) = (*(unsigned long *)value);
                #endif
              }
             for (i = 0; i < 10000; i++) ;
             for (i = 0; i < 10000; i++) ;
          }
		else {
              int i;

			devfn -= 1 << 3;
             for (i = 0; i < 10000; i++) ;

			if (devfn < PCI_DEVFN(MAX_SLOTS, 0))
            {   
				addr = PCICFG0_BASE  | (devfn << 8);
				
				#ifdef __BIG_ENDIAN
                      *PLD_WDMA_DST_ADDR = le32_to_cpu(((unsigned long)addr + ((where))) | \
                        (DEV1_IDSEL>>(devfn>>3)));
                      *PLD_WDMA_SRC_ADDR = le32_to_cpu(value);//((unsigned long)value_test + 0x03)&(~0x03);
                      *PLD_WDMA_CTRL =  le32_to_cpu((size << 8) |PLD_DMA_CONFIG_WRITE);
                      *PLD_WDMA_CTRL |= le32_to_cpu(PLD_DMA_START);
                       while (cpu_to_le32(*PLD_WDMA_CTRL) & PLD_DMA_START)
                      {
                      };//wait until the start bit cleared
    
                     if (cpu_to_le32(*PLD_CPU_ISTATUS) & 0x02)
                      {
                        //I set the value for dma error cause unknow data in
                               printk("Write error %x size %x src %x Status %x\n", where, size, value, *PLD_CPU_ISTATUS);
                      }
                      *PLD_CPU_ISTATUS &= le32_to_cpu(0x03);//bit 1 clear the bit, refer to datasheet
                #else
                       *PLD_WDMA_DST_ADDR = (((unsigned long)addr + ((where))) | (DEV1_IDSEL>>(devfn>>3)));
                      *PLD_WDMA_SRC_ADDR = (value);//((unsigned long)value_test + 0x03)&(~0x03);
                      *PLD_WDMA_CTRL =  ((size << 8) |PLD_DMA_CONFIG_WRITE);
                      *PLD_WDMA_CTRL |= (PLD_DMA_START);
                       while ((*PLD_WDMA_CTRL) & PLD_DMA_START)
                      {
                      };//wait until the start bit cleared
    
                     if ((*PLD_CPU_ISTATUS) & 0x02)
                      {
                        //I set the value for dma error cause unknow data in
                               printk("Write error %x size %x src %x Status %x\n", where, size, value, *PLD_CPU_ISTATUS);
                      }
                      *PLD_CPU_ISTATUS &= (0x03);//bit 1 clear the bit, refer to datasheet               
                #endif
	   	        BPRINTK("write value %x offset %x\n", *(unsigned int *)(value), where);
            }
		}
	} 
	else
  {
        printk("PCI Bridge Cross, maybe some error encountered\n");
		addr = PCICFG1_BASE | (dev->bus->number << 16) | (devfn << 8);
   }

	return addr;
}


static int dahua_read_config_byte(struct pci_dev *dev, int where, u8 *value)
{

   BPRINTK("read configure byte %x ", where);
   
   configure_read(dev, where, value, sizeof(u8));
   
   BPRINTK("with value readed %x\n", *value);

	return PCIBIOS_SUCCESSFUL;
}

static int dahua_read_config_word(struct pci_dev *dev, int where, u16 *value)
{
	BPRINTK("read configure word %x ", where);
	
    configure_read(dev, where, value, sizeof(u16));
    
    BPRINTK("with value readed %x\n", *value);
    	
	return PCIBIOS_SUCCESSFUL;
}

static int dahua_read_config_dword(struct pci_dev *dev, int where, u32 *value)
{
	BPRINTK("read configure dword %x ", where);
	
    configure_read(dev, where, value, sizeof(u32));
    
	BPRINTK("with value readed %x\n", *value);
    
	return PCIBIOS_SUCCESSFUL;
}

static int dahua_write_config_byte(struct pci_dev *dev, int where, u8 value)
{
    u32 value_read = 0;

    dahua_read_config_dword(dev, where&(~0x03), &value_read);
//    printk("dahua read %x value %x\n", where, value_read);
    value_read &= (~(0xff<<(((where)&0x03)<<3)));
    value_read |= ((value<<(((where)&0x03)<<3)));
    
    #ifdef __BIG_ENDIAN
        value_read = le32_to_cpu(value_read);
    #endif
    
	BPRINTK("write configure byte %x\n", value_read);
    	
    configure_write(dev, where, &value_read, sizeof(u32));
//    printk("after werite dahua write %x ", value_read);
//    dahua_read_config_dword(dev, where&(~0x03), &value_read);
//    printk("read %x\n", value_read);

	return PCIBIOS_SUCCESSFUL;
}

static int dahua_write_config_word(struct pci_dev *dev, int where, u16 value)
{
    u32 value_read = 0;

    dahua_read_config_dword(dev, where&(~0x03), &value_read);
 //   printk("word dahua read %x value %x write %x\n", where, value_read, value);
    value_read &= (~(0xffff<<(((where )&0x03)<<3)));
    value_read |= ((value<<(((where)&0x03)<<3)));
    
    #ifdef __BIG_ENDIAN
        value_read = le32_to_cpu(value_read);
    #endif
    
    BPRINTK("write configure word %x\n", value_read);
//    printk("w after werite dahua write %x ", value_read);
    configure_write(dev, where, &value_read, sizeof(u32));
 //   dahua_read_config_dword(dev, where&(~0x03), &value_read);
  //  printk("read %x\n", value_read);

	return PCIBIOS_SUCCESSFUL;
}

static int dahua_write_config_dword(struct pci_dev *dev, int where, u32 value)
{
    u32 value_send;

    #ifdef __BIG_ENDIAN
        value_send = le32_to_cpu(value);
    #else
        value_send = (value);
    #endif
    
    BPRINTK("write %x configure word %x\n", where, value);
    	
    configure_write(dev, where, &value_send, sizeof(u32));

	return PCIBIOS_SUCCESSFUL;
}

static struct pci_ops dahua_ops = {
	dahua_read_config_byte,
	dahua_read_config_word,
	dahua_read_config_dword,
	dahua_write_config_byte,
	dahua_write_config_word,
	dahua_write_config_dword,
};

static struct timer_list serr_timer;
static struct timer_list perr_timer;

static void dahua_enable_error(unsigned long __data)
{
	switch (__data) {
	case IRQ_PCI_SERR:
		del_timer(&serr_timer);
		break;

	case IRQ_PCI_PERR:
		del_timer(&perr_timer);
		break;
	}

	enable_irq(__data);
}

/*
 * Warn on PCI errors.
 */
 /*
static void dahua_abort_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned int cmd;
	unsigned int status;

	cmd = *CSR_PCICMD;
	status = cmd >> 16;
	cmd = cmd & 0xffff;

	if (status & PCI_STATUS_REC_MASTER_ABORT) {
		printk(KERN_DEBUG "PCI: master abort: ");
		pcibios_report_status(PCI_STATUS_REC_MASTER_ABORT, 1);
		printk("\n");

		cmd |= PCI_STATUS_REC_MASTER_ABORT << 16;
	}

	if (status & PCI_STATUS_REC_TARGET_ABORT) {
		printk(KERN_DEBUG "PCI: target abort: ");
		pcibios_report_status(PCI_STATUS_SIG_TARGET_ABORT, 1);
		printk("\n");

		cmd |= PCI_STATUS_REC_TARGET_ABORT << 16;
	}

	*CSR_PCICMD = cmd;
}

static void dahua_serr_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	struct timer_list *timer = dev_id;
	unsigned int cntl;

	printk(KERN_DEBUG "PCI: system error received: ");
	pcibios_report_status(PCI_STATUS_SIG_SYSTEM_ERROR, 1);
	printk("\n");

	cntl = *CSR_SA110_CNTL & 0xffffdf07;
	*CSR_SA110_CNTL = cntl | SA110_CNTL_RXSERR;

//	 back off this interrupt
	disable_irq(irq);
	timer->expires = jiffies + HZ;
	add_timer(timer);
}

static void dahua_discard_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	printk(KERN_DEBUG "PCI: discard timer expired\n");
	*CSR_SA110_CNTL &= 0xffffde07;
}

static void dahua_dparity_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned int cmd;

	printk(KERN_DEBUG "PCI: data parity error detected: ");
	pcibios_report_status(PCI_STATUS_PARITY | PCI_STATUS_DETECTED_PARITY, 1);
	printk("\n");

	cmd = *CSR_PCICMD & 0xffff;
	*CSR_PCICMD = cmd | 1 << 24;
}

static void dahua_parity_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	struct timer_list *timer = dev_id;
	unsigned int cmd;

	printk(KERN_DEBUG "PCI: parity error detected: ");
	pcibios_report_status(PCI_STATUS_PARITY | PCI_STATUS_DETECTED_PARITY, 1);
	printk("\n");

	cmd = *CSR_PCICMD & 0xffff;
	*CSR_PCICMD = cmd | 1 << 31;

//	 back off this interrupt
	disable_irq(irq);
	timer->expires = jiffies + HZ;
	add_timer(timer);
}
*/
void __init dahua_setup_resources(struct resource **resource)
{
	struct resource *busmem, *busmempf, *busio;

	busio = kmalloc(sizeof(*busio), GFP_KERNEL | GFP_DMA);
	busmem = kmalloc(sizeof(*busmem), GFP_KERNEL | GFP_DMA );
	busmempf = kmalloc(sizeof(*busmempf), GFP_KERNEL | GFP_DMA);
	memset(busmem, 0, sizeof(*busmem));
	memset(busio, 0, sizeof(*busio));
	memset(busmempf, 0, sizeof(*busmempf));

	busio->flags = IORESOURCE_IO;
	busio->name  = "IO-PORTS";
	busmem->flags = IORESOURCE_MEM;
	busmem->name  = "Footbridge non-prefetch";
	busmempf->flags = IORESOURCE_MEM | IORESOURCE_PREFETCH;
	busmempf->name  = "Footbridge prefetch";

	allocate_resource(&iomem_resource, busmempf, 0x01000000,
			  0x82000000, 0x82ffffff, 0x01000000, NULL, NULL);
	allocate_resource(&iomem_resource, busmem, 0x01000000,
			  0x81000000, 0x81ffffff, 0x0100000, NULL, NULL);
	allocate_resource(&ioport_resource, busio, 0x01000000,
			  0x83000000, 0x83ffffff, 0x0100000, NULL, NULL);

//	resource[0] = &ioport_resource;
//	resource[1] = busmem;
//	resource[2] = busmempf;

	resource[0] = busio;
	resource[1] = busmem;
	resource[2] = busmempf;
}

void __init dahua_init(void *sysdata)
{
	unsigned int mem_size, mem_mask;
	int cfn_mode;
    unsigned long value_test[4];

	mem_size = (unsigned int)high_memory - PAGE_OFFSET;
	for (mem_mask = 0x00100000; mem_mask < 0x10000000; mem_mask <<= 1)
		if (mem_mask >= mem_size)
			break;		

	/*
	 * These registers need to be set up whether we're the
	 * central function or not.
	 */
//	*PLD_SDRAMBASEMASK    = (mem_mask - 1) & 0x0ffc0000;
//	*PLD_SDRAMBASEOFFSET  = 0;
//	*PLD_ROMBASEMASK      = 0x80000000;
//	*PLD_PLDBASEMASK      = 0;
//	*PLD_PLDBASEOFFSET    = 0;
//	*PLD_PCIADDR_EXTN     = 0;
#define PLD_CLASSREV ((unsigned long *)(CKCOREPLD_BASE + 0x4c))
	cfn_mode = __footbridge_cfn_mode();

    printk("cfn_mode value %x\n", cfn_mode);
    cfn_mode = 1;
	printk(KERN_INFO "PCI: Dahua footbridge, revision %04lX, in "
		"%s mode\n", cpu_to_le32(*PLD_CLASSREV) , cfn_mode ?
		"central function" : "addin");

/*    {
          int i;

          printk("I am print ahb-pci configure:\n");
          for (i = 0; i < 256; i = i + 4)
            {
                printk("%x ", *((unsigned long *)(CKCOREPLD_BASE + i)));
                if (((i >> 2)&0x1f) == 0xf) printk("\n");
            }
     }
*/
	if (cfn_mode) 
	   {
        #ifdef __BIG_ENDIAN
            *PLD_PCIAHB_ADDR_NP = le32_to_cpu(0xd0000001);
            *PLD_PCIAHB_ADDR_PF = le32_to_cpu(0xc0000001);
            *PLD_PCIAHB_NP_SIZE = le32_to_cpu(0xFF000000);
            *PLD_PCIAHB_PF_SIZE = le32_to_cpu(0xff000000);
        #else
            *PLD_PCIAHB_ADDR_NP = (0xd0000001);
            *PLD_PCIAHB_ADDR_PF = (0xc0000001);
            *PLD_PCIAHB_NP_SIZE = (0xFF000000);
            *PLD_PCIAHB_PF_SIZE = (0xff000000);  
        
        #endif
        
		pci_scan_bus(0, &dahua_ops, sysdata);

        #ifdef __BIG_ENDIAN
            *PLD_CPU_IMASK |= le32_to_cpu(PLD_INTA_ENABLE);
            *PLD_CPU_ISTATUS |= le32_to_cpu(PLD_INTA_ENABLE);
        #else
            *PLD_CPU_IMASK |= (PLD_INTA_ENABLE);
            *PLD_CPU_ISTATUS |= (PLD_INTA_ENABLE);        
        #endif
        
        
        printk("Scan PCI BUS mask %x status %x end\n\n", *PLD_CPU_IMASK, *PLD_CPU_ISTATUS);
        printk("Scan PCI BUS fff begin %x %x %x %x\n\n", cpu_to_le32(*PLD_PCIAHB_ADDR_NP), cpu_to_le32(*PLD_PCIAHB_ADDR_PF), \
              cpu_to_le32(*PLD_PCIAHB_NP_SIZE), cpu_to_le32(*PLD_PCIAHB_PF_SIZE));
		/*
		 * Clear any existing errors - we aren't
		 * interested in historical data...
		 */
	} else if (footbridge_cfn_mode() != 0) {
		/*
		 * If we are not compiled to accept "add-in" mode, then
		 * we are using a constant virt_to_bus translation which
		 * can not hope to cater for the way the host BIOS  has
		 * set up the machine.
		 */
		panic("PCI: this kernel is compiled for central "
			"function mode only");
	}

	/*
	 * Initialise PCI error IRQ after we've finished probing
	 */
	 /*
	request_irq(IRQ_PCI_ABORT,     dahua_abort_irq,   SA_INTERRUPT, "PCI abort",       NULL);
	request_irq(IRQ_DISCARD_TIMER, dahua_discard_irq, SA_INTERRUPT, "Discard timer",   NULL);
	request_irq(IRQ_PCI_DPERR,     dahua_dparity_irq, SA_INTERRUPT, "PCI data parity", NULL);

	init_timer(&serr_timer);
	init_timer(&perr_timer);

	serr_timer.data = IRQ_PCI_SERR;
	serr_timer.function = dahua_enable_error;
	perr_timer.data = IRQ_PCI_PERR;
	perr_timer.function = dahua_enable_error;

	request_irq(IRQ_PCI_SERR, dahua_serr_irq, SA_INTERRUPT,
		    "PCI system error", &serr_timer);
	request_irq(IRQ_PCI_PERR, dahua_parity_irq, SA_INTERRUPT,
		    "PCI parity error", &perr_timer);
*/
	//register_isa_ports(DAHUA_PCI_MEM, DAHUA_PCI_IO, 0);
}


struct hw_pci pld_pci __initdata = {
	.setup_resources	= dahua_setup_resources,
	.init			= dahua_init,
	.mem_offset		= DAHUA_PCI_MEM,
	.swizzle		= no_swizzle,
	.map_irq		= personal_server_map_irq,
};

//extern struct hw_pci ebsa285_pci;
//extern struct hw_pci cats_pci;
//extern struct hw_pci netwinder_pci;
extern struct hw_pci pld_pci;
//extern struct hw_pci ftv_pci;
//extern struct hw_pci shark_pci;
//extern struct hw_pci integrator_pci;

void pcibios_init(void)
{
	struct pci_sys_data *root;
	struct hw_pci *hw = NULL;
	
	//add by zxj
	//*(unsigned int *)(0x6000001c) = 0x1f;

	do {
#ifdef CONFIG_ARCH_EBSA285
		if (machine_is_ebsa285()) {
			hw = &ebsa285_pci;
			break;
		}
#endif
#ifdef CONFIG_ARCH_SHARK
		if (machine_is_shark()) {
			hw = &shark_pci;
			break;
		}
#endif
#ifdef CONFIG_ARCH_CATS
		if (machine_is_cats()) {
			hw = &cats_pci;
			break;
		}
#endif
#ifdef CONFIG_ARCH_NETWINDER
		if (machine_is_netwinder()) {
			hw = &netwinder_pci;
			break;
		}
#endif
//#ifdef CONFIG_ARCH_PERSONAL_SERVER
//		if (machine_is_personal_server()) 
        {
			hw = &pld_pci;
			break;
		}
//#endif
#ifdef CONFIG_ARCH_FTVPCI
		if (machine_is_ftvpci()) {
			hw = &ftv_pci;
			break;
		}
#endif
#ifdef CONFIG_ARCH_INTEGRATOR
		if (machine_is_integrator()) {
			hw = &integrator_pci;
			break;
		}
#endif
	} while (0);

	if (hw == NULL)
		return;

	root = kmalloc(sizeof(*root), GFP_KERNEL | GFP_DMA );
	if (!root)
		panic("PCI: unable to allocate root data!");

	root->hw = hw;
	root->mem_offset = hw->mem_offset;

	memset(root->resource, 0, sizeof(root->resource));

	/*
	 * Setup the resources for this bus.
	 *   resource[0] - IO ports
	 *   resource[1] - non-prefetchable memory
	 *   resource[2] - prefetchable memory
	 */
	if (root->hw->setup_resources)
		root->hw->setup_resources(root->resource);
	else {
		root->resource[0] = &ioport_resource;
		root->resource[1] = &iomem_resource;
		root->resource[2] = NULL;
	}

	/*
	 * Set up the host bridge, and scan the bus.
	 */
	 printk("PCI core init\n");
	if (root->hw->init)
    {
      root->hw->init(root);
   }

	/*
	 * Assign any unassigned resources.
	 */
	pci_assign_unassigned_resources();
	pci_fixup_irqs(root->hw->swizzle, root->hw->map_irq);

	//add 2008-08-12 for test irq
	enable_irq(CKCORE_PCI_IRQ);
}

char * pcibios_setup(char *str)
{
	/*
	if (!strcmp(str, "debug")) {
		debug_pci = 1;
		return NULL;
	}
	*/
	return str;
	
}



/**
 * pcibios_set_master - Setup device for bus mastering.
 * @dev: PCI device to be setup
 */
void pcibios_set_master(struct pci_dev *dev)
{
}

/*
 * From arch/i386/kernel/pci-i386.c:
 *
 * We need to avoid collisions with `mirrored' VGA ports
 * and other strange ISA hardware, so we always want the
 * addresses to be allocated in the 0x000-0x0ff region
 * modulo 0x400.
 *
 * Why? Because some silly external IO cards only decode
 * the low 10 bits of the IO address. The 0x00-0xff region
 * is reserved for motherboard devices that decode all 16
 * bits, so it's ok to allocate at, say, 0x2800-0x28ff,
 * but we want to try to avoid allocating at 0x2900-0x2bff
 * which might be mirrored at 0x0100-0x03ff..
 */
void pcibios_align_resource(void *data, struct resource *res,
			    unsigned long size, unsigned long align)
{
	if (res->flags & IORESOURCE_IO) {
		unsigned long start = res->start;

		if (start & 0x300)
			res->start = (start + 0x3ff) & ~0x3ff;
	}
}
/**
 * pcibios_enable_device - Enable I/O and memory.
 * @dev: PCI device to be enabled
 */
int pcibios_enable_device(struct pci_dev *dev, int mask)
{
	u16 cmd, old_cmd;
	int idx;
	struct resource *r;
	unsigned int tmp;

	unsigned char line;
		
	pci_read_config_byte(dev, 0x64, &line); printk("read 0x34 data %x\n", line);
	pci_read_config_byte(dev, 0x68, &line); printk("read 0x60 data %x\n", line);
	pci_read_config_byte(dev, 0x6c, &line); printk("read 0x34 data %x\n", line);
	pci_read_config_byte(dev, 0x70, &line); printk("read 0x34 data %x\n", line);
	
	
	pci_read_config_dword(dev, 0x10, &tmp);
	printk("0x10 ==> %x\n", tmp);
	
    pci_read_config_dword(dev, 0x14, &tmp);
	printk("0x14 ==> %x\n", tmp);
	
    pci_read_config_dword(dev, 0x18, &tmp);
	printk("0x18 ==> %x\n", tmp);

	pci_read_config_word(dev, PCI_COMMAND, &cmd);
    printk("PCI Read Config %x\n", cmd);
	old_cmd = cmd;
	for (idx = 0; idx < 6; idx++) {
		/* Only set up the requested stuff */
		if (!(mask & (1 << idx)))
			continue;

		r = dev->resource + idx;
//         printk("dev resource idx %x, resource %x\n", idx, r->start);
		if (!r->start && r->end) {
			printk(KERN_ERR "PCI: Device %s not available because"
			       " of resource collisions\n", dev->slot_name);
			return -EINVAL;
		}
		if (r->flags & IORESOURCE_IO)
			cmd |= PCI_COMMAND_IO;
		if (r->flags & IORESOURCE_MEM)
			cmd |= PCI_COMMAND_MEMORY;
	}

	/*
	 * Bridges (eg, cardbus bridges) need to be fully enabled
	 */
	if ((dev->class >> 16) == PCI_BASE_CLASS_BRIDGE)
		cmd |= PCI_COMMAND_IO | PCI_COMMAND_MEMORY;

	if (cmd != old_cmd) {
		printk("PCI: enabling device %s (%04x -> %04x)\n",
		       dev->slot_name, old_cmd, cmd);
		pci_write_config_word(dev, PCI_COMMAND, cmd);
	}
	return 0;
}


unsigned long pci_device_read(struct pci_dev *dev, unsigned long base, int where, unsigned long *value)
{
	unsigned int devfn = dev->devfn;

        {
              int i;
              unsigned long value_test[5];//value_test should have DWORD align 

			devfn -= 1 << 3;
             for (i = 0; i < 10000; i++) ;

             memset(value_test, 0, 20);
			if (devfn < PCI_DEVFN(MAX_SLOTS, 0))
            {   
                  *PLD_RDMA_SRC_ADDR = base + where;
                  *PLD_RDMA_DST_ADDR = value_test;//((unsigned long)value_test + 0x03)&(~0x03);
                  *PLD_RDMA_CTRL = (0x4 << 8) |PLD_DMA_MEMORY_READ;
                  *PLD_RDMA_CTRL |= PLD_DMA_START;
                   while (*PLD_RDMA_CTRL & PLD_DMA_START)
                  {
                           printk("After Read cmd %x %x %x Status %x\n", *PLD_RDMA_SRC_ADDR, \
                           *PLD_RDMA_DST_ADDR,   *PLD_RDMA_CTRL, *PLD_CPU_ISTATUS);
                  };//wait until the start bit cleared

                 if (*PLD_CPU_ISTATUS & 0x0200)
                  {//I set the value for dma error cause unknow data in
                          memset(value_test, 0, 20);
                           printk("RR Read Status %x\n", *PLD_CPU_ISTATUS);
                  }
                 *PLD_CPU_ISTATUS &= 0x0300;//bit 1 clear the bit, refer to datasheet
            }
            {
                *(unsigned long *)value = *((unsigned long *)((unsigned long)value_test));
            }
//           printk("PLD Read %x dev %x Status %x\n", *PLD_RDMA_SRC_ADDR, \
//                  *PLD_RDMA_DST_ADDR, *PLD_RDMA_CTRL);
              printk("PLD Read %x %x %x %x \n", value_test[0], value_test[1], value_test[2], value_test[3]);
		}

	return value;
}


