/*
 *  linux/arch/m68k/kernel/process.c
 *
 *  Copyright (C) 1995  Hamish Macdonald
 *
 *  68060 fixes by Jesper Skov
 *
 *  uClinux changes Copyright (C) 2000-2002, David McCullough <davidm@snapgear.com>
 */

/*
 * This file handles the architecture-dependent parts of process handling..
 */

#include <linux/config.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#include <linux/stddef.h>
#include <linux/unistd.h>
#include <linux/ptrace.h>
#include <linux/slab.h>
#include <linux/user.h>
#include <linux/a.out.h>
#include <linux/reboot.h>

#include <asm/uaccess.h>
#include <asm/system.h>
#include <asm/traps.h>
#include <asm/machdep.h>
#include <asm/setup.h>
#include <asm/pgtable.h>

/*
 * Initial task structure. Make this a per-architecture thing,
 * because different architectures tend to have different
 * alignment requirements and potentially different initial
 * setup.
 */
static struct fs_struct init_fs = INIT_FS;
static struct files_struct init_files = INIT_FILES;
static struct signal_struct init_signals = INIT_SIGNALS;
struct mm_struct init_mm = INIT_MM(init_mm);

union task_union init_task_union 
      __attribute__((section(".data.init_task"), aligned(KTHREAD_SIZE))) = { task: INIT_TASK(init_task_union.task) };

asmlinkage void ret_from_fork(void);


/*
 * The idle loop on an ckcore 
 */
static void default_idle(void)
{
	while(1) {
		schedule();
#ifndef NO_MM
		check_pgt_cache();
#endif
	}
}

void (*idle)(void) = default_idle;

/*
 * The idle thread. There's no useful work to be
 * done, so just try to conserve power and have a
 * low exit latency (ie sit in a loop waiting for
 * somebody to say that they'd like to reschedule)
 */
void cpu_idle(void)
{
	/* endless idle loop with no priority at all */
	init_idle();
	current->nice = 20;
	current->counter = -100;
	idle();
}

void machine_restart(char * __unused)
{
	if (mach_reset)
		mach_reset();
	for (;;);
}

void machine_halt(void)
{
	if (mach_halt)
		mach_halt();
	for (;;);
}

void machine_power_off(void)
{
	if (mach_power_off)
		mach_power_off();
	for (;;);
}

void show_regs(struct pt_regs * regs)
{
	printk("\n");
	printk("PC: %08lx  Status: %04x    %s\n",
	        regs->pc, regs->sr, print_tainted());
	printk("origin_r2: %08lx  r1: %08lx  r2: %08lx  r3: %08lx\n",
	       regs->syscallr2, regs->r1, regs->r2, regs->r3);	//syscallr1->syscallr2, Modified by Li Chunqiang 20050626, please refer asm/ptrace.h
	printk("r4: %08lx  r5: %08lx  r6: %08lx\n",
	       regs->r4, regs->r5, regs->r6);
	printk("r7: %08lx  r8: %08lx  r9: %08lx\n",
	       regs->r7, regs->r8, regs->r9);
        printk("r10: %08lx  r11: %08lx  r12: %08lx\n",
               regs->r10, regs->r11, regs->r12);  
        printk("r13: %08lx  r14: %08lx  r15: %08lx\n",
               regs->r13, regs->r14, regs->r15);  
	if (!(regs->sr & PS_S))
		printk("USP: %08lx\n", rdusp());
}

/*
 * Create a kernel thread
 */
int arch_kernel_thread(int (*fn)(void *), void * arg, unsigned long flags)
{
	long retval;
	long clone_arg = flags | CLONE_VM;
	
        set_fs (KERNEL_DS);
	__asm__ __volatile__ (
			"mov	r7, %2\n\t"      /* Save function for later */ 
			"mov	r3, r0\n\t"      /* Syscall arg[2] = sp */
			"mov	r2, %3\n\t"      /* Syscall arg[1] = flags */
			"movi   r1, %4\n\t"          /* Syscall arg[0] = __NR_clone */
			"trap   0\n\t"           /* Linux CKcore system call */
                        "cmpne	r0, r3\n\t"      /* Child or Parent */
			"bf	1f\n\t"          /* Parent ... jump */
			"mov	r2, %1\n\t"      /* Push argument */
			"jsr	r7\n\t"          /* Call fn */
			"movi	r1, %5\n\t"      /* Exit */
			"trap	0\n\t"
			"1:\n\t"
                        "mov    %0, r2"          /* return value */
		: "=r" (retval)
		: "r" (arg),
		  "r" (fn),
		  "r" (flags),
		  "i" (__NR_clone),
		  "i" (__NR_exit)
		: "r1", "r2", "r3", "r7");

	return retval;
}

void flush_thread(void)
{

}

/*
 * "ckcore_fork()".. By the time we get here, the
 * non-volatile registers have also been saved on the
 * stack. We do some ugly pointer stuff here.. (see
 * also copy_thread)
 */

asmlinkage int ckcore_fork(struct pt_regs *regs)
{
#ifdef NO_MM
	/* fork almost works,  enough to trick you into looking elsewhere :-( */
	return(-EINVAL);
#else
	return do_fork(SIGCHLD, rdusp(), regs, 0);
#endif
}

asmlinkage int ckcore_vfork(struct pt_regs *regs)
{
//	printk("vfork cmnd found\n");
	return do_fork(CLONE_VFORK | CLONE_VM | SIGCHLD, rdusp(), regs, 0);
}

asmlinkage int ckcore_clone(struct pt_regs *regs)
{
	unsigned long clone_flags;
	unsigned long newsp;

	/* syscall2 puts clone_flags in d1 and usp in d2 */
	clone_flags = regs->r2;
	newsp = regs->r3;
	if (!newsp)
		newsp = rdusp();
	return do_fork(clone_flags, newsp, regs, 0);
}

int copy_thread(int nr, unsigned long clone_flags,
		unsigned long usp, unsigned long topstk,
		struct task_struct * p, struct pt_regs * regs)
{
	struct pt_regs * childregs;
	struct switch_stack * childstack, *stack;
	unsigned long stack_offset, *retp;

	stack_offset = KTHREAD_SIZE - sizeof(struct pt_regs);
	childregs = (struct pt_regs *) ((unsigned long) p + stack_offset);

	*childregs = *regs;
	childregs->r2 = 0;

	retp = ((unsigned long *) regs);
	stack = ((struct switch_stack *) retp) - 1;

	childstack = ((struct switch_stack *) childregs) - 1;
	*childstack = *stack;
	childstack->r15 = (unsigned long)ret_from_fork;

	p->thread.usp = usp;
	p->thread.ksp = (unsigned long)childstack;
	
        return 0;
}

/* Fill in the fpu structure for a core dump.  */

int dump_fpu (struct pt_regs *regs, struct user_ckorefp_struct *fpu)
{
	return 1;
}

/*
 * fill in the user structure for a core dump..
 */
void dump_thread(struct pt_regs * regs, struct user * dump)
{
	struct switch_stack *sw;

/* changed the size calculations - should hopefully work better. lbt */
	dump->magic = CMAGIC;
	dump->start_code = 0;
	dump->start_stack = rdusp() & ~(PAGE_SIZE - 1);
	dump->u_tsize = ((unsigned long) current->mm->end_code) >> PAGE_SHIFT;
	dump->u_dsize = ((unsigned long) (current->mm->brk +
					  (PAGE_SIZE-1))) >> PAGE_SHIFT;
	dump->u_dsize -= dump->u_tsize;
	dump->u_ssize = 0;

	if (dump->start_stack < TASK_SIZE)
		dump->u_ssize = ((unsigned long) (TASK_SIZE - dump->start_stack)) >> PAGE_SHIFT;

	dump->u_ar0 = (struct user_regs_struct *)((int)&dump->regs - (int)dump);
	sw = ((struct switch_stack *)regs) - 1;
	dump->regs = *regs;
        dump->regs2 = ((struct switch_stack *)regs)[-1];
}

/*
 * sys_execve() executes a new program.
 */
 
asmlinkage int sys_execve(char *name, char **argv, char **envp)
{
       register struct pt_regs *regs __asm__ ("r5");
       __asm__ __volatile__("mov %0, r0\n\t"  \
                            : "=r" (regs)     \
                            : : "r5"); 
       (unsigned long)regs += 8; /* adjust the regs point to the system call stack frame, 2 word */
       return __sys_execve(name, argv, envp, regs);
}
asmlinkage int __sys_execve(char *name, char **argv, char **envp, struct pt_regs *regs)
{
	/* struct pt_regs *regs; */ /* commended by sunkang 2004.11.18 */
	int error;
	char * filename;
        
        lock_kernel();
	filename = getname(name);
	error = PTR_ERR(filename);
	if (IS_ERR(filename))
		goto out;
	error = do_execve(filename, argv, envp, regs);
	putname(filename);
out:
	unlock_kernel();
	return error;
}

/*
 * These bracket the sleeping functions..
 */
extern void scheduling_functions_start_here(void);
extern void scheduling_functions_end_here(void);
#define first_sched	((unsigned long) scheduling_functions_start_here)
#define last_sched	((unsigned long) scheduling_functions_end_here)

unsigned long get_wchan(struct task_struct *p)
{
	unsigned long esp, pc;
	unsigned long stack_page;
	int count = 0;
	if (!p || p == current || p->state == TASK_RUNNING)
		return 0;

	stack_page = (unsigned long)p;
	esp = p->thread.esp0;
	do {
		if (esp < stack_page+sizeof(struct task_struct) ||
		    esp >= 8184+stack_page)
			return 0;
		pc = ((unsigned long *)esp)[1]; //FIXME: There's may be error here!
		/* FIXME: This depends on the order of these functions. */
		if (pc < first_sched || pc >= last_sched)
			return pc;
		esp = *(unsigned long *) esp;
	} while (count++ < 16);
	return 0;
}
