/*
 *  linux/arch/ckcorenommu/platform/CK1000EVB/signal.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 */

/*
 *
 * 1997-12-01  Modified for POSIX.1b signals by Andreas Schwab
 *
 * mathemu support by Roman Zippel
 *  (Note: fpstate in the signal context is completly ignored for the emulator
 *         and the internal floating point format is put on stack)
 */

/*
 * ++roman (07/09/96): implemented signal stacks (specially for tosemu on
 * Atari :-) Current limitation: Only one sigstack can be active at one time.
 * If a second signal with SA_ONSTACK set arrives while working on a sigstack,
 * SA_ONSTACK is ignored. This behaviour avoids lots of trouble with nested
 * signal handlers!
 */

#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/kernel.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/wait.h>
#include <linux/ptrace.h>
#include <linux/unistd.h>
#include <linux/stddef.h>
#include <linux/highuid.h>
#include <linux/personality.h>

#include <asm/setup.h>
#include <asm/uaccess.h>
#include <asm/pgtable.h>
#include <asm/traps.h>
#include <asm/ucontext.h>

#define _BLOCKABLE (~(sigmask(SIGKILL) | sigmask(SIGSTOP)))

asmlinkage long sys_wait4(pid_t pid, unsigned int * stat_addr, int options,
			struct rusage * ru);

asmlinkage int do_signal(sigset_t *oldset, struct pt_regs *regs);

#ifndef CONFIG_COLDFIRE
const int frame_extra_sizes[16] = {
  0,
  -1, /* sizeof(((struct frame *)0)->un.fmt1), */
  sizeof(((struct frame *)0)->un.fmt2),
  sizeof(((struct frame *)0)->un.fmt3),
  sizeof(((struct frame *)0)->un.fmt4),
  -1, /* sizeof(((struct frame *)0)->un.fmt5), */
  -1, /* sizeof(((struct frame *)0)->un.fmt6), */
  sizeof(((struct frame *)0)->un.fmt7),
  -1, /* sizeof(((struct frame *)0)->un.fmt8), */
  sizeof(((struct frame *)0)->un.fmt9),
  sizeof(((struct frame *)0)->un.fmta),
  sizeof(((struct frame *)0)->un.fmtb),
  -1, /* sizeof(((struct frame *)0)->un.fmtc), */
  -1, /* sizeof(((struct frame *)0)->un.fmtd), */
  -1, /* sizeof(((struct frame *)0)->un.fmte), */
  -1, /* sizeof(((struct frame *)0)->un.fmtf), */
};
#endif

/*
 * Atomically swap in the new signal mask, and wait for a signal.
 */
asmlinkage int do_sigsuspend(struct pt_regs *regs)
{
        /* FIXME: perhaps not correct here! */
	old_sigset_t mask = regs->r2;
	sigset_t saveset;

	mask &= _BLOCKABLE;
	spin_lock_irq(&current->sigmask_lock);
	saveset = current->blocked;
	siginitset(&current->blocked, mask);
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);

	regs->r2 = -EINTR;
	while (1) {
		current->state = TASK_INTERRUPTIBLE;
		schedule();
		if (do_signal(&saveset, regs))
			return regs->r2; /* -EINTR; */
	}
}

asmlinkage int
do_rt_sigsuspend(struct pt_regs *regs)
{
        /* FIXME: perhaps not correct here! */
	sigset_t *unewset = (sigset_t *)regs->r2; 
        /* FIXME: perhaps not correct here! */
	size_t sigsetsize = (size_t)regs->r3; 
	sigset_t saveset, newset;

	/* XXX: Don't preclude handling different sized sigset_t's.  */
	if (sigsetsize != sizeof(sigset_t))
		return -EINVAL;

	if (copy_from_user(&newset, unewset, sizeof(newset)))
		return -EFAULT;
	sigdelsetmask(&newset, ~_BLOCKABLE);

	spin_lock_irq(&current->sigmask_lock);
	saveset = current->blocked;
	current->blocked = newset;
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);

	regs->r2 = -EINTR;
	while (1) {
		current->state = TASK_INTERRUPTIBLE;
		schedule();
		if (do_signal(&saveset, regs))
			return regs->r2; /* -EINTR; */
	}
}

asmlinkage int 
sys_sigaction(int sig, const struct old_sigaction *act,
	      struct old_sigaction *oact)
{
	struct k_sigaction new_ka, old_ka;
	int ret;

	if (act) {
		old_sigset_t mask;
		if (verify_area(VERIFY_READ, act, sizeof(*act)) ||
		    __get_user(new_ka.sa.sa_handler, &act->sa_handler) ||
		    __get_user(new_ka.sa.sa_restorer, &act->sa_restorer))
			return -EFAULT;
		__get_user(new_ka.sa.sa_flags, &act->sa_flags);
		__get_user(mask, &act->sa_mask);
		siginitset(&new_ka.sa.sa_mask, mask);
	}

	ret = do_sigaction(sig, act ? &new_ka : NULL, oact ? &old_ka : NULL);

	if (!ret && oact) {
		if (verify_area(VERIFY_WRITE, oact, sizeof(*oact)) ||
		    __put_user(old_ka.sa.sa_handler, &oact->sa_handler) ||
		    __put_user(old_ka.sa.sa_restorer, &oact->sa_restorer))
			return -EFAULT;
		__put_user(old_ka.sa.sa_flags, &oact->sa_flags);
		__put_user(old_ka.sa.sa_mask.sig[0], &oact->sa_mask);
	}

	return ret;
}

asmlinkage int
sys_sigaltstack(const stack_t *uss, stack_t *uoss)
{
	return do_sigaltstack(uss, uoss, rdusp());
}


/*
 * Do a signal return; undo the signal stack.
 *
 * Keep the return code on the stack quadword aligned!
 * That makes the cache flush below easier.
 */

struct sigframe
{
        /* Commented by Li Chunqiang 20050620 */
	/* char *pretcode; */	
	int sig;
	int code;
	struct sigcontext *psc;
	char retcode[4];
	unsigned long extramask[_NSIG_WORDS-1];
	struct sigcontext sc;
};

struct rt_sigframe
{
        /* Commented by Li Chunqiang 20050620 */
	/* char *pretcode; */	
	int sig;
	struct siginfo *pinfo;
	void *puc;
	char retcode[4];
	struct siginfo info;
	struct ucontext uc;
};


int copy_siginfo_to_user(siginfo_t *to, siginfo_t *from)
{
	if (!access_ok (VERIFY_WRITE, to, sizeof(siginfo_t)))
		return -EFAULT;
	if (from->si_code < 0)
		return __copy_to_user(to, from, sizeof(siginfo_t));
	else {
		int err;

		/* If you change siginfo_t structure, please be sure
		   this code is fixed accordingly.
		   It should never copy any pad contained in the structure
		   to avoid security leaks, but must copy the generic
		   3 ints plus the relevant union member.  */
		err = __put_user(from->si_signo, &to->si_signo);
		err |= __put_user(from->si_errno, &to->si_errno);
		err |= __put_user((short)from->si_code, &to->si_code);
		/* First 32bits of unions are always present.  */
		err |= __put_user(from->si_pid, &to->si_pid);
		switch (from->si_code >> 16) {
		case __SI_FAULT >> 16:
			break;
		case __SI_CHLD >> 16:
			err |= __put_user(from->si_utime, &to->si_utime);
			err |= __put_user(from->si_stime, &to->si_stime);
			err |= __put_user(from->si_status, &to->si_status);
		default:
			err |= __put_user(from->si_uid, &to->si_uid);
			break;
		/* case __SI_RT: This is not generated by the kernel as of now.  */
		}
		return err;
	}
}




static inline int
restore_sigcontext(struct pt_regs *regs, struct sigcontext *usc, void *fp,
		   int *pr2)
{
	struct sigcontext context;
	int err = 0;

	/* get previous context */
	if (copy_from_user(&context, usc, sizeof(context)))
		goto badframe;
	/* restore passed registers */
         
        regs->r1 = context.sc_r1;
        regs->r2 = context.sc_r2;
        regs->r3 = context.sc_r3;
        regs->r4 = context.sc_r4;
        regs->r5 = context.sc_r5;
        regs->r6 = context.sc_r6;
        regs->r7 = context.sc_r7;
        regs->r8 = context.sc_r8;
        regs->r9 = context.sc_r9;
        regs->r10 = context.sc_r10;
        regs->r11 = context.sc_r11;
        regs->r12 = context.sc_r12;
        regs->r13 = context.sc_r13;
        regs->r14 = context.sc_r14;
	regs->r15 = context.sc_r15;
        regs->sr = context.sc_sr;
        regs->pc = context.sc_pc;
        /*
         * syscallr1->syscallr2, Modified by Li Chunqiang 20050626, 
         * please refer asm/ptrace.h
         */
        regs->syscallr2 = -1;		
	wrusp(context.sc_usp);


	*pr2 = context.sc_r2;
	return err;

badframe:
	return 1;
}

static inline int
rt_restore_ucontext(struct pt_regs *regs, struct switch_stack *sw,
		    struct ucontext *uc, int *pr2)
{
	int temp;
	greg_t *gregs = uc->uc_mcontext.gregs;
	unsigned long usp;
	int err;

	err = __get_user(temp, &uc->uc_mcontext.version);
	if (temp != MCONTEXT_VERSION)
		goto badframe;
	/* restore passed registers */
	err |= __get_user(regs->pc, &gregs[0]);
	err |= __get_user(regs->r1, &gregs[1]);
        /* 
         * syscallr1->syscallr2, Modified by Li Chunqiang 20050626, 
         * please refer asm/ptrace.h 
         */
	err |= __get_user(regs->syscallr2, &gregs[2]);		
	err |= __get_user(regs->sr, &gregs[3]);
	err |= __get_user(regs->r2, &gregs[4]);
	err |= __get_user(regs->r3, &gregs[5]);
	err |= __get_user(regs->r4, &gregs[6]);
	err |= __get_user(regs->r5, &gregs[7]);
	err |= __get_user(regs->r6, &gregs[8]);
	err |= __get_user(regs->r7, &gregs[9]);
	err |= __get_user(regs->r8, &gregs[10]);
	err |= __get_user(regs->r9, &gregs[11]);
	err |= __get_user(regs->r10, &gregs[12]);
	err |= __get_user(regs->r11, &gregs[13]);
	err |= __get_user(regs->r12, &gregs[14]);
        err |= __get_user(regs->r13, &gregs[15]);
	err |= __get_user(regs->r14, &gregs[16]);
        err |= __get_user(regs->r15, &gregs[17]);
	err |= __get_user(usp, &gregs[18]);
	wrusp(usp);
        /* disable syscall checks */
        /*
         * syscallr1->syscallr2, Modified by Li Chunqiang 20050626,
         * please refer asm/ptrace.h
         */
	regs->syscallr2 = -1;		

	if (do_sigaltstack(&uc->uc_stack, NULL, usp) == -EFAULT)
		goto badframe;

	*pr2= regs->r2;
	return err;

badframe:
	return 1;
}

/*
 * Modified by Li Chunqiang 20050615
 * asmlinkage int do_sigreturn(unsigned long __unused)
 */
asmlinkage int do_sigreturn(struct pt_regs *regs)
{
	/* Modified by Li Chunqiang 20050615
	 * struct switch_stack *sw = (struct switch_stack *) &__unused;
	 * struct pt_regs *regs = (struct pt_regs *) (sw + 1);
         */
	unsigned long usp = rdusp();
	/* Modified by Li Chunqiang 20050617 */
	struct sigframe *frame = (struct sigframe *)usp; /* (usp - 4); */
	sigset_t set;
	int r2;

	if (verify_area(VERIFY_READ, frame, sizeof(*frame)))
		goto badframe;
	if (__get_user(set.sig[0], &frame->sc.sc_mask) ||
	    (_NSIG_WORDS > 1 &&
	     __copy_from_user(&set.sig[1], &frame->extramask,
			      sizeof(frame->extramask))))
		goto badframe;

	sigdelsetmask(&set, ~_BLOCKABLE);
	spin_lock_irq(&current->sigmask_lock);
	current->blocked = set;
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);

	if (restore_sigcontext(regs, &frame->sc, frame + 1, &r2))
		goto badframe;
	return r2;

badframe:
	force_sig(SIGSEGV, current);
	return 0;
}

/*
 * Modified by Li Chunqiang 20050615
 * asmlinkage int do_rt_sigreturn(unsigned long __unused)
 */
asmlinkage int do_rt_sigreturn(struct pt_regs *regs)
{
	/*
         * Modified by Li Chunqiang 20050615
	 * struct switch_stack *sw = (struct switch_stack *) &__unused;
         */
	struct switch_stack *sw = (struct switch_stack *) regs - 1;
	unsigned long usp = rdusp();
	/* Modified by Li Chunqiang 20050617 */
	struct rt_sigframe *frame = (struct rt_sigframe *)usp; /* (usp - 4); */
	sigset_t set;
	int r2;

	if (verify_area(VERIFY_READ, frame, sizeof(*frame)))
		goto badframe;
	if (__copy_from_user(&set, &frame->uc.uc_sigmask, sizeof(set)))
		goto badframe;

	sigdelsetmask(&set, ~_BLOCKABLE);
	spin_lock_irq(&current->sigmask_lock);
	current->blocked = set;
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);
	
	if (rt_restore_ucontext(regs, sw, &frame->uc, &r2))
		goto badframe;
	return r2;

badframe:
	force_sig(SIGSEGV, current);
	return 0;
}


static void setup_sigcontext(struct sigcontext *sc, struct pt_regs *regs,
			     unsigned long mask)
{
	sc->sc_mask = mask;
	sc->sc_usp = rdusp();
	sc->sc_r1 = regs->r1;
	sc->sc_r2 = regs->r2;
	sc->sc_r3 = regs->r3;
	sc->sc_r4 = regs->r4;
	sc->sc_r5 = regs->r5;
	sc->sc_r6 = regs->r6;
	sc->sc_r7 = regs->r7;
	sc->sc_r8 = regs->r8;
	sc->sc_r9 = regs->r9;
	sc->sc_r10 = regs->r10;
	sc->sc_r11 = regs->r11;
	sc->sc_r12 = regs->r12;
	sc->sc_r13 = regs->r13;
	sc->sc_r14 = regs->r14;
	sc->sc_r15 = regs->r15;
	sc->sc_sr = regs->sr;
	sc->sc_pc = regs->pc;
}

static inline int rt_setup_ucontext(struct ucontext *uc, struct pt_regs *regs)
{
	struct switch_stack *sw = (struct switch_stack *)regs - 1;
	greg_t *gregs = uc->uc_mcontext.gregs;
	int err = 0;

	err |= __put_user(MCONTEXT_VERSION, &uc->uc_mcontext.version);
	err |= __put_user(regs->pc, &gregs[0]);
	err |= __put_user(regs->r1, &gregs[1]);
        /*
         * syscallr1->syscallr2, Modified by Li Chunqiang 20050626,
         * please refer asm/ptrace.h
         */
	err |= __put_user(regs->syscallr2, &gregs[2]);
	err |= __put_user(regs->sr, &gregs[3]);
	err |= __put_user(regs->r2, &gregs[4]);
	err |= __put_user(regs->r3, &gregs[5]);
	err |= __put_user(regs->r4, &gregs[6]);
	err |= __put_user(regs->r5, &gregs[7]);
	err |= __put_user(regs->r6, &gregs[8]);
	err |= __put_user(regs->r7, &gregs[9]);
	err |= __put_user(regs->r8, &gregs[10]);
	err |= __put_user(regs->r9, &gregs[11]);
	err |= __put_user(regs->r10, &gregs[12]);
	err |= __put_user(regs->r11, &gregs[13]);
	err |= __put_user(regs->r12, &gregs[14]);
        err |= __put_user(regs->r13, &gregs[15]);
	err |= __put_user(regs->r14, &gregs[16]);
        err |= __put_user(regs->r15, &gregs[17]);
	err |= __put_user(rdusp(), &gregs[18]);
	
	return err;
}

static inline void push_cache (unsigned long vaddr)
{
    int  temp = 0x11;
    __asm__ __volatile__("mtcr %0, cr17": :"r"(temp));
}

static inline void *
get_sigframe(struct k_sigaction *ka, struct pt_regs *regs, size_t frame_size)
{
	unsigned long usp;

	/* Default to using normal stack.  */
	usp = rdusp();

	/* This is the X/Open sanctioned signal stack switching.  */
	if (ka->sa.sa_flags & SA_ONSTACK) {
		if (!on_sig_stack(usp))
			usp = current->sas_ss_sp + current->sas_ss_size;
	}
	return (void *)((usp - frame_size) & -8UL);
}

static void setup_frame (int sig, struct k_sigaction *ka,
			 sigset_t *set, struct pt_regs *regs)
{
	struct sigframe *frame;
	int fsize = 0;
	struct sigcontext context;
	unsigned long a5;
	int err = 0;
	unsigned long retcode;

	if (fsize < 0) {
#ifdef DEBUG
		printk ("setup_frame: Unknown frame format %#x\n",
			regs->format);
#endif
		goto give_sigsegv;
	}

	frame = get_sigframe(ka, regs, sizeof(*frame) + fsize);

	if (fsize) {
		err |= copy_to_user (frame + 1, regs + 1, fsize);
	}

	err |= __put_user((current->exec_domain
			   && current->exec_domain->signal_invmap
			   && sig < 32
			   ? current->exec_domain->signal_invmap[sig]
			   : sig),
			  &frame->sig);

	err |= __put_user(&frame->sc, &frame->psc);

	if (_NSIG_WORDS > 1)
		err |= copy_to_user(frame->extramask, &set->sig[1],
				    sizeof(frame->extramask));

	setup_sigcontext(&context, regs, set->sig[0]);
	err |= copy_to_user (&frame->sc, &context, sizeof(context));

	/* Set up to return from userspace.  */
        /* We did not support SA_RESTORER.  Modified by Li Chunqiang 20050615 */
#if 0
	if (ka->sa.sa_flags & SA_RESTORER)	
	{
		retcode = ka->sa.sa_restorer;
	}
	else
#endif
	{
		retcode = frame->retcode;
		/* movi r1, #__NR_sigreturn; trap #0 */
                /* __NR_sigreturn = 119 < 127 */
		err |= __put_user(0x6000 + (__NR_sigreturn << 4) + 1,		
				 (short *)(frame->retcode + 0));
		err |= __put_user(0x08, (short *)(frame->retcode + 2));
	}

	if (err)
		goto give_sigsegv;

	push_cache ((unsigned long) &frame->retcode);

	/* Set up registers for signal handler */
	wrusp ((unsigned long) frame);
	regs->pc = (unsigned long) ka->sa.sa_handler;
	/*
         * Added by Li Chunqiang 20050616
         * Return to user space;
         * Maybe it must return to user space(if so, we comment this line)
         */
	if (!(regs->sr & 0x80000000))		
	{
		regs->r15 = retcode;
	}

        regs->r2 = sig;
	return;	

give_sigsegv:
	if (sig == SIGSEGV)
		ka->sa.sa_handler = SIG_DFL;
	force_sig(SIGSEGV, current);
}

static void setup_rt_frame (int sig, struct k_sigaction *ka, siginfo_t *info,
			    sigset_t *set, struct pt_regs *regs)
{
	struct rt_sigframe *frame;
	int fsize = 0;
	unsigned long a5;
	int err = 0;
	unsigned long retcode;

	if (fsize < 0) {
#ifdef DEBUG
		printk ("setup_rt_frame: Unknown frame format %#x\n",
			regs->format);
#endif
		goto give_sigsegv;
	}

	frame = get_sigframe(ka, regs, sizeof(*frame));


	err |= __put_user((current->exec_domain
			   && current->exec_domain->signal_invmap
			   && sig < 32
			   ? current->exec_domain->signal_invmap[sig]
			   : sig),
			  &frame->sig);
	err |= __put_user(&frame->info, &frame->pinfo);
	err |= __put_user(&frame->uc, &frame->puc);
	err |= copy_siginfo_to_user(&frame->info, info);

	/* Create the ucontext.  */
	err |= __put_user(0, &frame->uc.uc_flags);
	err |= __put_user(0, &frame->uc.uc_link);
	err |= __put_user((void *)current->sas_ss_sp,
			  &frame->uc.uc_stack.ss_sp);
	err |= __put_user(sas_ss_flags(rdusp()),
			  &frame->uc.uc_stack.ss_flags);
	err |= __put_user(current->sas_ss_size, &frame->uc.uc_stack.ss_size);
	err |= rt_setup_ucontext(&frame->uc, regs);
	err |= copy_to_user (&frame->uc.uc_sigmask, set, sizeof(*set));

	/* Set up to return from userspace.  */
	/* We did not define SA_RESTORER. Modified by Li Chunqiang 20050615 */
#if 0
	if (ka->sa.sa_flags & SA_RESTORER)
	{
		retcode = ka->sa.sa_restorer;
	}
	else
#endif
	{
		retcode = frame->retcode;

		/* movi r1, #__NR_sigreturn; trap #0. __NR_sigreturn = 119 < 127 */
		err |= __put_user(0x6000 + (__NR_sigreturn << 4) + 1,		
				 (short *)(frame->retcode + 0));
		err |= __put_user(0x08, (short *)(frame->retcode + 2));
	}

	if (err)
		goto give_sigsegv;

	push_cache ((unsigned long) &frame->retcode);

	/* Set up registers for signal handler */
	wrusp ((unsigned long) frame);
	regs->pc = (unsigned long) ka->sa.sa_handler;
	/*
         * Added by Li Chunqiang 20050616
         * Return to user space;
         * Maybe it must return to user space(if so, we comment this line)
         */
	if (!(regs->sr & 0x80000000))
	{
		regs->r15 = retcode;
	}

        regs->r2 = sig;
adjust_stack:
	return;

give_sigsegv:
	if (sig == SIGSEGV)
		ka->sa.sa_handler = SIG_DFL;
	force_sig(SIGSEGV, current);
	goto adjust_stack;
}

static inline void
handle_restart(struct pt_regs *regs, struct k_sigaction *ka, int has_handler)
{
	switch (regs->r2) {
	case -ERESTARTNOHAND:
		if (!has_handler)
			goto do_restart;
		regs->r2 = -EINTR;
		break;

	case -ERESTARTSYS:
		if (has_handler && !(ka->sa.sa_flags & SA_RESTART)) {
			regs->r2 = -EINTR;
			break;
		}
	/* fallthrough */
	case -ERESTARTNOINTR:
	do_restart:
		regs->r2 = regs->syscallr2;
		regs->pc -= 2;
		break;
	}
}

/*
 * OK, we're invoking a handler
 */
static void
handle_signal(int sig, struct k_sigaction *ka, siginfo_t *info,
	      sigset_t *oldset, struct pt_regs *regs)
{
	/* are we from a system call? */
        /* syscallr1->syscallr2;
         * Modified by Li Chunqiang 20050626, please refer asm/ptrace.h */
	if (regs->syscallr2 >= 0)		
		/* If so, check system call restarting.. */
		handle_restart(regs, ka, 1);

	/* set up the stack frame */
	if (ka->sa.sa_flags & SA_SIGINFO)
		setup_rt_frame(sig, ka, info, oldset, regs);
	else
		setup_frame(sig, ka, oldset, regs);

	if (ka->sa.sa_flags & SA_ONESHOT)
		ka->sa.sa_handler = SIG_DFL;

	if (!(ka->sa.sa_flags & SA_NODEFER)) {
		spin_lock_irq(&current->sigmask_lock);
		sigorsets(&current->blocked,&current->blocked,&ka->sa.sa_mask);
		sigaddset(&current->blocked,sig);
		recalc_sigpending(current);
		spin_unlock_irq(&current->sigmask_lock);
	}
}

/*
 * Note that 'init' is a special process: it doesn't get signals it doesn't
 * want to handle. Thus you cannot kill init even with a SIGKILL even by
 * mistake.
 *
 * Note that we go through the signals twice: once to check the signals
 * that the kernel can handle, and then we build all the user-level signal
 * handling stack-frames in one go after that.
 */
asmlinkage int do_signal(sigset_t *oldset, struct pt_regs *regs)
{
	siginfo_t info;
	struct k_sigaction *ka;

	current->thread.esp0 = (unsigned long) regs;

	if (!oldset)
		oldset = &current->blocked;

	for (;;) {
		int signr;

		spin_lock_irq(&current->sigmask_lock);
		signr = dequeue_signal(&current->blocked, &info);
		spin_unlock_irq(&current->sigmask_lock);

		if (!signr)
			break;

		if ((current->ptrace & PT_PTRACED) && signr != SIGKILL) {
			current->exit_code = signr;
			current->state = TASK_STOPPED;
			regs->sr &= ~PS_TM;

			/* Did we come from a system call? */
                        /* 
                         * syscallr1->syscallr2, Modified by Li Chunqiang 20050626, 
                         * please refer asm/ptrace.h 
                         */
			if (regs->syscallr2 >= 0) {		
				/* Restart the system call the same way as
				   if the process were not traced.  */
				struct k_sigaction *ka =
					&current->sig->action[signr-1];
				int has_handler =
					(ka->sa.sa_handler != SIG_IGN &&
					 ka->sa.sa_handler != SIG_DFL);
				handle_restart(regs, ka, has_handler);
			}
			notify_parent(current, SIGCHLD);
			schedule();

			/* We're back.  Did the debugger cancel the sig?  */
			if (!(signr = current->exit_code)) {
			discard_frame:
			    continue;
			}
			current->exit_code = 0;

			/* The debugger continued.  Ignore SIGSTOP.  */
			if (signr == SIGSTOP)
				goto discard_frame;

			/* Update the siginfo structure.  Is this good?  */
			if (signr != info.si_signo) {
				info.si_signo = signr;
				info.si_errno = 0;
				info.si_code = SI_USER;
				info.si_pid = current->p_pptr->pid;
				info.si_uid = current->p_pptr->uid;
			}

			/* If the (new) signal is now blocked, requeue it.  */
			if (sigismember(&current->blocked, signr)) {
				send_sig_info(signr, &info, current);
				continue;
			}
		}

		ka = &current->sig->action[signr-1];
		if (ka->sa.sa_handler == SIG_IGN) {
			if (signr != SIGCHLD)
				continue;
			/* Check for SIGCHLD: it's special.  */
			while (sys_wait4(-1, NULL, WNOHANG, NULL) > 0)
				/* nothing */;
			continue;
		}

		if (ka->sa.sa_handler == SIG_DFL) {
			int exit_code = signr;

			if (current->pid == 1)
				continue;

			switch (signr) {
			case SIGCONT: case SIGCHLD:
			case SIGWINCH: case SIGURG:
				continue;

			case SIGTSTP: case SIGTTIN: case SIGTTOU:
				if (is_orphaned_pgrp(current->pgrp))
					continue;
				/* FALLTHRU */

			case SIGSTOP: {
				struct signal_struct *sig;
				current->state = TASK_STOPPED;
				current->exit_code = signr;
                                sig = current->p_pptr->sig;
                                if (sig &&
                                    !(sig->action[SIGCHLD-1].sa.sa_flags & SA_NOCLDSTOP))
                                        notify_parent(current, SIGCHLD);
				schedule();
				continue;
			}

			case SIGQUIT: case SIGILL: case SIGTRAP:
			case SIGIOT: case SIGFPE: case SIGSEGV:
			case SIGBUS: case SIGSYS: case SIGXCPU: case SIGXFSZ:
#if defined(CONFIG_NETtel) || defined(CONFIG_SNAPGEAR)
				printk("%s is core dumping at 0x%x\n", 
						current->comm, regs->pc);
				show_regs(regs);
#endif
				if (do_coredump(signr, regs))
					exit_code |= 0x80;
				/* FALLTHRU */

			default:
				sig_exit(signr, exit_code, &info);
				/* NOTREACHED */
			}
		}

		/* Whee!  Actually deliver the signal.  */
		handle_signal(signr, ka, &info, oldset, regs);
		return 1;
	}

	/* Did we come from a system call? */
        /*
         * syscallr1->syscallr2, Modified by Li Chunqiang 20050626,
         * please refer asm/ptrace.h
         */
	if (regs->syscallr2 >= 0)	
		/* Restart the system call - no handlers present */
		handle_restart(regs, NULL, 0);

	/* If we are about to discard some frame stuff we must copy
	   over the remaining frame. */
	return 0;
}

