#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <asm/uaccess.h> 

#include "./inc/avilsi.h"
#include "./inc/zlib.h"


extern u32 avi_drvsta;
env_t *avi_env;

/****** flash *********/

#define	FlashReadU16(addr)			(*((volatile unsigned short *)(addr)))
#define FlashReadU8(addr)			(*((volatile unsigned char *)(addr)))
#define	FlashWriteU16(addr, value)	(*((volatile unsigned short *)(addr)) = (value))
#define FlashWriteU8(addr,value) 	(*((volatile unsigned char *)(addr)) = (value))

#define AVI_CFG_FLASH_BASE			0x20000000
#define AVI_FLASH_PAGE_MASK 		0x1F
#define AVI_FLASH_BUFFER_SIZE 		32

/****** end flash ****/


/*	CRC	*/
#define local static
#define ZEXPORT	/* empty */

//					  buffer address	   total length
unsigned long crc32 (const unsigned char *, unsigned int);

local const uLongf crc_table[256] = {
  0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L,
  0x706af48fL, 0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L,
  0xe0d5e91eL, 0x97d2d988L, 0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L,
  0x90bf1d91L, 0x1db71064L, 0x6ab020f2L, 0xf3b97148L, 0x84be41deL,
  0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L, 0x83d385c7L, 0x136c9856L,
  0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL, 0x63066cd9L,
  0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L,
  0xa2677172L, 0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL,
  0x35b5a8faL, 0x42b2986cL, 0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L,
  0x45df5c75L, 0xdcd60dcfL, 0xabd13d59L, 0x26d930acL, 0x51de003aL,
  0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L, 0x56b3c423L, 0xcfba9599L,
  0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L, 0xb10be924L,
  0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L,
  0x01db7106L, 0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL,
  0x9fbfe4a5L, 0xe8b8d433L, 0x7807c9a2L, 0x0f00f934L, 0x9609a88eL,
  0xe10e9818L, 0x7f6a0dbbL, 0x086d3d2dL, 0x91646c97L, 0xe6635c01L,
  0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L, 0xf262004eL, 0x6c0695edL,
  0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L, 0x12b7e950L,
  0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L,
  0xfbd44c65L, 0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L,
  0x4adfa541L, 0x3dd895d7L, 0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL,
  0x346ed9fcL, 0xad678846L, 0xda60b8d0L, 0x44042d73L, 0x33031de5L,
  0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL, 0x270241aaL, 0xbe0b1010L,
  0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L, 0xce61e49fL,
  0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L,
  0x2eb40d81L, 0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L,
  0x03b6e20cL, 0x74b1d29aL, 0xead54739L, 0x9dd277afL, 0x04db2615L,
  0x73dc1683L, 0xe3630b12L, 0x94643b84L, 0x0d6d6a3eL, 0x7a6a5aa8L,
  0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L, 0x7d079eb1L, 0xf00f9344L,
  0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL, 0x806567cbL,
  0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL,
  0x67dd4accL, 0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L,
  0xd6d6a3e8L, 0xa1d1937eL, 0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L,
  0xa6bc5767L, 0x3fb506ddL, 0x48b2364bL, 0xd80d2bdaL, 0xaf0a1b4cL,
  0x36034af6L, 0x41047a60L, 0xdf60efc3L, 0xa867df55L, 0x316e8eefL,
  0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L, 0x5268e236L,
  0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL,
  0xb2bd0b28L, 0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L,
  0x2cd99e8bL, 0x5bdeae1dL, 0x9b64c2b0L, 0xec63f226L, 0x756aa39cL,
  0x026d930aL, 0x9c0906a9L, 0xeb0e363fL, 0x72076785L, 0x05005713L,
  0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL, 0x0cb61b38L, 0x92d28e9bL,
  0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L, 0xf1d4e242L,
  0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L,
  0x18b74777L, 0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL,
  0x8f659effL, 0xf862ae69L, 0x616bffd3L, 0x166ccf45L, 0xa00ae278L,
  0xd70dd2eeL, 0x4e048354L, 0x3903b3c2L, 0xa7672661L, 0xd06016f7L,
  0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL, 0xd9d65adcL, 0x40df0b66L,
  0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL, 0x30b5ffe9L,
  0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L,
  0xcdd70693L, 0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L,
  0x5d681b02L, 0x2a6f2b94L, 0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL,
  0x2d02ef8dL
};

/* ========================================================================= */
#define DO1(buf) crc = crc_table[((int)crc ^ (*buf++)) & 0xff] ^ (crc >> 8);
#define DO2(buf)  DO1(buf); DO1(buf);
#define DO4(buf)  DO2(buf); DO2(buf);
#define DO8(buf)  DO4(buf); DO4(buf);

static DECLARE_MUTEX(spare_lock);

/* ========================================================================= */
/*
uLong ZEXPORT crc32(crc, buf, len, alen)
    uLong crc;
    const Bytef *buf;
    uInt len;
*/

static uLong ZEXPORT crc32(const unsigned char *buf, u32 len)
{
	u32 crc;
	int tmp = 0;
	
    crc = 0 ^ 0xffffffffL;
    while (len >= 8)
    {
      DO8(buf);
      len -= 8;
    }
    
    if (len) do 
    {
      DO1(buf);
    } 
    while (--len);
    
    return crc ^ 0xffffffffL;
}

/* end crc */


/********** Flash write *************/

static void FlashDelay(unsigned int delay)
{
	unsigned int i = delay*10;

	while (i--);
}

/************************************************************************
 * DataToggle
 ************************************************************************/
static int DataToggle(ulong addr, u8 data)
{
	u8 oldd, curd;
	u8 pd;
	
	pd = data & 0x80;
	
	while(1)
	{
		oldd = FlashReadU8(addr);
		if( (oldd & 0x80) == pd)
			return 0;
		
		if( (oldd & 0x20) )
		{
t1:
			oldd = FlashReadU8(addr);
			if( (oldd & 0x80) == pd )
				return 0;
			else
				return -1;
		}
		else
		{
			if( oldd & 0x02 )
				goto t1;
		}	
	}
}

/*
	addr : write address
*/
static void FlashWriteToBufferCmd(u32 base_addr)
{       
	FlashWriteU8(AVI_CFG_FLASH_BASE + 0xAAA, 0xAA);
    FlashWriteU8(AVI_CFG_FLASH_BASE + 0x555, 0x55);
  	
  	/* Write Write Buffer Load Command */
  	FlashWriteU8(base_addr, 0x25);
}

/*
	addr : write address
*/
static void FlashProgramBufferToFlashCmd(u32 base_addr)
{       
  /* Transfer Buffer to Flash Command */
  /* Write Write Buffer Load Command */
  FlashWriteU8(base_addr, 0x29);
}


/*
	addr : write address
	word_count : number of words to program
	data_buf : buffer containing data to program
*/
static int FlashWriteBufferProgramPage(u32 base_addr, u32 offset,u32 word_count,u8 * data_buf)
{       
  u8 *p = data_buf;
  u32   last_loaded_addr;
  u32   current_offset;
  u32   end_offset;
  u8 wcount;

  /* Initialize variables */
  current_offset   = base_addr + offset;
  end_offset       = base_addr + offset + word_count - 1;
  last_loaded_addr = base_addr + offset;

  /* don't try with a count of zero */
  if (!word_count) 
  	return 0;

  if (((offset & AVI_FLASH_PAGE_MASK) + word_count -1) >  AVI_FLASH_PAGE_MASK)  
  	return 0;
  	
  FlashDelay(10);
  
  /* Issue Load Write Buffer Command Sequence */
  FlashWriteToBufferCmd(base_addr);

  /* Write # of locations to program */
  wcount = (u8)word_count - 1;
  FlashWriteU8(base_addr,wcount);

  /* Load Data into Buffer */
  while(current_offset <= end_offset)
  {
    /* Store last loaded address & data value (for polling) */
    last_loaded_addr = current_offset;

    /* Write Data */
    FlashWriteU8(current_offset,*p);
    current_offset++;
    p++;
  }

  /* Issue Program Buffer to Flash command */
  FlashProgramBufferToFlashCmd(base_addr);
	p--;

	if( 0 == DataToggle(last_loaded_addr,*p) )
  		return(word_count);
  	else
  		return 0;
}

/*-----------------------------------------------------------------------
 */

static void flash_erase (void)
{
	int sect;

	sect = AVILSI_BOOTENVADDR;	/* total address */
	
	FlashWriteU8(AVI_CFG_FLASH_BASE + 0xAAA, 0xAA);
	FlashWriteU8(AVI_CFG_FLASH_BASE + 0x555, 0x55);
	FlashWriteU8(AVI_CFG_FLASH_BASE + 0xAAA, 0x80);
	FlashWriteU8(AVI_CFG_FLASH_BASE + 0xAAA, 0xAA);
	FlashWriteU8(AVI_CFG_FLASH_BASE + 0x555, 0x55); 
	FlashWriteU8(sect, 0x30); 
      	
	FlashDelay(10);
	
	
	while ((FlashReadU8(sect) & 0xFF) != 0xFF)
	{
		FlashDelay(1);
	}
	      
	/*
	* Flash reset
	*/
	
	FlashWriteU8(AVI_CFG_FLASH_BASE, 0xF0);
}

/*-----------------------------------------------------------------------
 * Copy memory to flash.
 */
 
static int write_buff (unsigned char * src, u32 cnt)
{
	u32 i, lcnt;
	u8 *p = src;
	u32 offset;
	u32 b_addr;
	u32 flags;
	int ret = 0;
	
	//set_current_state(TASK_UNINTERRUPTIBLE);
	
	down(&spare_lock);
	save_flags(flags);
	cli();
	
#ifdef CONFIG_DH2004_WDT
	
	//Disable WatchDog

#endif
	
	
	b_addr = AVILSI_BOOTENVADDR;
	
	//erase
	flash_erase();
	
	//write
	offset = 0;
	while(1)
	{
		if( offset == cnt )
		{
			break;
		}
			
		if( ( cnt - offset)  > AVI_FLASH_BUFFER_SIZE )
			lcnt = AVI_FLASH_BUFFER_SIZE;
		else
			lcnt = ( cnt - offset);
					
		if ( 0 == FlashWriteBufferProgramPage(b_addr,offset,lcnt,p) )
		{
			ret = -1;
			break;
		}
		
		offset += lcnt;
		p += lcnt;	
	}/* end write sector */
	
#ifdef CONFIG_DH2004_WDT
	
	//Enable WatchDog

#endif

	restore_flags(flags);
	up(&spare_lock);
	
	return ret;
}

/********** end Flash write *************/



//get pointer
static int avi_env_getenvpos(char *name)
{
	int i, j, rc;
	char *p;
	char dstname[32];
	
	rc = NULL;
	
	if( NULL == avi_env )
		return rc;
	
	p = avi_env->data;
	
	i = strlen(name);
	while(1)
	{
		if( '\0' == *p )
			break;
			
		memset(dstname,'\0',i+4);
		
		strncpy(dstname,p,i);
		
		if( 0 == strcmp(dstname,name) )
		{
			rc = p;
			break;
		}
		
		j = strlen(p);
		p += j + 1;
	}
	
	return rc;
}

//get env value
static char * avi_env_getenv(char *name)
{
	int i, j;
	char *p;
	
	p = avi_env_getenvpos(name);
	
	if( p != NULL )
	{
		j = strlen(p);
		for( i=0; i<j; i++ )
		{
			if( '=' == *p++ )
				break;
		}
	}
	
	return p;
}

//print all env
static void avi_env_printenv(void)
{
	u32 len,end;
	char *p;
	
	if( NULL == avi_env )
		return 0;
	
	//end =0;
	
	p = avi_env->data;
	while(1)
	{
		if( '\0' == *p )
			break;
		
		printk("%s\n",p);
			
		len = strlen(p);
		p += len + 1;
	}
	
	len = p - avi_env->data;
	
	end = crc32(avi_env->data,AVILSI_BOOTENVMAXSIZE - 4);
	
	printk("total szie = %d; crc = 0x%08lx\n",len,end);
}


static int avi_env_setenv(char *name, char *val)
{
	u32 i, j;
	char *p;
	char tmp[512];
	
	if( ( NULL == name ) || ( NULL == val ) )
		return -EFAULT;
		
	p = avi_env_getenvpos(name);
	i = avi_env_getend();
	
	if( p != NULL )
	{
		//copy form uer layer and cat
		strcpy(tmp,name);
		strcat(tmp,"=");
		j = strlen(tmp);
		if ( copy_from_user(&tmp[j], val, strlen(val) + 1) )
		{
			return -EFAULT;
		}
	
		//get current command line length
		j = strlen(p);
		j++;
		
		memcpy( p, (p+j), (i-(u32)p) );
		
		memcpy( (char *)(i-j),tmp, strlen(tmp)+1 );
		
		//avi_env_printenv();
		
		//printk("\n");
		
		return 0;
	}
	
	return -EFAULT;
}

static u32 avi_env_getend(void)
{
	u32 len;
	char *p;
	
	if( NULL == avi_env )
		return 0;
		
	p = avi_env->data;
	while(1)
	{
		if( '\0' == *p )
			break;
			
		len = strlen(p);
		p += len + 1;
	}
	return (u32)p;
}

static int avi_env_save(void)
{
	//update crc
	avi_env->crc = crc32(avi_env->data,AVILSI_BOOTENVMAXSIZE - 4);

	write_buff(avi_env,AVILSI_BOOTENVMAXSIZE);
	
	return 0;
}

int avi_env_ioctl( unsigned int ioctl_num, unsigned long ioctl_param )
{
	int rc;
	char *str;
	u32 tmp;
	
	rc = 0;
	switch(ioctl_num)
	{
		case AVILSI_IOC_ENVINIT:
		{
			rc = avi_env_init();
			break;
		}
		case AVILSI_IOC_ENVEXIT:
		{
			avi_env_cleanup();
			break;
		}
		case AVILSI_IOC_GETBOOTARGS:
		case AVILSI_IOC_GETMAC:
		case AVILSI_IOC_GETIP:
		case AVILSI_IOC_GETSERIP:
		{
			if( AVILSI_IOC_GETBOOTARGS ==  ioctl_num )
				str = avi_env_getenv(AVILSI_ENV_BOOTARGS);
			else if( AVILSI_IOC_GETMAC == ioctl_num )
				str = avi_env_getenv(AVILSI_ENV_MAC);
			else if( AVILSI_IOC_GETIP == ioctl_num )
				str = avi_env_getenv(AVILSI_ENV_IP);
			else if( AVILSI_IOC_GETSERIP == ioctl_num )
				str = avi_env_getenv(AVILSI_ENV_SERIP);
				
			if( NULL == str )
			{
				rc = -EFAULT;
			}
			else
			{
				//copy result to user layer
				tmp = strlen(str);
				tmp++;
				rc = copy_to_user((void *)ioctl_param, str, tmp) ? -EFAULT : 0;
				
			}
			break;
		}
		case AVILSI_IOC_SETBOOTARGS:
		case AVILSI_IOC_SETMAC:
		case AVILSI_IOC_SETIP:
		case AVILSI_IOC_SETSERIP:
		{
			if( AVILSI_IOC_SETBOOTARGS ==  ioctl_num )
				rc = avi_env_setenv(AVILSI_ENV_BOOTARGS, (char *)ioctl_param);
			else if( AVILSI_IOC_SETMAC == ioctl_num )
				rc = avi_env_setenv(AVILSI_ENV_MAC, (char *)ioctl_param);
			else if( AVILSI_IOC_SETIP == ioctl_num )
				rc = avi_env_setenv(AVILSI_ENV_IP, (char *)ioctl_param);
			else if( AVILSI_IOC_SETSERIP == ioctl_num )
				rc = avi_env_setenv(AVILSI_ENV_SERIP, (char *)ioctl_param);
			break;
		}
		case AVILSI_IOC_SAVEENV:
		{
			rc = avi_env_save();
			break;
		}
		default:
		{
			break;
		}
	}
	
	return rc;
}

static int avi_env_init(void)
{
	char *val;
	
	if( NULL == avi_env )
	{
		avi_env = (env_t *)kmalloc(AVILSI_BOOTENVMAXSIZE,GFP_KERNEL);
		if( NULL == avi_env )
		{
			printk("avi_env_init malloc failed!");
		}
		else
		{
			//flash to ram
			memcpy(avi_env,AVILSI_BOOTENVADDR,sizeof(env_t));
			//avi_env_getend();
			//val = avi_env_getenv("ipaddr");
			//printk("val = %s\n",val);
			
			//avi_env_setenv(AVILSI_ENV_IP,"100.100.18.253");
		}
		
		
	}
	
	avi_drvsta |= AVILSI_DRV_ENV;
	
	
	return avi_env;
}

static void avi_env_cleanup(void)
{
	if( avi_env != NULL)
	{
		kfree(avi_env);
	}
	
	avi_drvsta &= ~AVILSI_DRV_ENV;
}




