/************************************************************
Copyright (C), 2007-2009, AVILSI Tech. Co., Ltd.
FileName: ds1338.c
Author: Version : Date: Zhengxj v1.0.0 2009-01-06
Description: It's a RTC Driver for the DS1338 at uClinux-2.4.x
             the rtc register to /dev/misc/rtc
Function List:
History:
NO  Date(YYYY-MM-DD)    Author  Modification
1.  2009-01-05          zxj         Create
***********************************************************/

#include <linux/config.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/i2c.h>
#include <linux/rtc.h>
#include <linux/init.h>
#include <linux/module.h> 
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/rtc.h>
#include <linux/proc_fs.h>
#include <asm/uaccess.h> 

/* DS1338 driver's version */
#define DS1338_DRV_VERSION "1.0.0"

/* DS1338 driver's  name */
#define DS1338_DRV_NAME "rtc"

/* DS1338 device's name */
#define DS1338_DEV_NAME "DS1338C"

/* DS1338 driver's debug message output*/
/* It's debug on */
#define DS1338_DEBUG

/* It's debug off */
#if 0
#undef DS1338_DEBUG
#endif

/* set the stdout to printk or null */
#ifdef DS1338_DEBUG
    #define DSMsg(fmt, args...) printk(fmt, ## args)
#else
    #define DSMsg(fmt, args...)
#endif

/* check the BCD_to_BIN or BIN_TO_BCD, if undef, then define it */
#ifndef BCD_TO_BIN
#define BCD_TO_BIN(val)     ( ( (val) & 15 ) + ( (val) >> 4 ) * 10 )
#endif

#ifndef BIN_TO_BCD
#define BIN_TO_BCD(val)     ( ( ( (val) / 10 ) << 4 ) + (val) % 10 )
#endif

/* define the DS1338 registers */
#define DS1338_REG_SECOND   ( 0x00 )    /* seconds and the bit7 is century */
#define DS1338_REG_MINUTE   ( 0x01 )    /* minutes */
#define DS1338_REG_HOUR     ( 0x02 )    /* hours; bit5 is AM/PM; bit6 is 12/24 */
#define DS1338_REG_WEEK     ( 0x03 )    /* week */
#define DS1338_REG_DAY      ( 0x04 )    /* days */
#define DS1338_REG_MONTH    ( 0x05 )    /* month */
#define DS1338_REG_YEAR     ( 0x06 )    /* year */
#define DS1338_REG_CONTROL  ( 0x07 )    /* control register */

#define DS1338_REG_MAX      ( 0x08 )    /* 8, the ds1338's registers max count */

/* define the DS1338 registers bits */
#define DS1338_BITS_CH      ( 0x80 )    /* century set */
#define DS1338_BITS_AMPM    ( 0x20 )    /* AM or PM; 0 = AM; 1 = PM */
#define DS1338_BITS_1224    ( 0x40 )    /* 12 or 24 hours; 0 = 24; 1 = 12 */
#define DS1338_BITS_RS0     ( 0x01 )    /* control register's bit0, Rate Select RS0 = 0 RS1 = 0 => 1Hz */
#define DS1338_BITS_RS1     ( 0x02 )    /* control register's bit1, Rate Select RS0 = 1 RS1 = 1 => 32.768KHz */
#define DS1338_BITS_SQWE    ( 0x10 )    /* control register's bit4, Square-Wave Enable */
#define DS1338_BITS_OSF     ( 0x20 )    /* control register's bit5, Oscillator Stop Flag */
#define DS1338_BITS_OUT     ( 0x80 )    /* control register's bit7, output control */

/* define the DS1338 century register */
#define DS1338_CENTURY_IDX  ( 0x00 )
#define DS1338_IS_CENTURY   ( 0x00 )

/* define the ds1338 device status */
#define DS1338_IS_CLOSE     ( 0x00 )    /* the /dev/misc/rtc is close */
#define DS1338_IS_OPEN      ( 0x01 )    /* means /dev/misc/rtc is in used */

/* define fix */
#define DS1338_FIX_MONTH    ( 0x01 )
#define DS1338_MIN_YEAR     ( 1970 )
#define DS1338_FIX_EPOCH    ( 1900 )
#define DS1338_FIX_YEAR     ( 100 )


/* define ds1338 register mark */
#define DS1338_MARK_SEC     ( 0x7F )
#define DS1338_MARK_MIN     ( 0x7F )
#define DS1338_MARK_HOUR    ( 0x3F )
#define DS1338_MARK_WEEK    ( 0x07 )
#define DS1338_MARK_DAY     ( 0x3F )
#define DS1338_MARK_MON     ( 0x1F )
#define DS1338_MARK_YERA    ( 0xFF )
#define DS1338_MARK_CH      ( 0x80 )

/* the ds1338 timer register count */
#define DS1338_REG_TIMECOUNTS   ( DS1338_REG_YEAR - DS1338_REG_SECOND + 1 )

/* ----------------end define ---------------------- */

/* var the 19'th */
static unsigned int s_uiEPOCH = 1900;   /* epoch status byte. it's initialize to 1970 */

/* the ds1338 driver current status */
static unsigned int s_uiDSStatus = 0;   /* bitmapped status byte. it's initialize to 0 */

/* the ds1338 driver */
static struct i2c_driver s_tDS1338Driver;

/* the i2c_client for ds1338 driver */
static struct i2c_client *s_tpDS1338C;

/* the i2c adapter id for ds1338 driver */
static unsigned int s_uiDS1338AdapterId;

/* var the days of pre month in a year */
static unsigned char s_ucDaysInMo[] = {0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

/* var the days of offset in a year */
static int s_iMOffset[] = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334};

/* i2c device id for i2c core driver probe */
static unsigned short g_usIgnore[] = { I2C_CLIENT_END };
static unsigned short g_usNormalAddr[] = { I2C_DRIVERID_DS1338C >> 1, I2C_CLIENT_END };

/* i2c device i2c_client_address_data for i2c core driver probe */
static struct i2c_client_address_data s_tAddrData = {
    normal_i2c:         g_usNormalAddr,
    normal_i2c_range:   g_usIgnore,
    probe:              g_usIgnore,
    probe_range:        g_usIgnore,
    ignore:             g_usIgnore,
    ignore_range:       g_usIgnore,
    force:              g_usIgnore,
};

/*************************************************
Function: DS1338Attach
Description: backcall for i2c core driver
Input:
Output:
Return: return i2c_attach_client's result
*************************************************/
static int DS1338Attach(struct i2c_adapter *adap, int addr, unsigned short flags, int kind)
{
    struct i2c_client *c;

    c = kmalloc(sizeof(*c), GFP_KERNEL);
    if ( NULL == c )
    {
        return -ENOMEM;
    }

    /* copy the ds1338 driver info to i2c_client */
    strcpy( c->name, DS1338_DEV_NAME );
    c->id       = s_tDS1338Driver.id;
    c->flags    = I2C_CLIENT_ALLOW_USE;
    c->addr     = addr;
    c->adapter  = adap;
    c->driver   = &s_tDS1338Driver;
    c->data     = NULL;

    return i2c_attach_client(c);
}


/*************************************************
Function: DS1338Probe
Description: i2c core driver will call this function to Probe device
Input:
Output:
Return: return i2c_probe's result
*************************************************/
static int DS1338Probe(struct i2c_adapter *adap)
{
    return i2c_probe(adap, &s_tAddrData, DS1338Attach);
}


/*************************************************
Function: DS1338Detach
Description: i2c core driver will call this function to detach client
Input:
Output:
Return: return 0
*************************************************/
static int DS1338Detach(struct i2c_client *client)
{
    i2c_detach_client(client);
    kfree(client);

    return 0;
}


/*************************************************
Function: DS1338ReadBus
Description: read ds1338 from i2c bus
Input:  pos : ds1338 start register address;
        buf : the buffer poniter of the read data from i2c bus;
        count : read count.
Output:
Return: return read count
*************************************************/
static int DS1338ReadBus(int pos, char *buf, unsigned int count)
{
    int i;
    unsigned char i2cbuf[ DS1338_REG_MAX ], ad[1];
    struct i2c_msg msgs[2] = {
        { s_tpDS1338C->addr, 0,        1, ad  },
        { s_tpDS1338C->addr, I2C_M_RD, 1, i2cbuf }
    };
    
    i = 0;

    if (NULL != s_tpDS1338C)
    {
        /* get address and count */
        ad[0] = pos;
        msgs[1].len = count;

        /* transfer */
        if (i2c_transfer(s_tpDS1338C->adapter, msgs, 2) == 2)
        {
            for (i = 0; (i < count); i++, buf++)
            {
                put_user( i2cbuf[i], buf);
            }
        }
    }

    return i;
}


/*************************************************
Function: DS1338WriteBus
Description: write ds1338 from i2c bus
Input:  pos : ds1338 start register address;
        buf : the buffer poniter of the write data from i2c bus;
        count : write count.
Output:
Return: return write count
*************************************************/
static int DS1338WriteBus(int pos, char *buf, unsigned int count)
{
    int i;
    unsigned char i2cbuf[ DS1338_REG_MAX + 1 ];
    struct i2c_msg msgs[1];

    if (NULL == s_tpDS1338C)
    {
        return -ENODEV;
    }
    
    i = 0;

    /* get address */
    i2cbuf[0] = pos;
    
    /* copy buffer */
    for (i = 0; (i < count); i++, buf++)
    {
        get_user( i2cbuf[ i +1 ], buf );
    }

    i++;
    msgs[0].addr = s_tpDS1338C->addr;
    msgs[0].flags = 0;
    msgs[0].len = i;
    msgs[0].buf = i2cbuf;

    /* transfer */
    if (i2c_transfer(s_tpDS1338C->adapter, msgs, 1) != 1)
    {
        return 0;
    }

    return(i);
}


/*************************************************
Function: DS1338Read
Description: read ds1338 by call DS1338ReadBus
Input:  fp : the device driver
        buf : the buffer poniter of the read data from i2c bus;
        count : read count;
        ptr : unused
Output:
Return: return read count
*************************************************/
static int DS1338Read(struct file *fp, char *buf, unsigned int count, loff_t *ptr)
{
    int total;
    
    if (NULL == s_tpDS1338C)
    {
        return -ENODEV;
    }
    
    total = 0;
    
    if (fp->f_pos < DS1338_REG_MAX)
    {
        if (count > (DS1338_REG_MAX - fp->f_pos))
        {
            count = DS1338_REG_MAX - fp->f_pos;
        }
        
        /* transfer */
        if (count == (DS1338ReadBus(fp->f_pos, buf, count)))
        {
            total = count;
        }
        
        fp->f_pos += total;
    }
    
    return(total);
}


/*************************************************
Function: DS1338GetNow
Description: read ds1338 by call DS1338ReadBus, get now time
Input:  rtct : the rtc_time struct
Output:
Return: 0 = sucessed; other = failed
*************************************************/
static int DS1338GetNow( struct rtc_time *rtct  )
{
    unsigned char buffer[ DS1338_REG_MAX + 1 ];
    
    if (NULL == s_tpDS1338C)
    {
        return -ENODEV;
    }
    
    if (DS1338_REG_MAX == (DS1338ReadBus( DS1338_REG_SECOND, &buffer, DS1338_REG_MAX)))
    {
        /* copy to rtc_time */
        rtct->tm_sec  = BCD_TO_BIN((buffer[DS1338_REG_SECOND] & DS1338_MARK_SEC));
        rtct->tm_min  = BCD_TO_BIN((buffer[DS1338_REG_MINUTE] & DS1338_MARK_MIN));
        rtct->tm_hour = BCD_TO_BIN((buffer[DS1338_REG_HOUR] & DS1338_MARK_HOUR));
        rtct->tm_mday = BCD_TO_BIN((buffer[DS1338_REG_DAY] & DS1338_MARK_DAY));
        rtct->tm_mon  = BCD_TO_BIN((buffer[DS1338_REG_MONTH] & DS1338_MARK_MON));
        rtct->tm_year = BCD_TO_BIN((buffer[DS1338_REG_YEAR] & DS1338_MARK_YERA));
        rtct->tm_wday = BCD_TO_BIN((buffer[DS1338_REG_WEEK] & DS1338_MARK_WEEK));
        
        /* for linux */
        rtct->tm_mon--;
        rtct->tm_yday   = s_iMOffset[rtct->tm_mon] + rtct->tm_mday - DS1338_FIX_MONTH;
        rtct->tm_isdst  = 0;
        if (DS1338_IS_CENTURY == (buffer[DS1338_CENTURY_IDX] & DS1338_MARK_CH))
        {
            rtct->tm_year += DS1338_FIX_YEAR;
        }
    }
    
    return 0;
}


/*************************************************
Function: DS1338Command
Description: unused
Input:
Output:
Return: -EINVAL
*************************************************/
static int DS1338Command(struct i2c_client *client, unsigned int cmd, void *arg)
{
    return -EINVAL;
}


/*************************************************
Function: DS1338Write
Description: write ds1338 by call DS1338WriteBus
Input:  fp : the device driver
        buf : the buffer poniter of the write data from i2c bus;
        count : write count;
        ptr : unused
Output:
Return: return write count
*************************************************/
static int DS1338Write(struct file *fp, const char *buf, unsigned int count, loff_t *ptr)
{
    int total;
    
    if (NULL == s_tpDS1338C)
    {
        return -ENODEV;
    }
    
    total = 0;
    
    if (fp->f_pos < DS1338_REG_MAX)
    {
        if (count > (DS1338_REG_MAX - fp->f_pos))
        {
            count = DS1338_REG_MAX - fp->f_pos;
        }
        
        /* transfer */
        if (( count + 1) == DS1338WriteBus(fp->f_pos, buf, count))
        {
            total = count ;
            fp->f_pos += total;
            total ++;
        }
    }
    
    return(total);
}


/*************************************************
Function: DS1338WeekDay
Description: Caculate WeekDay
Input:  y : year
        m : month
        d : day
Output:
Return: return weekday
*************************************************/
static int DS1338WeekDay(int y, int m, int d)
{
    int week;
    
    if (1 == m)
    {
        m = 13;
    }
    if (2 == m)
    {
        m = 14;
    }
    
    week = (d + 2 * m + 3 * (m + 1) / 5 + y + y / 4 - y / 100 + y / 400) % 7;
    
    if( 6 == week )
    {
        /* 6 is sunday, fix to 1 for linux */
        week = 1;
    }
    else
    {
        /* monday is 2,....Saturday is 7 */
        week += 2;
    }
    
    return (week);
}


/*************************************************
Function: DS1338IOCtl
Description: ioctl calls that are permitted to the /dev/misc/rtc interface, if
              any of the RTC drivers are enabled. 
Input:  inode :
        file :
        cmd :
        arg :
Output:
Return: return result
*************************************************/
static int DS1338IOCtl(struct inode *inode, struct file *file, unsigned int cmd,
                        unsigned long arg)
{
    struct rtc_time wtime; 

    /* no irq */
    switch (cmd) 
    {
    case RTC_AIE_OFF:
    case RTC_AIE_ON:
    case RTC_PIE_OFF:
    case RTC_PIE_ON:
    case RTC_UIE_OFF:
    case RTC_UIE_ON:
    case RTC_IRQP_READ:
    case RTC_IRQP_SET:
        {
            return -EINVAL;
        }
    };

    switch (cmd) 
    {
    case RTC_ALM_READ:
        {
            break; 
        }
    case RTC_ALM_SET:
        {
            return 0;
        }
    case RTC_RD_TIME:   /* Read the time/date from RTC */
        {
            memset(&wtime, 0, sizeof(struct rtc_time));
            DS1338GetNow(&wtime);
            break;
        }
    case RTC_SET_TIME:
        {
            unsigned char leap_yr;
            unsigned int yrs;
            unsigned char tmbuf[0x08];

            if (!capable(CAP_SYS_TIME))
            {
                return -EACCES;
            }
            
            if (copy_from_user(&wtime, (struct rtc_time*)arg, sizeof(struct rtc_time)))
            {
                return -EFAULT;
            }

            tmbuf[DS1338_REG_SECOND] = wtime.tm_sec;
            tmbuf[DS1338_REG_MINUTE] = wtime.tm_min;
            tmbuf[DS1338_REG_HOUR] = wtime.tm_hour;
            tmbuf[DS1338_REG_DAY] = wtime.tm_mday;
            //tmbuf[DS1338_REG_WEEK] = wtime.tm_wday;
            tmbuf[DS1338_REG_MONTH] = wtime.tm_mon + DS1338_FIX_MONTH;
            yrs = wtime.tm_year + DS1338_FIX_EPOCH;
            
            tmbuf[DS1338_REG_WEEK] = DS1338WeekDay(yrs, tmbuf[DS1338_REG_MONTH], tmbuf[DS1338_REG_DAY]);

            
            /* check the year > 1970 */
            if (yrs < DS1338_MIN_YEAR)
            {
                return -EINVAL;
            }

            /* check month and day */
            if ((tmbuf[DS1338_REG_MONTH] > 12) || (0 == tmbuf[DS1338_REG_DAY]))
            {
                return -EINVAL;
            }
            
            /* check leap year */
            leap_yr = ((!(yrs % 4) && (yrs % 100)) || !(yrs % 400));
            if (tmbuf[DS1338_REG_DAY] > (s_ucDaysInMo[tmbuf[ DS1338_REG_MONTH ]] \
                                        + (2 == (tmbuf[ DS1338_REG_MONTH ]) && leap_yr)))
            {
                return -EINVAL;
            }
            
            /* check hour, minute, second */
            if ((tmbuf[DS1338_REG_HOUR] >= 24) || (tmbuf[ DS1338_REG_MINUTE ] >= 60) \
                 || (tmbuf[ DS1338_REG_SECOND ] >= 60))
            {
                return -EINVAL;
            }

            /* check year */
            if ((yrs -= s_uiEPOCH) > 255)
            {
                return -EINVAL;
            }

            tmbuf[DS1338_REG_YEAR] = yrs;

            /* fix year */
            if (tmbuf[DS1338_REG_YEAR] >= 100)
            {
                tmbuf[DS1338_REG_YEAR] -= 100;
            }

            tmbuf[DS1338_REG_SECOND] = BIN_TO_BCD(tmbuf[DS1338_REG_SECOND]);
            tmbuf[DS1338_REG_MINUTE] = BIN_TO_BCD(tmbuf[DS1338_REG_MINUTE]);
            tmbuf[DS1338_REG_HOUR] = BIN_TO_BCD(tmbuf[DS1338_REG_HOUR]);
            tmbuf[DS1338_REG_DAY] = BIN_TO_BCD(tmbuf[DS1338_REG_DAY]);
            tmbuf[DS1338_REG_MONTH] = BIN_TO_BCD(tmbuf[DS1338_REG_MONTH]);
            tmbuf[DS1338_REG_YEAR] = BIN_TO_BCD(tmbuf[DS1338_REG_YEAR]);

            /* transfer */
            if ((DS1338_REG_TIMECOUNTS + 1) \
                == DS1338WriteBus(DS1338_REG_SECOND, &(tmbuf[DS1338_REG_SECOND]), DS1338_REG_TIMECOUNTS))
            {
                return 0;
            }
            else
            {
                return -EINVAL;
            }
        }
    case RTC_EPOCH_READ:
        {
            /* Read the epoch. */
            return put_user(s_uiEPOCH, (unsigned long *)arg);
        }
    case RTC_EPOCH_SET:
        {
            /* Set the epoch. */
            /* There were no RTC clocks before 1900. */
            if (arg < 1900)
            {
                return -EINVAL;
            }
                
            if (!capable(CAP_SYS_TIME))
            {
                return -EACCES;
            }

            s_uiEPOCH = arg;
            return 0;
        }
    default:
        {
            return -EINVAL;
        }
    }

    return copy_to_user((void *)arg, &wtime, sizeof wtime) ? -EFAULT : 0;
}


/*************************************************
Function: DS1338Open
Description: open calls that are permitted to the /dev/misc/rtc interface
Input:  inode :
        file :
Output:
Return: 0 : open sucessed; other : failed
*************************************************/
static int DS1338Open(struct inode *inode, struct file *file)
{
    if (s_uiDSStatus & DS1338_IS_OPEN)
    {
        return -EBUSY;
    }
    
    /* check the i2c_client */
    if (NULL == s_tpDS1338C)
    {
        printk("finded client failed!\n");
        return -ENODEV;
    }
    
    MOD_INC_USE_COUNT;
    
    /* set status to open */
    s_uiDSStatus |= DS1338_IS_OPEN;
    
    return 0;
}


/*************************************************
Function: DS1338Release
Description: Release calls that are permitted to the /dev/misc/rtc interface
Input:  inode :
        file :
Output:
Return: 0 : open sucessed; other : failed
*************************************************/
static int DS1338Release(struct inode *inode, struct file *file)
{
    if (s_uiDSStatus & DS1338_IS_OPEN)
    {
        s_uiDSStatus = DS1338_IS_CLOSE;
        MOD_DEC_USE_COUNT;
    }
    
    return 0;
}


/* DS1338 file_operations */
static struct file_operations s_tDS1338fops =
{
    owner:      THIS_MODULE, 
    read:       DS1338Read,
    write:      DS1338Write,
    ioctl:      DS1338IOCtl,
    open:       DS1338Open,
    release:    DS1338Release,
};

/* DS1338 miscdevice */
static struct miscdevice s_tDS1338Dev =
{
    RTC_MINOR,
    DS1338_DRV_NAME,
    &s_tDS1338fops
};

/* DS1338 i2c_driver */
static struct i2c_driver s_tDS1338Driver = {
    name:           DS1338_DEV_NAME,
    id:             I2C_DRIVERID_DS1338C,
    flags:          I2C_DF_NOTIFY,
    attach_adapter: DS1338Probe,
    detach_client:  DS1338Detach,
    command:        DS1338Command
};


/*************************************************
Function: DS1338Init
Description: DS1338 driver  init
Input:
Output:
Return: 0 : sucessed; other : failed
*************************************************/
static __init int DS1338Init(void)
{
    int rc;
    
    DSMsg("Add DS1338 i2c driver\n");
    
    /* add driver */
    rc = i2c_add_driver(&s_tDS1338Driver);
    if (0 != rc)
    {
        printk("Add pcf8563 i2c driver failed!\n");
        return rc;
    }
    
    DSMsg("Register DS1338 rtc device\n");
    
    /* register ds1338 device */
    rc = misc_register(&s_tDS1338Dev);
    if (0 != rc)
    {
        printk("Register pcf8563 rtc device failed\n");
        return rc;
    }
    
    DSMsg("get i2c client for rtc device\n");
    
    /* get ds1338 adapter */
    s_tpDS1338C = i2c_get_client(I2C_DRIVERID_DS1338C, s_uiDS1338AdapterId, 0);
    if (NULL == s_tpDS1338C)
    {
        printk("finded i2c client failed!\n");
        return -ENODEV;
    }
    
    s_uiDSStatus = DS1338_IS_CLOSE;
    
    /* display version infomation */
    printk("DS1338 driver %s register ok\n", DS1338_DRV_VERSION);
    
    return 0;
}


/*************************************************
Function: DS1338Exit
Description: DS1338 driver exit
Input:
Output:
Return: 0 : sucessed; other : failed
*************************************************/
static void __exit DS1338Exit(void)
{
    kfree(s_tpDS1338C);
    s_tpDS1338C = NULL;
    
    misc_deregister(&s_tDS1338Dev);
    i2c_del_driver(&s_tDS1338Driver);
}


/* register kernel */
module_init(DS1338Init);
module_exit(DS1338Exit);
MODULE_LICENSE("GPL");
