 /*
 *	DAHUA CK510 WatchDog driver for Linux 2.1.x
 *
 *	(c) Copyright 1996-1997 Alan Cox <alan@redhat.com>, All Rights Reserved.
 *				http://www.redhat.com
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *	
 *	Neither Alan Cox nor CymruNet Ltd. admit liability nor provide 
 *	warranty for any of this software. This material is provided 
 *	"AS-IS" and at no charge.	
 *
 *	(c) Copyright 1995    Alan Cox <alan@lxorguk.ukuu.org.uk>
 *
 *	Release 0.08.
 *
 *	Fixes
 *		Dave Gregorich	:	Modularisation and minor bugs
 *		Alan Cox	:	Added the watchdog ioctl() stuff
 *		Alan Cox	:	Fixed the reboot problem (as noted by
 *					Matt Crocker).
 *		Alan Cox	:	Added wdt= boot option
 *		Alan Cox	:	Cleaned up copy/user stuff
 *		Tim Hockin	:	Added insmod parameters, comment cleanup
 *					Parameterized timeout
 *		Tigran Aivazian	:	Restructured wdt_init() to handle failures
 *		Joel Becker	:	Added WDIOC_GET/SETTIMEOUT
 *		Matt Domsch	:	Added nowayout module option
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/smp_lock.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <asm/ckcore.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>

static unsigned long ck_wdt_is_open;

/*
 *	watchdog define
 */
 
#define CK_WDT_TIMER	( *(volatile u32 *)(0x70000038) )
#define CK_WDT_CW		( *(volatile u32 *)(0x7000003C) )

#define CK_WDT_CLK		CK_BUSCLK

#define CK_WDT_ON		0x01
#define CK_WDT_PREOFF	0x03
#define CK_WDT_OFF		0x02
#define CK_WDT_KNOCK	0x08

#define CK_DEF_TIMEOUT	( 0xFFFFFFFF - 2 * CK_WDT_CLK ) //2 sec

#define CK_WDT_MAXSEC	( ( 0xFFFFFFFF / CK_WDT_CLK ) - 1 )


//name		: ck_wdt_ctr_load
//para 		: sec = unit sec,set timeout
//return 	: NULL
//note		:
//author	: zxj
//date		: 2008-09-28
//history 	:
static void ck_wdt_ctr_load(unsigned int sec)
{
	//it's force to reboot system
	if( 0 == sec )
	{
		printk("reboot now\n");
		CK_WDT_TIMER = (0xFFFFEFFF);
		return;
		while(1)
		{
			//do nothing;
		}
	}
	
	if( sec > CK_WDT_MAXSEC  )
		sec = CK_WDT_MAXSEC;
	
	CK_WDT_TIMER = ( 0xFFFFFFFF - sec * CK_WDT_CLK );
		
}
 
//name		: ck_wdt_ping
//para 		: NULL
//return 	: NULL
//note		:
//author	: zxj
//date		: 2008-09-28
//history 	:
static void ck_wdt_ping(void)
{
	CK_WDT_CW |= CK_WDT_KNOCK;
}


//name		: ck_wdt_ioctl
//para 		: inode = inode of the device
//			  file = file handle to the device
//			  cmd = watchdog command
//			  arg = argument pointer
//return 	: result
//note		:
//author	: zxj
//date		: 2008-09-28
//history 	:
static int ck_wdt_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
	unsigned long arg)
{
	unsigned long tmp;
	
	if( ck_wdt_is_open )
	{
		switch(cmd)
		{
			case WDIOC_KEEPALIVE:
			{
				ck_wdt_ping();
				return 0;
			}
			
			case WDIOC_SETTIMEOUT:
			{
				ck_wdt_ctr_load(arg);
				ck_wdt_ping();
				return 0;
			}
			
			case WDIOC_GETTIMEOUT:
			{
				tmp = CK_WDT_TIMER;
				tmp = ( (0xFFFFFFFF - CK_WDT_TIMER) / CK_WDT_CLK );
				
				return put_user( tmp , (unsigned long *)arg);
			}
			
			default:
				return -ENOTTY;
		}
	}
	else
	{
		return -EIO;
	}
}

//name		: ck_wdt_open
//para 		: inode = inode of the device
//			  file = file handle to the device
//return 	: result
//note		:
//author	: zxj
//date		: 2008-09-28
//history 	:
static int ck_wdt_open(struct inode *inode, struct file *file)
{
	//check bus
	if( ck_wdt_is_open )
	{
		return -EBUSY;
	}
	else
	{
		ck_wdt_is_open = 1;
		CK_WDT_TIMER = CK_DEF_TIMEOUT;
		CK_WDT_CW = CK_WDT_ON;
	}
	
	return 0;
	
}

//name		: ck_wdt_close
//para 		: NULL
//return 	: NULL
//note		:
//author	: zxj
//date		: 2008-09-28
//history 	:
static void ck_wdt_close(void)
{
	if( ck_wdt_is_open )
	{
		//first knock the watch dog to keep alive
		CK_WDT_CW |= CK_WDT_KNOCK;
		
		//close wdt
		ck_wdt_is_open = 0;
		CK_WDT_CW = CK_WDT_PREOFF;
		CK_WDT_CW = CK_WDT_OFF;
	}
}

//name		: ck_wdt_release
//para 		: inode = inode of the device
//			  file = file handle to the device
//return 	: result
//note		:
//author	: zxj
//date		: 2008-09-28
//history 	:
static int ck_wdt_release(struct inode *inode, struct file *file)
{
	ck_wdt_close();
	return 0;
}

//name		: ck_wdt_notify_sys
//para 		: this = our notifier block
//			  code = the event being reported
//			  unused = unused
//return 	: result
//note		:
//				Our notifier is called on system shutdowns. We want to turn the card
//				off at reboot otherwise the machine will reboot again during memory
//				test or worse yet during the following fsck. This would suck, in fact
//				trust me - if it happens it does suck.
//author	: zxj
//date		: 2008-09-28
//history 	:
static int ck_wdt_notify_sys(struct notifier_block *this, unsigned long code,
	void *unused)
{
	if(code==SYS_DOWN || code==SYS_HALT)
	{
		//Turn the wdt off
		ck_wdt_close();
	}
	
	return NOTIFY_DONE;
}
 
/*
 *	Kernel Interfaces
 */
 
 
static struct file_operations ck_wdt_fops = {
	owner:		THIS_MODULE,
	llseek:		no_llseek,
	ioctl:		ck_wdt_ioctl,
	open:		ck_wdt_open,
	release:	ck_wdt_release,
};

static struct miscdevice ck_wdt_miscdev=
{
	WATCHDOG_MINOR,
	"watchdog",
	&ck_wdt_fops
};

//name		: ck_wdt_exit
//para 		: NULL
//return 	: result
//note		:
//				The WDT card needs to learn about soft shutdowns in order to
//				turn the timebomb registers off. 
//author	: zxj
//date		: 2008-09-28
//history 	: 
 
static struct notifier_block ck_wdt_notifier=
{
	ck_wdt_notify_sys,
	NULL,
	0
};


//name		: ck_wdt_exit
//para 		: NULL
//return 	: result
//note		:
//				cleanup_module
//				Unload the watchdog. You cannot do this with any file handles open.
//				If your watchdog is set to continue ticking on close and you unload
//				it, well it keeps ticking. We won't get the interrupt but the board
//				will not touch PC memory so all is fine. You just have to load a new
//				module in 60 seconds or reboot.
//author	: zxj
//date		: 2008-09-28
//history 	: 

static void __exit ck_wdt_exit(void)
{
	misc_deregister(&ck_wdt_miscdev);
	unregister_reboot_notifier(&ck_wdt_notifier);
}

//name		: ck_wdt_init
//para 		: NULL
//return 	: result
//note		:
//				Set up the WDT watchdog board. All we have to do is grab the
//				resources we require and bitch if anyone beat us to them.
//				The open() function will actually kick the board off.
//author	: zxj
//date		: 2008-09-28
//history 	:
static int __init ck_wdt_init(void)
{
	int ret;
	ret = misc_register(&ck_wdt_miscdev);
	if (ret) {
		printk(KERN_ERR "ck510wdt: can't misc_register on minor=%d\n", WATCHDOG_MINOR);
		goto out;
	}
	
	ret = register_reboot_notifier(&ck_wdt_notifier);
	if(ret) {
		printk(KERN_ERR "ck510wdt: can't register reboot notifier (err=%d)\n", ret);
		goto outmisc;
	}

	ck_wdt_is_open = 0;
	ret = 0;
	printk(KERN_INFO "CK510 watchdog driver 0.01!\n");
out:
	return ret;

outmisc:
	misc_deregister(&ck_wdt_miscdev);
	goto out;
}

module_init(ck_wdt_init);
module_exit(ck_wdt_exit);

MODULE_AUTHOR("Zxj");
MODULE_DESCRIPTION("Driver for CK510 watchdog");
MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;
