/*
 * MTD map driver for AMD compatible flash chips (non-CFI)
 *
 * Author: Jonas Holmberg <jonas.holmberg@axis.com>
 *
 * $Id: amd_flash.c,v 1.1.1.1 2005/01/17 01:39:56 licq Exp $
 *
 * Copyright (c) 2001 Axis Communications AB
 *
 * This file is under GPL.
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/mtd/map.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/flashchip.h>

#include "./spi.h"
#include "./spi_flash.h"

#define DEVICE_TYPE_X8	(8 / 8)
#define DEVICE_TYPE_X16	(16 / 8)
#define DEVICE_TYPE_X32	(32 / 8)

struct spi_flash_private 
{
	int device_type;	
	int interleave;	
	int numchips;	
	unsigned long chipshift;
//	const char *im_name;
	struct flchip chips[0];
};

struct spi_flash_info {
	const __u16 mfr_id;
	const __u16 dev_id;
	const char *name;
	const u_long size;
	const struct mtd_partition partition_info[ 4 ];
};



static int spi_flash_read(struct mtd_info *, loff_t, size_t, size_t *,
			  u_char *);
static int spi_flash_write(struct mtd_info *, loff_t, size_t, size_t *,
			   const u_char *);
static int spi_flash_erase(struct mtd_info *, struct erase_info *);
static void spi_flash_sync(struct mtd_info *);
static int spi_flash_suspend(struct mtd_info *);
static void spi_flash_resume(struct mtd_info *);
static void spi_flash_destroy(struct mtd_info *);
static struct mtd_info *spi_flash_probe(struct map_info *map);


static struct mtd_chip_driver spi_flash_chipdrv = {
	probe: spi_flash_probe,
	destroy: spi_flash_destroy,
	name: "dh5k__spi_flash",
	module: THIS_MODULE
};


/* register r/w */
static void Write_SPI_Reg( __u32 ctr_num, __u32 reg_offset, __u32 val )
{
    __u32 reg_addr;
    if( 0 == ctr_num )
    {
        reg_addr = SPI0_BASE_ADDR + reg_offset;
    }
    else ( 1 == ctr_num )
    {
        reg_addr = SPI1_BASE_ADDR + reg_offset;
    }
    
    *(volatile __u32 *)reg_addr = val;
}

static __u32 void Read_SPI_Reg( __u32 ctr_num, __u32 reg_offset )
{
    unsigned int reg_addr;
    
    if( 0 == ctr_num )
    {
        reg_addr = SPI0_BASE_ADDR + reg_offset;
    }
    else( 1 == ctr_num )
    {
        reg_addr = SPI1_BASE_ADDR + reg_offset;
    }
    
    
    return *(volatile unsigned int *)reg_addr;
}

/* spi opration */
void set_spi_dev_cs( __u32 ctr_num, __u32 dev_cs )
{
    unsigned int filter = 0x01;
    unsigned int reg_temp;
    if(dev_cs > 3 || dev_cs < 0)
    {
        printk("set slave device out of range\n");
        return;
    }
    reg_temp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    reg_temp &= ~((filter) << (8+dev_cs));
	Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,reg_temp);
}

void clear_spi_dev_cs( __u32 ctr_num, __u32 dev_cs )
{
    unsigned int filter = 0x01;
    unsigned int reg_temp;
    if(dev_cs > 3 || dev_cs < 0)
    {
        printk("set slave device out of range\n");
        return;
    }
    reg_temp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    reg_temp |= ((filter) << (8+dev_cs));
	Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,reg_temp);
}

static void set_spi_mode(unsigned int ctr_num,int mode)
{
    unsigned int reg_temp;
    if(mode == SPI_FLASH_MODE)
    {
        reg_temp = Read_SPI_Reg(ctr_num,SPI_MODECONF_OFFSET);
        reg_temp |= 1 << 8;
        Write_SPI_Reg(ctr_num,SPI_MODECONF_OFFSET,reg_temp);

    }else if(mode == SPI_NORMAL_MODE)
    {
        reg_temp = Read_SPI_Reg(ctr_num,SPI_MODECONF_OFFSET);
        reg_temp &= ~(1 << 8);
        Write_SPI_Reg(ctr_num,SPI_MODECONF_OFFSET,reg_temp);
 
    }else
    {
        printk("set spi mode err!\n");
    }
    
}
static void spi_init(void)
{
	unsigned int modecfg = 0;
    unsigned int irqcfg = 0;
    unsigned int contrlcfg = 0;
	modecfg = 	(PHASE_CLK << 31) +
				(BITS << 24) +
				(CLOCK_DELAY_TIMES << 16) +
				(RX_DELAY << 15) +
				(CPHA << 14) +
				(CPOL << 13) +
				(LSB << 12) +
				(PCK_MOD << 10) +
				(HOLD_EN << 9) +
				(FLASH_MOD << 8) +
				(TX_DELAY << 1) + 1;
	irqcfg	=  (TXDFFI_LEVEL <<16) + (RXDFFI_LEVEL << 8) + (COPMLETEIE << 2) + ( 1UL << 1) +1;
    contrlcfg = (REG_FF_ST_EN << 26) +
				(REG_CRC_EN << 25) +
			    (REG_BMULT << 24) +
			    (CR_CS << 8) + (DIRECTION << 1) + (STARTED);


    /*init the 1th spi*/
    Write_SPI_Reg(SPI1TH,SPI_MODECONF_OFFSET,modecfg);
    Write_SPI_Reg(SPI1TH,SPI_INTCONF_OFFSET,irqcfg);
    Write_SPI_Reg(SPI1TH,SPI_TC_OFFSET,TRANS_CNT);
    Write_SPI_Reg(SPI1TH,SPI_BOUND_OFFSET,BAUD);
    Write_SPI_Reg(SPI1TH,SPI_CTRL_OFFSET,contrlcfg);
    
	/*init the 2th spi*/

    Write_SPI_Reg(SPI2TH,SPI_MODECONF_OFFSET,modecfg);
    Write_SPI_Reg(SPI2TH,SPI_INTCONF_OFFSET,irqcfg);
    Write_SPI_Reg(SPI2TH,SPI_TC_OFFSET,TRANS_CNT);
    Write_SPI_Reg(SPI2TH,SPI_BOUND_OFFSET,BAUD);
    Write_SPI_Reg(SPI2TH,SPI_CTRL_OFFSET,contrlcfg);	  
}


static u32 spi_check_complete( u8 ctr_num, u32 u32TimeOut )
{
    u32 ret;
    u32 timeout;
    
    ret = 0;
    timeout = 0;
    
    while ( Read_SPI_Reg( ctr_num, SPI_STATE_OFFSET )  & ( 0x1 << 6) )
	{
		 timeout ++;
		 
		 if( timeout > u32TimeOut )
		 	{
                Write_SPI_Reg( ctr_num,SPI_STATE_OFFSET, 0 );
		 		printk("time out %s function! \n",__FUNCTION__);
                ret = -1;
		 		break;
		 	}
	}
	
	return ret;
    
}

static int spi_normal_mode_send(unsigned char ctr_num, unsigned int *tbuf, unsigned int tcnt )
{
	unsigned int i,tmp,ret = 0;
	unsigned int cnt;
	unsigned int timeout = 0;
	unsigned int *p = NULL;
	
	/* check parameter */
	if ( ( NULL == tbuf ) || ( 0 == tcnt ))
	{
		return -1;
	}
	p = tbuf;
	/* set the spi mode to normal */
    set_spi_mode(ctr_num,SPI_NORMAL_MODE);
	cnt = tcnt >> 2;
	if( tcnt % 4 )
	{
		cnt++;
	}
	
    for ( i = 0; i < cnt; i ++ )
    {
        Write_SPI_Reg(ctr_num,SPI_TXDB_OFFSET,*p++);
    }
    Write_SPI_Reg(ctr_num,SPI_TC_OFFSET,tcnt - 1);

    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp |= 0x03;
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);
    
    ret = spi_check_complete( ctr_num, 0xffffff );

    tmp = Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET);
    tmp |= (1UL << 20);;
    Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,tmp);
    
	//SPI_STR(spi_cs) |= (1UL << 20);

	/* set the spi mode from normal to flash */
	//SPI_MDR(spi_cs) |= ( 1 << AVI_SPI_MDR_FMODE );
	//set_spi_mode(ctr_num,SPI_FLASH_MODE);
	return ret;
}

static int spi_normal_mode_recive( unsigned char ctr_num, unsigned int *rbuf, unsigned int rcnt )
{
    unsigned int i,ret = 0;
	unsigned int tmp;
	unsigned int cnt;
	unsigned int timeout = 0;
	unsigned int *p = NULL;
	timeout = 0;	
	
	if ( (NULL == rbuf) || ( 0 == rcnt ) )
	{
		return 0;
	}
	/* set the spi mode  normal */
	set_spi_mode(ctr_num,SPI_NORMAL_MODE);
	Write_SPI_Reg(ctr_num,SPI_TC_OFFSET,rcnt - 1);
    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp &= SPI_CMDLEN_MASK;
	tmp |= 1UL;
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);

    p = rbuf;

    cnt = rcnt >> 2;
	if (  rcnt % 0x04  )
    {
        cnt ++;
    }
    
    ret = spi_check_complete( ctr_num, 0xffffff );

    for ( i = 0 ; i < cnt; i ++ )
    {
        *p ++ = Read_SPI_Reg(ctr_num,SPI_RXDB_OFFSET);
    }
    Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET)|(1UL << 20));

	return ret;
}


static int spi_flash_mode_send_for_recv( u8 ctr_num, u32 cmd, u32 cmdcnt, u32 *recvbuf, u32 recvcnt )
{
    unsigned int i, ret = 0;
	unsigned int tmp;
	unsigned int cnt;
	
    unsigned char *pu8 = NULL;
    

	/* check input param */
	if( ctr_num > 1 || ctr_num < 0 )
    {
        printk("set spi contrl num out of range\n");
        return -1;
    }
    
	if ( ( NULL == cmd ) || ( 0 == cmdcnt ) )
	{
        printk("please check argument put in!\n");
        return -1;
	}
    
    if( ( 0 == recvcnt ) || ( NULL == recvbuf ) || ( recvcnt > SPI_READ_FIFO_MAX ) )
    {  
        printk("please check argument put in!\n");
        return -1;
    }
    
    set_spi_mode( ctr_num,SPI_FLASH_MODE );
    

	/* send the command to the fifo */
    Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, cmd );
    
    /* send the recv count */
    Write_SPI_Reg( ctr_num, SPI_TC_OFFSET, ( recvcnt - 1 ) );
 
    /* set ctrl */
    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp &= SPI_CMDLEN_MASK;
	tmp |= ( cmdcnt - 1 ) << 16;
	tmp |= 1UL;
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);
    	
    cnt = recvcnt >> 2;
    pu8 = ( u8 *)( recvbuf );

	ret = spi_check_complete( ctr_num, 0xffffff );
	
	for ( i = 0; i < cnt; i ++ )
	{
	     tmp = Read_SPI_Reg( ctr_num, SPI_RXDB_OFFSET );
        *pu8 ++ = ( ( tmp >> 24 ) & 0xFF );
        *pu8 ++ = ( ( tmp >> 16 ) & 0xFF );
        *pu8 ++ = ( ( tmp >> 8 ) & 0xFF );
        *pu8 ++ = ( ( tmp >> 0 ) & 0xFF );
	}
    
    /* read the left */
    cnt = recvcnt & 0x03;
    
    if ( cnt )
    {
        tmp = Read_SPI_Reg( ctr_num, SPI_RXDB_OFFSET );
        
        for ( i = 0; i < cnt; i ++ )
        {
            *pu8 ++ = ( ( tmp >> ( ( 3 - i ) * 8 ) ) & 0xFF );
        }
    }
	
	
	/* clear status */
    Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET)|(1UL << 20));
    
    
	return ret;
}

/*
    send to flash 
    
    8       24      32      32
    cmd     addr    dat0    dat1
*/
static int spi_flash_mode_send_cmd_data( u8 ctr_num, u32 cmd, u32 addr, u32 *psendbuf, u32 sendcnt )
{
    u32 i, tmp, cnt;
	u32 ret;
	u8 *pu8 = NULL;
	
    /* check input param */
    if ( ( NULL == psendbuf ) || ( 0 == sendcnt ) )
    {
        return -1;
    }
    
    /* set flash mode */
    set_spi_mode( ctr_num, SPI_FLASH_MODE );
    
    /* push command */
    tmp = ( ( cmd << 24 ) & 0xFF000000 );
    tmp |= ( addr & 0xFFFFFF );
    
    Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, tmp );
    
    
    
    /* ΪSPIģflash ģʽ£ȷ͵Ǹ8λбֽڽ
        ֤ǰС˸ʽд뵽flashȥCPUֽͻ
     */
    cnt = ( sendcnt >> 2 );
    pu8 = ( u8 * )( psendbuf );
    
    for ( i = 0; i < cnt; i ++ )
    {
        tmp = ( ( *pu8++ ) << 24 );
        tmp |= ( ( *pu8++ ) << 16 );
        tmp |= ( ( *pu8++ ) << 8 );
        tmp |= ( ( *pu8++ ) << 0 );
        
        Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, tmp );
    }
    
    /* left */
    cnt = ( sendcnt & 0x03 );
    tmp = 0;
    
    if ( cnt )
    {
        for ( i = 0; i < cnt; i ++ )
        {
            tmp |= ( *pu8++ ) << ( ( 3 - i ) * 8 );
        }
        
        Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, tmp );
    }
    

	Write_SPI_Reg(ctr_num,SPI_TC_OFFSET, ( sendcnt - 1 ) );

    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp &= SPI_CMDLEN_MASK;
	tmp |= ( ( ( 4 - 1 ) << 16 ) + 0x3 );
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);

    /* wait for complete */
    ret = spi_check_complete( ctr_num, 0xffffff );

    /* clear status */
	Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET)|(1UL << 20));
	
	return ret;
}


static int spi_flash_mode_send_cmd_addr( u8 ctr_num, u32 cmd, u32 addr )
{
    u32 tmp, ret = 0;
	
    set_spi_mode(ctr_num,SPI_FLASH_MODE);
    
    tmp = ( ( cmd << 24 ) & 0xFF000000 );
    tmp |= ( addr & 0xFFFFFF );
    
    printk("tmp = %#x\n", tmp );

    Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, tmp );
    
	Write_SPI_Reg( ctr_num, SPI_TC_OFFSET, 3 - 1 );

    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp &= SPI_CMDLEN_MASK;
	tmp |= ( ( ( 1 - 1 ) << 16 ) + 0x3 );
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);

    ret = spi_check_complete( ctr_num, 0xffffff );

	Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET)|(1UL << 20));
	
	return ret;
}



/* function */

static inline int flash_is_busy( struct map_info *map, unsigned long addr, int interleave )
{
    u32 timeout, cmd, buf;
    int ret;
    
    /* check status */
    timeout = 0;
    ret = 0;
    
    while( 1 )
    {
        timeout ++;
        
        cmd = CMD_STATUS_READ << 24;
        spi_flash_mode_send_for_recv( SPI1TH, cmd, 1, &buf, 1 );
        
        if ( 0 == ( buf & 0x1 ) )
        {
            break;
        }
		
		if( timeout > 0xFFFFF )
		{
		    ret = -1;
			printk("time out %s function! status = %#x \n",__FUNCTION__, buf );
			
			break;
		}
    }
    
    return ret;
}



static int spi_flash_unlock(struct mtd_info *mtd, loff_t ofs, size_t len)
{
	return 0;
}

static int spi_flash_lock(struct mtd_info *mtd, loff_t ofs, size_t len)
{
	return 0;
}


static void  spi_flash_read_device_id( __u32 *mf_id, __u32 *dev_id )
{ 
	u32 buf = 0,cmd = 0, readdata = 0;
    u16 devid;
    
    set_spi_dev_cs( SPI1TH, W25Q64CV_ADDR );
    
	cmd = CMD_READ_ID << 24;
	spi_flash_mode_send_for_recv(SPI1TH, cmd,1,&buf, 3);
	
	*mf_id = buf & 0xFF;
	*dev_id = (buf >> 8) & 0xFFFF;
    
    clear_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);
}

void spi_nand_flash_read_id(struct mtd_info *mtd,int *mf_id, int *dev_id)
{
    w25q64cv_read_device_id(mf_id,dev_id);
    return;
}


/*
 * Reads JEDEC manufacturer ID and device ID and returns the index of the first
 * matching table entry (-1 if not found or alias for already found chip).
 */ 
static int probe_new_chip(struct mtd_info *mtd, __u32 base,
			  struct flchip *chips,
			  struct spi_flash_private *private,
			  const struct spi_flash_info *table, int table_size)
{
	__u32 mfr_id;
	__u32 dev_id;
	struct map_info *map = mtd->priv;
	struct spi_flash_private temp;
	int i;

	temp.device_type = DEVICE_TYPE_X8;	// Assume X16 (FIXME)
	temp.interleave = 1;
	map->fldrv_priv = &temp;

	/* Enter autoselect mode. */
	spi_flash_read_device_id( &mfr_id, &dev_id );

	for (i = 0; i < table_size; i++) 
	{
		if ( ( mfr_id == table[i].mfr_id ) && ( dev_id == table[i].dev_id ) ) 
		{
			printk("%s: Found %d x %ldMiB %s at 0x%x\n", map->name,
			       temp.interleave, (table[i].size)/(1024*1024),
			       table[i].name, base);

			mtd->size += table[i].size * temp.interleave;
			mtd->numeraseregions += table[i].numeraseregions;

			break;
		}
	}

	/* Exit autoselect mode. */
	send_cmd(map, base, CMD_RESET_DATA);

	if (i == table_size) {
		printk(KERN_DEBUG "%s: unknown flash device at 0x%x, "
		       "mfr id 0x%x, dev id 0x%x\n", map->name,
		       base, mfr_id, dev_id);
		map->fldrv_priv = NULL;

		return -1;
	}

	private->device_type = temp.device_type;
	private->interleave = temp.interleave;

	return i;
}

#if 0	  
	{ name: "w25q64cv spi nand for userfs",
		  offset: 2*1024*1024,
		  size: 6*1024*1024 }
#endif
    { name: "user",
		  offset: 2*1024*1024,
		  size: 1*1024*1024 },/*1M*/
	{ name: "config",
		  offset: 3*1024*1024,
		  size: 512*1024 },	/*512K*/  
    { name: "backup",
		  offset: 3584*1024,
		  size: 512*1024 }	/*512K*/ 


static struct mtd_info *spi_flash_probe( struct map_info *map )
{
	/* Keep this table on the stack so that it gets deallocated after the
	 * probe is done.
	 */
	const struct spi_flash_info table[] = {
	{
		mfr_id: MXIC,
		dev_id: 0x1620,
		name: "MXIC 4MB",
		size: 0x00800000,
		partition_info: {
			{ name : "kernel",  offset: 0x000000, size: 0x200000 },
			{ name : "user",    offset: 0x200000, size: 0x100000 },
			{ name : "config",  offset: 0x300000, size: 0x80000 },
			{ name : "backup",  offset: 0x380000, size: 0x80000 }
		}
	},
	
	{
		mfr_id: WINBOND,
		dev_id: 0x4017,
		name: "AMD WINBOND 8MB",
		size: 0x00200000,
		partition_info: {
			{ name : "kernel",  offset: 0x000000, size: 0x200000 },
			{ name : "user",    offset: 0x200000, size: 0x100000 },
			{ name : "config",  offset: 0x300000, size: 0x80000 },
			{ name : "backup",  offset: 0x380000, size: 0x80000 }
		}
	}
	};

	struct mtd_info *mtd;
	struct flchip chips[MAX_AMD_CHIPS];
	int table_pos[MAX_AMD_CHIPS];
	struct spi_flash_private temp;
	struct spi_flash_private *private;
	u_long size;
	unsigned long base;
	int i;
	int reg_idx;
	int offset;

	mtd = (struct mtd_info*)kmalloc(sizeof(*mtd), GFP_KERNEL);
	if (!mtd) {
		printk(KERN_WARNING
		       "%s: kmalloc failed for info structure\n", map->name);
		return NULL;
	}
	memset(mtd, 0, sizeof(*mtd));
	mtd->priv = map;

	memset(&temp, 0, sizeof(temp));

	printk("%s: Probing for dh5k spi flash...\n", map->name);

	if ((table_pos[0] = probe_new_chip(mtd, 0, NULL, &temp, table,
					   sizeof(table)/sizeof(table[0])))
	    == -1) {
		printk(KERN_WARNING
		       "%s: Found no AMD compatible device at location zero\n",
		       map->name);
		kfree(mtd);

		return NULL;
	}

	chips[0].start = 0;
	chips[0].state = FL_READY;
	chips[0].mutex = &chips[0]._spinlock;
	temp.numchips = 1;
	for (size = mtd->size; size > 1; size >>= 1) {
		temp.chipshift++;
	}
	switch (temp.interleave) {
		case 2:
			temp.chipshift += 1;
			break;
		case 4:
			temp.chipshift += 2;
			break;
	}

	/* Find out if there are any more chips in the map. */
	for (base = (1 << temp.chipshift);
	     base < map->size;
	     base += (1 << temp.chipshift)) {
	     	int numchips = temp.numchips;
		table_pos[numchips] = probe_new_chip(mtd, base, chips,
			&temp, table, sizeof(table)/sizeof(table[0]));
	}

	mtd->eraseregions = kmalloc(sizeof(struct mtd_erase_region_info) *
				    mtd->numeraseregions, GFP_KERNEL);
	if (!mtd->eraseregions) { 
		printk(KERN_WARNING "%s: Failed to allocate "
		       "memory for MTD erase region info\n", map->name);
		kfree(mtd);
		map->fldrv_priv = NULL;
		return 0;
	}

	reg_idx = 0;
	offset = 0;
	for (i = 0; i < temp.numchips; i++) {
		int dev_size;
		int j;

		dev_size = 0;
		for (j = 0; j < table[table_pos[i]].numeraseregions; j++) {
			mtd->eraseregions[reg_idx].offset = offset +
				(table[table_pos[i]].regions[j].offset *
				 temp.interleave);
			mtd->eraseregions[reg_idx].erasesize =
				table[table_pos[i]].regions[j].erasesize *
				temp.interleave;
			mtd->eraseregions[reg_idx].numblocks =
				table[table_pos[i]].regions[j].numblocks;
			if (mtd->erasesize <
			    mtd->eraseregions[reg_idx].erasesize) {
				mtd->erasesize =
					mtd->eraseregions[reg_idx].erasesize;
			}
			dev_size += mtd->eraseregions[reg_idx].erasesize *
				    mtd->eraseregions[reg_idx].numblocks;
			reg_idx++;
		}
		offset += dev_size;
	}
	mtd->type = MTD_NORFLASH;
	mtd->flags = MTD_CAP_NORFLASH;
	mtd->name = map->name;
	mtd->erase = amd_flash_erase;	
	mtd->read = amd_flash_read;	
	mtd->write = amd_flash_write;	
	mtd->sync = amd_flash_sync;	
	mtd->suspend = amd_flash_suspend;	
	mtd->resume = amd_flash_resume;	
	mtd->lock = amd_flash_lock;
	mtd->unlock = amd_flash_unlock;

	private = kmalloc(sizeof(*private) + (sizeof(struct flchip) *
					      temp.numchips), GFP_KERNEL);
	if (!private) {
		printk(KERN_WARNING
		       "%s: kmalloc failed for private structure\n", map->name);
		kfree(mtd);
		map->fldrv_priv = NULL;
		return NULL;
	}
	memcpy(private, &temp, sizeof(temp));
	memcpy(private->chips, chips,
	       sizeof(struct flchip) * private->numchips);
	for (i = 0; i < private->numchips; i++) {
		init_waitqueue_head(&private->chips[i].wq);
		spin_lock_init(&private->chips[i]._spinlock);
	}

	map->fldrv_priv = private;

	map->fldrv = &amd_flash_chipdrv;
	MOD_INC_USE_COUNT;

	return mtd;
}



static inline int read_one_chip(struct map_info *map, struct flchip *chip,
			       loff_t adr, size_t len, u_char *buf)
{
	DECLARE_WAITQUEUE(wait, current);
	unsigned long timeo = jiffies + HZ;

retry:
	spin_lock_bh(chip->mutex);

	if (chip->state != FL_READY){
		printk(KERN_INFO "%s: waiting for chip to read, state = %d\n",
		       map->name, chip->state);
		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
                
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);

		if(signal_pending(current)) {
			return -EINTR;
		}

		timeo = jiffies + HZ;

		goto retry;
	}	

	adr += chip->start;

	chip->state = FL_READY;

	map->copy_from(map, buf, adr, len);

	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);

	return 0;
}



static int amd_flash_read(struct mtd_info *mtd, loff_t from, size_t len,
			  size_t *retlen, u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct amd_flash_private *private = map->fldrv_priv;
	unsigned long ofs;
	int chipnum;
	int ret = 0;

	if ((from + len) > mtd->size) {
		printk(KERN_WARNING "%s: read request past end of device "
		       "(0x%lx)\n", map->name, (unsigned long)from + len);

		return -EINVAL;
	}

	/* Offset within the first chip that the first read should start. */
	chipnum = (from >> private->chipshift);
	ofs = from - (chipnum <<  private->chipshift);

	*retlen = 0;

	while (len) {
		unsigned long this_len;

		if (chipnum >= private->numchips) {
			break;
		}

		if ((len + ofs - 1) >> private->chipshift) {
			this_len = (1 << private->chipshift) - ofs;
		} else {
			this_len = len;
		}

		ret = read_one_chip(map, &private->chips[chipnum], ofs,
				    this_len, buf);
		if (ret) {
			break;
		}

		*retlen += this_len;
		len -= this_len;
		buf += this_len;

		ofs = 0;
		chipnum++;
	}

	return ret;
}



static int write_one_word(struct map_info *map, struct flchip *chip,
			  unsigned long adr, __u32 datum)
{
	unsigned long timeo = jiffies + HZ;
	struct amd_flash_private *private = map->fldrv_priv;
	DECLARE_WAITQUEUE(wait, current);
	int ret = 0;
	int times_left;

retry:
	spin_lock_bh(chip->mutex);

	if (chip->state != FL_READY){
		printk("%s: waiting for chip to write, state = %d\n",
		       map->name, chip->state);
		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
                
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);
		printk(KERN_INFO "%s: woke up to write\n", map->name);
		if(signal_pending(current))
			return -EINTR;

		timeo = jiffies + HZ;

		goto retry;
	}	

	chip->state = FL_WRITING;

	adr += chip->start;
	ENABLE_VPP(map);
	send_cmd(map, chip->start, CMD_PROGRAM_UNLOCK_DATA);
	wide_write(map, datum, adr);

	times_left = 500000;
	while (times_left-- && flash_is_busy(map, adr, private->interleave)) { 
		if (need_resched()) {
			spin_unlock_bh(chip->mutex);
			schedule();
			spin_lock_bh(chip->mutex);
		}
	}

	if (!times_left) {
		printk(KERN_WARNING "%s: write to 0x%lx timed out!\n",
		       map->name, adr);
		ret = -EIO;
	} else {
		__u32 verify;
		if ((verify = wide_read(map, adr)) != datum) {
			printk(KERN_WARNING "%s: write to 0x%lx failed. "
			       "datum = %x, verify = %x\n",
			       map->name, adr, datum, verify);
			ret = -EIO;
		}
	}

	DISABLE_VPP(map);
	chip->state = FL_READY;
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);

	return ret;
}



static int amd_flash_write(struct mtd_info *mtd, loff_t to , size_t len,
			   size_t *retlen, const u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct amd_flash_private *private = map->fldrv_priv;
	int ret = 0;
	int chipnum;
	unsigned long ofs;
	unsigned long chipstart;

	*retlen = 0;
	if (!len) {
		return 0;
	}

	chipnum = to >> private->chipshift;
	ofs = to  - (chipnum << private->chipshift);
	chipstart = private->chips[chipnum].start;

	/* If it's not bus-aligned, do the first byte write. */
	if (ofs & (map->buswidth - 1)) {
		unsigned long bus_ofs = ofs & ~(map->buswidth - 1);
		int i = ofs - bus_ofs;
		int n = 0;
		u_char tmp_buf[4];
		__u32 datum;

		map->copy_from(map, tmp_buf,
			       bus_ofs + private->chips[chipnum].start,
			       map->buswidth);
		while (len && i < map->buswidth)
			tmp_buf[i++] = buf[n++], len--;

		if (map->buswidth == 2) {
			datum = *(__u16*)tmp_buf;
		} else if (map->buswidth == 4) {
			datum = *(__u32*)tmp_buf;
		} else {
			return -EINVAL;  /* should never happen, but be safe */
		}

		ret = write_one_word(map, &private->chips[chipnum], bus_ofs,
				     datum);
		if (ret) {
			return ret;
		}
		
		ofs += n;
		buf += n;
		(*retlen) += n;

		if (ofs >> private->chipshift) {
			chipnum++;
			ofs = 0;
			if (chipnum == private->numchips) {
				return 0;
			}
		}
	}
	
	/* We are now aligned, write as much as possible. */
	while(len >= map->buswidth) {
		__u32 datum;

		if (map->buswidth == 1) {
			datum = *(__u8*)buf;
		} else if (map->buswidth == 2) {
			datum = *(__u16*)buf;
		} else if (map->buswidth == 4) {
			datum = *(__u32*)buf;
		} else {
			return -EINVAL;
		}

		ret = write_one_word(map, &private->chips[chipnum], ofs, datum);

		if (ret) {
			return ret;
		}

		ofs += map->buswidth;
		buf += map->buswidth;
		(*retlen) += map->buswidth;
		len -= map->buswidth;

		if (ofs >> private->chipshift) {
			chipnum++;
			ofs = 0;
			if (chipnum == private->numchips) {
				return 0;
			}
			chipstart = private->chips[chipnum].start;
		}
	}

	if (len & (map->buswidth - 1)) {
		int i = 0, n = 0;
		u_char tmp_buf[2];
		__u32 datum;

		map->copy_from(map, tmp_buf,
			       ofs + private->chips[chipnum].start,
			       map->buswidth);
		while (len--) {
			tmp_buf[i++] = buf[n++];
		}

		if (map->buswidth == 2) {
			datum = *(__u16*)tmp_buf;
		} else if (map->buswidth == 4) {
			datum = *(__u32*)tmp_buf;
		} else {
			return -EINVAL;  /* should never happen, but be safe */
		}

		ret = write_one_word(map, &private->chips[chipnum], ofs, datum);

		if (ret) {
			return ret;
		}
		
		(*retlen) += n;
	}

	return 0;
}



static inline int erase_one_block(struct map_info *map, struct flchip *chip,
				  unsigned long adr, u_long size)
{
	unsigned long timeo = jiffies + HZ;
	struct amd_flash_private *private = map->fldrv_priv;
	DECLARE_WAITQUEUE(wait, current);

retry:
	spin_lock_bh(chip->mutex);

	if (chip->state != FL_READY){
		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
                
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);

		if (signal_pending(current)) {
			return -EINTR;
		}

		timeo = jiffies + HZ;

		goto retry;
	}	

	chip->state = FL_ERASING;

	adr += chip->start;
	ENABLE_VPP(map);
	send_cmd(map, chip->start, CMD_SECTOR_ERASE_UNLOCK_DATA);
	send_cmd_to_addr(map, chip->start, CMD_SECTOR_ERASE_UNLOCK_DATA_2, adr);
	
	timeo = jiffies + (HZ * 20);

	spin_unlock_bh(chip->mutex);
	schedule_timeout(HZ);
	spin_lock_bh(chip->mutex);
	
	while (flash_is_busy(map, adr, private->interleave)) {

		if (chip->state != FL_ERASING) {
			/* Someone's suspended the erase. Sleep */
			set_current_state(TASK_UNINTERRUPTIBLE);
			add_wait_queue(&chip->wq, &wait);
			
			spin_unlock_bh(chip->mutex);
			printk(KERN_INFO "%s: erase suspended. Sleeping\n",
			       map->name);
			schedule();
			remove_wait_queue(&chip->wq, &wait);
			
			if (signal_pending(current)) {
				return -EINTR;
			}
			
			timeo = jiffies + (HZ*2); /* FIXME */
			spin_lock_bh(chip->mutex);
			continue;
		}

		/* OK Still waiting */
		if (time_after(jiffies, timeo)) {
			chip->state = FL_READY;
			spin_unlock_bh(chip->mutex);
			printk(KERN_WARNING "%s: waiting for erase to complete "
			       "timed out.\n", map->name);
			DISABLE_VPP(map);

			return -EIO;
		}
		
		/* Latency issues. Drop the lock, wait a while and retry */
		spin_unlock_bh(chip->mutex);

		if (need_resched())
			schedule();
		else
			udelay(1);
		
		spin_lock_bh(chip->mutex);
	}

	/* Verify every single word */
	{
		int address;
		int error = 0;
		__u8 verify;

		for (address = adr; address < (adr + size); address++) {
			if ((verify = map->read8(map, address)) != 0xFF) {
				error = 1;
				break;
			}
		}
		if (error) {
			chip->state = FL_READY;
			spin_unlock_bh(chip->mutex);
			printk(KERN_WARNING
			       "%s: verify error at 0x%x, size %ld.\n",
			       map->name, address, size);
			DISABLE_VPP(map);

			return -EIO;
		}
	}
	
	DISABLE_VPP(map);
	chip->state = FL_READY;
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);

	return 0;
}



static int amd_flash_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	struct map_info *map = mtd->priv;
	struct amd_flash_private *private = map->fldrv_priv;
	unsigned long adr, len;
	int chipnum;
	int ret = 0;
	int i;
	int first;
	struct mtd_erase_region_info *regions = mtd->eraseregions;

	if (instr->addr > mtd->size) {
		return -EINVAL;
	}

	if ((instr->len + instr->addr) > mtd->size) {
		return -EINVAL;
	}

	/* Check that both start and end of the requested erase are
	 * aligned with the erasesize at the appropriate addresses.
	 */

	i = 0;

        /* Skip all erase regions which are ended before the start of
           the requested erase. Actually, to save on the calculations,
           we skip to the first erase region which starts after the
           start of the requested erase, and then go back one.
        */

        while ((i < mtd->numeraseregions) &&
	       (instr->addr >= regions[i].offset)) {
               i++;
	}
        i--;

	/* OK, now i is pointing at the erase region in which this
	 * erase request starts. Check the start of the requested
	 * erase range is aligned with the erase size which is in
	 * effect here.
	 */

	if (instr->addr & (regions[i].erasesize-1)) {
		return -EINVAL;
	}

	/* Remember the erase region we start on. */

	first = i;

	/* Next, check that the end of the requested erase is aligned
	 * with the erase region at that address.
	 */

	while ((i < mtd->numeraseregions) && 
	       ((instr->addr + instr->len) >= regions[i].offset)) {
                i++;
	}

	/* As before, drop back one to point at the region in which
	 * the address actually falls.
	 */

	i--;

	if ((instr->addr + instr->len) & (regions[i].erasesize-1)) {
                return -EINVAL;
	}

	chipnum = instr->addr >> private->chipshift;
	adr = instr->addr - (chipnum << private->chipshift);
	len = instr->len;

	i = first;

	while (len) {
		ret = erase_one_block(map, &private->chips[chipnum], adr,
				      regions[i].erasesize);

		if (ret) {
			return ret;
		}

		adr += regions[i].erasesize;
		len -= regions[i].erasesize;

		if ((adr % (1 << private->chipshift)) ==
		    ((regions[i].offset + (regions[i].erasesize *
		    			   regions[i].numblocks))
		     % (1 << private->chipshift))) {
			i++;
		}

		if (adr >> private->chipshift) {
			adr = 0;
			chipnum++;
			if (chipnum >= private->numchips) {
				break;
			}
		}
	}
		
	instr->state = MTD_ERASE_DONE;
	if (instr->callback) {
		instr->callback(instr);
	}
	
	return 0;
}



static void amd_flash_sync(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct amd_flash_private *private = map->fldrv_priv;
	int i;
	struct flchip *chip;
	int ret = 0;
	DECLARE_WAITQUEUE(wait, current);

	for (i = 0; !ret && (i < private->numchips); i++) {
		chip = &private->chips[i];

	retry:
		spin_lock_bh(chip->mutex);

		switch(chip->state) {
		case FL_READY:
		case FL_STATUS:
		case FL_CFI_QUERY:
		case FL_JEDEC_QUERY:
			chip->oldstate = chip->state;
			chip->state = FL_SYNCING;
			/* No need to wake_up() on this state change - 
			 * as the whole point is that nobody can do anything
			 * with the chip now anyway.
			 */
		case FL_SYNCING:
			spin_unlock_bh(chip->mutex);
			break;

		default:
			/* Not an idle state */
			add_wait_queue(&chip->wq, &wait);
			
			spin_unlock_bh(chip->mutex);

			schedule();

		        remove_wait_queue(&chip->wq, &wait);
			
			goto retry;
		}
	}

	/* Unlock the chips again */
	for (i--; i >= 0; i--) {
		chip = &private->chips[i];

		spin_lock_bh(chip->mutex);
		
		if (chip->state == FL_SYNCING) {
			chip->state = chip->oldstate;
			wake_up(&chip->wq);
		}
		spin_unlock_bh(chip->mutex);
	}
}



static int spi_flash_suspend(struct mtd_info *mtd)
{
    printk("amd_flash_suspend(): not implemented!\n");
	return -EINVAL;
}



static void spi_flash_resume(struct mtd_info *mtd)
{
    printk("amd_flash_resume(): not implemented!\n");
}



static void spi_flash_destroy(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct spi_flash_private *private = map->fldrv_priv;
	kfree(private);
}

int __init spi_flash_init(void)
{
	register_mtd_chip_driver( &spi_flash_chipdrv );
	return 0;
}

void __exit spi_flash_exit(void)
{
	unregister_mtd_chip_driver( &spi_flash_chipdrv );
}

module_init( spi_flash_init );
module_exit( spi_flash_exit );

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jonas Holmberg <jonas.holmberg@axis.com>");
MODULE_DESCRIPTION("Old MTD chip driver for AMD flash chips");
