#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/spi_nand.h>
#include <linux/interrupt.h>
#include <asm/io.h>


static int spi_nand_read (struct mtd_info *mtd, loff_t from, size_t len, size_t * retlen, u_char * buf);

static int spi_nand_write (struct mtd_info *mtd, loff_t to, size_t len, size_t * retlen, const u_char * buf);
static int spi_nand_erase (struct mtd_info *mtd, struct erase_info *instr);



/*
 *	Get chip for selected access
 */
static inline void spi_nand_get_chip (struct spi_nand_chip *this, struct mtd_info *mtd, int new_state, int *erase_state)
{

	DECLARE_WAITQUEUE (wait, current);

	/* 
	 * Grab the lock and see if the device is available 
	 * For erasing, we keep the spinlock until the
	 * erase command is written. 
	*/
retry:
	spin_lock_bh (&this->chip_lock);

	if (this->state == FL_READY) {
		this->state = new_state;
		if (new_state != FL_ERASING)
			spin_unlock_bh (&this->chip_lock);
		return;
	}

	if (this->state == FL_ERASING) {
		if (new_state != FL_ERASING) {
			this->state = new_state;
			spin_unlock_bh (&this->chip_lock);
			//nand_select ();	/* select in any case */
			//this->cmdfunc(mtd, NAND_CMD_RESET, -1, -1);
			return;
		}
	}

	set_current_state (TASK_UNINTERRUPTIBLE);
	add_wait_queue (&this->wq, &wait);
	spin_unlock_bh (&this->chip_lock);
	schedule ();
	remove_wait_queue (&this->wq, &wait);
	goto retry;
}


/*
*	Use spi NAND read 
*/
static int spi_nand_read (struct mtd_info *mtd, loff_t from, size_t len, size_t * retlen, u_char * buf)
{
	int erase_state = 0;
	struct spi_nand_chip *this = mtd->priv;
    /* Do not allow reads past end of device */
	if ((from + len) > mtd->size) 
	{
		DEBUG (MTD_DEBUG_LEVEL0, "spi_nand_read: Attempt read beyond end of device\n");
		*retlen = 0;
		return -EINVAL;
	}
	spi_nand_get_chip (this, mtd ,FL_READING, &erase_state);
    
	/* Send the read command */
	this->read(mtd,from, len, retlen, buf);
	/* Wake up anyone waiting on the device */
	spin_lock_bh (&this->chip_lock);
	this->state = FL_READY;
	wake_up (&this->wq);
	spin_unlock_bh (&this->chip_lock);

	return 0;

}			   


/*
*	Use NAND write
*/
static int spi_nand_write (struct mtd_info *mtd, loff_t to, size_t len, size_t * retlen, const u_char * buf)
{
	struct spi_nand_chip *this = mtd->priv;
    int *p,*q;
	/* Do not allow write past end of device */
	if ((to + len) > mtd->size) 
	{
		DEBUG (MTD_DEBUG_LEVEL0, "nand_write_oob: Attempt to write past end of page\n");
		return -EINVAL;
	}
    p = &to;
    q = p+1;
    //printk("spi_nand_write:to %x %x,len %x,buf %x\n",*q,*p,len,buf);
	/* Grab the lock and see if the device is available */
	spi_nand_get_chip (this, mtd, FL_WRITING, NULL);
	this->write(mtd,to, len, retlen, buf);
	/* Wake up anyone waiting on the device */
	spin_lock_bh (&this->chip_lock);
	this->state = FL_READY;
	wake_up (&this->wq);
	spin_unlock_bh (&this->chip_lock);
    *retlen = len;
	return 0;

}			   

static int spi_nand_writev (struct mtd_info *mtd, const struct iovec *vecs, unsigned long count, loff_t to, size_t * retlen)
{
	int i, cnt, len, total_len, ret = 0, written = 0;
    int *p,*q;
	struct spi_nand_chip *this = mtd->priv;

	/* Calculate total length of data */
	total_len = 0;
	for (i = 0; i < count; i++)
		total_len += (int) vecs[i].iov_len;

	/* Do not allow write past end of page */
	if ((to + total_len) > mtd->size) 
    {
		DEBUG (MTD_DEBUG_LEVEL0, "spi_nand_writev: Attempted write past end of device\n");
		return -EINVAL;
	}
	/* Grab the lock and see if the device is available */
	spi_nand_get_chip (this, mtd, FL_WRITING, NULL);
	while (count) 
    {
        p = &to;
        q = p+1;
        //printk("spi_nand_writev:count %x,to %x  %x, len %x, buf %x\n",count,*q,*p,vecs->iov_len,vecs->iov_base);
        if(vecs->iov_len != 0)
        this->write(mtd,to, vecs->iov_len, &cnt, vecs->iov_base);
        to += vecs->iov_len;
        vecs++;
	    count--;
        //write_flash_delay(0xffffff);
	}

	/* Wake up anyone waiting on the device */
	spin_lock_bh (&this->chip_lock);
	this->state = FL_READY;
	wake_up (&this->wq);
	spin_unlock_bh (&this->chip_lock);

	*retlen = total_len;
	return 0;
}

/*
 * NAND erase a block
 */
static int spi_nand_erase (struct mtd_info *mtd, struct erase_info *instr)
{
	int len, ret;
	struct spi_nand_chip *this = mtd->priv;
	//DECLARE_WAITQUEUE (wait, current);

	//printk ("nand_erase: start = 0x%08x, len = %x\n", (unsigned int) instr->addr, (unsigned int) instr->len);

	/* Start address must align on block boundary */
	if (instr->addr & (mtd->erasesize - 1)) {
		DEBUG (MTD_DEBUG_LEVEL0, "nand_erase: Unaligned address\n");
		return -EINVAL;
	}

	/* Length must align on block boundary */
	if (instr->len & (mtd->erasesize - 1)) {
		DEBUG (MTD_DEBUG_LEVEL0, "nand_erase: Length not block aligned\n");
		return -EINVAL;
	}

	/* Do not allow erase past end of device */
	if ((instr->len + instr->addr) > mtd->size) {
		DEBUG (MTD_DEBUG_LEVEL0, "nand_erase: Erase past end of device\n");
		return -EINVAL;
	}

	/* Grab the lock and see if the device is available */
	spi_nand_get_chip (this, mtd, FL_ERASING, NULL);

	len = instr->len;

	this->erase(mtd,instr->addr,instr->addr+len);
	
	instr->state = MTD_ERASE_DONE;
	spin_unlock_bh (&this->chip_lock);

	ret = instr->state == MTD_ERASE_DONE ? 0 : -EIO;
	/* Do call back function */
	if (!ret && instr->callback)
		instr->callback (instr);

	/* The device is ready */
	spin_lock_bh (&this->chip_lock);
	this->state = FL_READY;
	spin_unlock_bh (&this->chip_lock);

	/* Return more or less happy */
	return ret;
}


/*
 * spi NAND sync
 */
static void spi_nand_sync (struct mtd_info *mtd)
{
	struct spi_nand_chip *this = mtd->priv;
	DECLARE_WAITQUEUE (wait, current);

	DEBUG (MTD_DEBUG_LEVEL3, "nand_sync: called\n");

retry:
	/* Grab the spinlock */
	spin_lock_bh (&this->chip_lock);

	/* See what's going on */
	switch (this->state) {
	case FL_READY:
	case FL_SYNCING:
		this->state = FL_SYNCING;
		spin_unlock_bh (&this->chip_lock);
		break;

	default:
		/* Not an idle state */
		add_wait_queue (&this->wq, &wait);
		spin_unlock_bh (&this->chip_lock);
		schedule ();

		remove_wait_queue (&this->wq, &wait);
		goto retry;
	}

	/* Lock the device */
	spin_lock_bh (&this->chip_lock);

	/* Set the device to be ready again */
	if (this->state == FL_SYNCING) {
		this->state = FL_READY;
		wake_up (&this->wq);
	}

	/* Unlock the device */
	spin_unlock_bh (&this->chip_lock);
}

/*
 * Scan for the spi NAND device
 */
int spi_nand_scan (struct mtd_info *mtd)
{
	int i, nand_maf_id, nand_dev_id;
	struct spi_nand_chip *this = mtd->priv;

	/* reading device ID */
	this->read_device_id(mtd,&nand_maf_id,&nand_dev_id);
	printk("------------nand_maf_id=%#x, nand_dev_id=%#x",nand_maf_id,nand_dev_id);
	/* Print and store flash device information */
	for (i = 0; spi_nand_flash_ids[i].name != NULL; i++) 
	{
		if(nand_dev_id == spi_nand_flash_ids[i].id && !mtd->size)
		{
			mtd->name = spi_nand_flash_ids[i].name;
			mtd->erasesize = spi_nand_flash_ids[i].erasesize;
			mtd->size = (1 << spi_nand_flash_ids[i].chipshift);
		
			/* Try to identify manufacturer */
			for (i = 0; spi_nand_manuf_ids[i].id != 0x0; i++) 
			{
				if (spi_nand_manuf_ids[i].id == nand_maf_id)
					break;
			}	
			printk (KERN_INFO "SPI NAND device: Manufacture ID:"
				" 0x%02x, Chip ID: 0x%02x (%s %s)\n", nand_maf_id, nand_dev_id, 
				spi_nand_manuf_ids[i].name , mtd->name);
			break;
		}
	}

	/* Print warning message for no device */
	if (!mtd->size) 
	{
		printk (KERN_WARNING "No SPI_NAND device found!!!\n");
		return 1;
	}

	this->state = FL_READY;
	init_waitqueue_head (&this->wq);
	spin_lock_init (&this->chip_lock);
	/* Fill in remaining MTD driver data */
	
	mtd->type = MTD_NORFLASH;
	mtd->flags = MTD_CAP_NORFLASH;
	mtd->module = THIS_MODULE;
	mtd->ecctype = MTD_ECC_NONE;
	mtd->erase = spi_nand_erase;
	mtd->point = NULL;
	mtd->unpoint = NULL;
	mtd->read = spi_nand_read;
	mtd->write = spi_nand_write;
	mtd->read_ecc = NULL;
	mtd->write_ecc = NULL;
	mtd->read_oob = NULL;
	mtd->write_oob = NULL;
	mtd->readv = NULL;
	mtd->writev = spi_nand_writev;
	mtd->writev_ecc = NULL;
	mtd->sync = spi_nand_sync;
	mtd->lock = NULL;
	mtd->unlock = NULL;
	mtd->suspend = NULL;
	mtd->resume = NULL;

	/* Return happy */
	return 0;
}

EXPORT_SYMBOL (spi_nand_scan);

MODULE_LICENSE ("GPL");
MODULE_AUTHOR ("Steven J. Hill <sjhill@cotw.com>, Thomas Gleixner <tglx@linutronix.de>");
MODULE_DESCRIPTION ("Generic SPI NAND flash driver code");
