/* $Id: cskymac.c,v 1.4 2005/10/19 06:14:48 licq Exp $
 *
 * cskymac.c: Driver for CSKY CK1000EVB MAC 10/100baseT ethernet adapters.
 *
 * Copyright (C) 2005, Li Chunqiang (chunqiang_li@c-sky.com)
 */

#include <linux/module.h>

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/crc32.h>
#include <linux/errno.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>

#include "cskymac.h"


static struct cskymac *root_cskymac_dev;

static char version[] __initdata =
        "CSKY MAC driver v1.0 11/5/2005 Li Chunqiang (chunqiang_li@c-sky.com)\n";

const  unsigned long CSKYMACADDR[2] = { 0x00800000, 0x00000041 };
/*
#undef DEBUG_PROBE
#undef DEBUG_TX
#undef DEBUG_IRQ
#undef DEBUG_RX
#undef DEBUG_CSKY_MAC
*/

//#define DEBUG_PROBE
//#define DEBUG_TX
//#define DEBUG_IRQ
//#define DEBUG_RX
//#define DEBUG_CSKY_MAC

#ifdef DEBUG_PROBE
#define DP(x)  printk x
#else
#define DP(x)
#endif

#ifdef DEBUG_TX
#define DTX(x)  printk x
#else
#define DTX(x)
#endif

#ifdef DEBUG_IRQ
#define DIRQ(x)  printk x
#else
#define DIRQ(x)
#endif

#ifdef DEBUG_RX
#define DRX(x) printk x
#else
#define DRX(x)
#endif


static unsigned long cskymac_phyr_read(struct cskymac *dp, unsigned char phyr)
{
	unsigned long	tmp;

	tmp = 0x02;
	tmp |= (phyr << 8);
	CKMAC_REG_BASEADDR[CKMAC_MIIADD] = tmp;	
	CKMAC_REG_BASEADDR[CKMAC_MIICMD] = CKMAC_MIICMD_RSTAT;

	while (CKMAC_REG_BASEADDR[CKMAC_MIISTS] & CKMAC_MIISTS_BUSY);
	return (CKMAC_REG_BASEADDR[CKMAC_MIIRX] & 0xffff);
}

static void cskymac_phyr_write(struct cskymac *dp, unsigned char phyr, unsigned long w_value)
{
	unsigned long	tmp;

	tmp = 0x02;
	tmp |= (phyr << 8);
	CKMAC_REG_BASEADDR[CKMAC_MIIADD] = tmp;
	CKMAC_REG_BASEADDR[CKMAC_MIITX] = w_value & 0xffff;
	CKMAC_REG_BASEADDR[CKMAC_MIICMD] = CKMAC_MIICMD_WCTRLDATA;

	while (CKMAC_REG_BASEADDR[CKMAC_MIISTS] & CKMAC_MIISTS_BUSY);
	return;
}


static void cskymac_get_macaddr (unsigned char macaddr[])
{
	unsigned long  temp[2];
/*
	temp[1] = (CKMAC_REG_BASEADDR[CKMAC_ADDR0]);
	temp[0] = (CKMAC_REG_BASEADDR[CKMAC_ADDR1]);

	memcpy(macaddr, (unsigned char *)&temp[0]+2, 0x02);
	memcpy(&macaddr[2], (unsigned char *)&temp[1], 0x04);
*/
	temp[1] = CSKYMACADDR[1];
	temp[0] = CSKYMACADDR[0];

        memcpy(macaddr, (unsigned char *)&temp[1]+2, 0x02);
        memcpy(&macaddr[2], (unsigned char *)&temp[0], 0x04);
}

static void cskymac_set_macaddr (struct cskymac *dp)
{
	unsigned char	macaddr[6];
	CKMAC_REG_BASEADDR[CKMAC_ADDR1] = (CSKYMACADDR[1]);
	CKMAC_REG_BASEADDR[CKMAC_ADDR0] = (CSKYMACADDR[0]);
//
        unsigned long  temp[2];
        temp[1] = (CKMAC_REG_BASEADDR[CKMAC_ADDR0]);
        temp[0] = (CKMAC_REG_BASEADDR[CKMAC_ADDR1]);

        memcpy(macaddr, (unsigned char *)&temp[0]+2, 0x02);
        memcpy(&macaddr[2], (unsigned char *)&temp[1], 0x04);
	printk ("Mac Address: %2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x\n", macaddr[0], macaddr[1], macaddr[2],
                                               macaddr[3], macaddr[4], macaddr[5]);
/*
	cskymac_get_macaddr(macaddr);
	dp->dev->addr_len = 6;
	memcpy(dp->dev->dev_addr, macaddr, dp->dev->addr_len);
*/
}

static PTXDESC GetTxDescBaseAddr ( void )
{   
    return (PTXDESC)CKMAC_DESC_BASEADDR;
}   

static PRXDESC GetRxDescBaseAddr ( void )
{
    return (PRXDESC)CKMAC_DESC_BASEADDR + TX_RING_SIZE;
}

static void cskymac_tx_reset()
{
	int	txbdindex;
	PTXDESC ptxbd;

	ptxbd = GetTxDescBaseAddr();
	for (txbdindex = 0; txbdindex < TX_RING_SIZE; txbdindex++)
	{
		ClrBit(ptxbd->status, TX_RD_BIT);
		ptxbd++;
	}
}

static void cskymac_rx_reset()
{
	int	rxbdindex;
	PRXDESC	prxbd;

	prxbd  = GetRxDescBaseAddr();
	for (rxbdindex = 0; rxbdindex < RX_RING_SIZE; rxbdindex++)
	{
		//ClrBit(prxbd->status, RX_ETY_BIT);
		prxbd->status |=  RX_ETY_BIT;
		prxbd++;
	}

}

/* Reset the transmitter and receiver. */
static void cskymac_stop()
{
	unsigned long   tmp;

	//Disable tr & rx
	tmp = CKMAC_REG_BASEADDR[CKMAC_MODE];
	tmp &= ~CKMAC_MODE_TXEN;
	tmp &= ~CKMAC_MODE_RXEN;
	CKMAC_REG_BASEADDR[CKMAC_MODE] = tmp;
	//Reset the tx&rx BD
	cskymac_tx_reset();
	cskymac_rx_reset();
}


static void cskymac_clean_stats(struct cskymac *dp)
{
    memset(&dp->enet_stats, 0, sizeof (struct net_device_stats));
}


static void cskymac_clean_rings(struct cskymac *dp)
{
	int i;

	for (i = 0; i < RX_RING_SIZE; i++) 
	{
		if (dp->rx_skbs[i] != NULL) 
		{
			dev_kfree_skb_any(dp->rx_skbs[i]);
			dp->rx_skbs[i] = NULL;
		}
	}

	for (i = 0; i < TX_RING_SIZE; i++) 
	{
		if (dp->tx_skbs[i] != NULL) 
		{
			dev_kfree_skb_any(dp->tx_skbs[i]);
			dp->tx_skbs[i] = NULL;
		}
	}
}

static void cskymac_init_rings(struct cskymac *dp, int from_irq)
{
	struct net_device	*dev = dp->dev;
	int			i, gfp_flags = GFP_KERNEL;
        PRXDESC 		prxbd;
        PTXDESC			ptxbd;

	if (from_irq || in_interrupt())
		gfp_flags = GFP_ATOMIC;

	dp->rx_new = dp->rx_old = dp->tx_new = dp->tx_old = 0;

	/* Free any skippy bufs left around in the rings. */
	cskymac_clean_rings(dp);

	/* Now get new skbufs for the receive ring. */
        prxbd  = GetRxDescBaseAddr();
	for (i = 0; i < RX_RING_SIZE; i++,prxbd++) 
	{
		struct sk_buff *skb;

		skb = alloc_skb(RX_BUF_ALLOC_SIZE, gfp_flags);
		if (!skb)
		{
			continue;
		}

		dp->rx_skbs[i] = skb;
		skb->dev = dev;

		/* Because we reserve afterwards. */
		skb_put(skb, ETH_FRAME_LEN);
		skb_reserve(skb, 34);
		prxbd->buf = skb->data;
		if (i == RX_RING_SIZE - 1)
		{
			prxbd->status = RX_ETY_BIT | RX_IRQ_BIT | RX_WR_BIT;
		}
		else
		{
			prxbd->status = RX_ETY_BIT | RX_IRQ_BIT;
		}

	}

	ptxbd = GetTxDescBaseAddr();
	for (i = 0; i < TX_RING_SIZE; i++,ptxbd++)
	{
		if (i == TX_RING_SIZE - 1)
		{
			ptxbd->status = TX_IRQ_BIT | TX_WR_BIT | TX_PAD_BIT;
		}
		else
		{
			ptxbd->status = TX_IRQ_BIT | TX_PAD_BIT;
		}
		ptxbd->buf = 0;
	}
}


static int cskymac_init(struct cskymac *, int);

static int try_next_permutation(struct cskymac *dp)
{
	unsigned long	tmp;

	tmp = cskymac_phyr_read(dp, RTL8201_MODECTRL);
	if (tmp & RTL8201_MODECTRL_ANE)
	{
		int timeout;
                // Reset the PHY.
                tmp = RTL8201_MODECTRL_RESET;
                cskymac_phyr_write(dp, RTL8201_MODECTRL, tmp);

                timeout = 64;
                while (--timeout)
                {
                        tmp = cskymac_phyr_read(dp, RTL8201_MODECTRL);
                        if ((tmp & RTL8201_MODECTRL_RESET) == 0)
                        {
                                break;
                        }
                        udelay(20);
                }
                if (timeout == 0)
                {
                        printk(KERN_ERR "%s: PHY reset failed.\n", dp->dev->name);
                }

                tmp = cskymac_phyr_read(dp, RTL8201_MODECTRL);

                // Now we try 100baseT. 
                tmp = RTL8201_MODECTRL_SPD100 | RTL8201_MODECTRL_DUPLEX | RTL8201_MODECTRL_LOOPBACK;
                cskymac_phyr_write(dp, RTL8201_MODECTRL, tmp);
                return 0;
	}
	else if (tmp & RTL8201_MODECTRL_SPD100) 
	{
		int timeout;

		// Reset the PHY.
		tmp = RTL8201_MODECTRL_RESET;
		cskymac_phyr_write(dp, RTL8201_MODECTRL, tmp);

		timeout = 64;
		while (--timeout) 
		{
			tmp = cskymac_phyr_read(dp, RTL8201_MODECTRL);
			if ((tmp & RTL8201_MODECTRL_RESET) == 0)
			{
				break;
			}
			udelay(20);
		}
		if (timeout == 0)
		{
			printk(KERN_ERR "%s: PHY reset failed.\n", dp->dev->name);
		}

		tmp = cskymac_phyr_read(dp, RTL8201_MODECTRL);

		// Now we try 10baseT. 
		tmp = RTL8201_MODECTRL_DUPLEX | RTL8201_MODECTRL_LOOPBACK;
		cskymac_phyr_write(dp, RTL8201_MODECTRL, tmp);
		return 0;
	}

	return -1;
}


static void cskymac_timer(unsigned long data)
{
	struct cskymac	*dp = (struct cskymac *) data;
	int		restart_timer = 0;
	unsigned long	tmp;

	dp->timer_ticks++;
	if (dp->timer_state == ltrywait) 
	{
		tmp = cskymac_phyr_read(dp, RTL8201_MODESTS);
		if (tmp & RTL8201_MODESTS_LINKSTS)
		{
			printk(KERN_INFO "%s: Link is now up at %s.\n", dp->dev->name,
				(tmp & RTL8201_MODESTS_100BTXFD) ?  "100baseT" : "10baseT");
			dp->timer_state = asleep;
			restart_timer = 0;
		}
		else
		{
			if (dp->timer_ticks >= 4) 
			{
				int ret;

				ret = try_next_permutation(dp);
				if (ret == -1) 
				{
					printk(KERN_ERR "%s: Link down, cable problem?\n", dp->dev->name);
					ret = cskymac_init(dp, 0);
					if (ret) 
					{
						printk(KERN_ERR "%s: Error, cannot re-init the CSKY MAC.\n", dp->dev->name);
					}
					return;
				}
				dp->timer_ticks = 0;
				restart_timer = 1;
			}
			else
			{
				restart_timer = 1;
			}
		}
	}
	else
	{
		// Can't happens.... 
		printk(KERN_ERR "%s: Aieee, link timer is asleep but we got one anyways!\n",
		       dp->dev->name);
		restart_timer = 0;
		dp->timer_ticks = 0;
		dp->timer_state = asleep; // foo on you 
	}

	if (restart_timer != 0)
	{
		dp->cskymac_timer.expires = jiffies + ((12 * HZ)/10); // 1.2 sec. 
		add_timer(&dp->cskymac_timer);
	}
}


/* Well, really we just force the chip into 100baseT then
 * 10baseT, each time checking for a link status.
 */

static void cskymac_begin_auto_negotiation(struct cskymac *dp)
{
	unsigned long	tmp;
	int 		timeout;

	// Reset the PHY. 
	tmp = RTL8201_MODECTRL_RESET;
	cskymac_phyr_write(dp, RTL8201_MODECTRL, tmp);

	timeout = 64;
	while (--timeout) 
	{
		tmp = cskymac_phyr_read(dp, RTL8201_MODECTRL);
		if ((tmp & RTL8201_MODECTRL_RESET) == 0)
		{
			break;
		}
		udelay(20);
	}
	if (timeout == 0)
	{
		printk(KERN_ERR "%s: PHY reset failed.\n", dp->dev->name);
	}

	tmp = cskymac_phyr_read(dp, RTL8201_MODECTRL);

	// First we try auto-negotiation (not 100baseT). 
	//tmp = RTL8201_MODECTRL_SPD100 | RTL8201_MODECTRL_DUPLEX | RGL8201_MODECTRL_LOOPBACK;
	tmp = RTL8201_MODECTRL_ANE | RTL8201_MODECTRL_LOOPBACK;
	cskymac_phyr_write(dp, RTL8201_MODECTRL, tmp);

	dp->timer_state = ltrywait;
	dp->timer_ticks = 0;
	dp->cskymac_timer.expires = jiffies + (12 * HZ) / 10;
	dp->cskymac_timer.data = (unsigned long) dp;
	dp->cskymac_timer.function = &cskymac_timer;
	add_timer(&dp->cskymac_timer);
}

static int cskymac_init(struct cskymac *dp, int from_irq)
{
	//Set mac working mode
	CKMAC_REG_BASEADDR[CKMAC_MODE]   = CKMAC_MODE_LPMD_RXEN |
			CKMAC_MODE_PAD | CKMAC_MODE_CRCEN | CKMAC_MODE_FULLD;	//0x0002a400.

	//Initial PHY
	CKMAC_REG_BASEADDR[CKMAC_MIIADD] = 0x00000102;		// mii address
	CKMAC_REG_BASEADDR[CKMAC_MIICMD] = CKMAC_MIICMD_RSTAT;	// mii command (read status register phy)
	// waiting for set PHY Addr and REG ADDR
	//while (CKMAC_REG_BASEADDR[CKMAC_MIISTS] & 0x01);

	// set transmit buffer descriptor
	CKMAC_REG_BASEADDR[CKMAC_TXBD]   = TX_RING_SIZE;

	// set interrupt mask.
	CKMAC_REG_BASEADDR[CKMAC_IMASK]  = CKMAC_IMASK_RXC | CKMAC_IMASK_TXC |
				CKMAC_IMASK_BUSY | CKMAC_IMASK_RXE | CKMAC_IMASK_RXB |
				CKMAC_IMASK_TXE | CKMAC_IMASK_TXB;	//0x0000007e;
	// set packet length, Min length = 0x40, Max length = 0x600;
	CKMAC_REG_BASEADDR[CKMAC_SIZE]  = 0x00400600;
	CKMAC_REG_BASEADDR[CKMAC_INT] = 0xff;		//clear the interrupt source register

	// Latch current counters into statistics.
	cskymac_clean_stats(dp);
	// Alloc and reset the tx/rx descriptor chains.
	cskymac_init_rings(dp, from_irq);
	// Stop transmitter and receiver.
	cskymac_stop();
	// Set hardware ethernet address.
	cskymac_set_macaddr(dp);

	// Clear the hash table until mc upload occurs.
	CKMAC_REG_BASEADDR[CKETH_HASH0] = 0;
	CKMAC_REG_BASEADDR[CKETH_HASH1] = 0;
	// Enable transmitter and receiver.
	CKMAC_REG_BASEADDR[CKMAC_MODE] |= CKMAC_MODE_TXEN | CKMAC_MODE_RXEN;
	//CKMAC_REG_BASEADDR[CKMAC_MODE] |= CKMAC_MODE_RXEN;

	//cskymac_begin_auto_negotiation(dp);

	return 0;
}

/* Error interrupts get sent here. */
static void cskymac_error(struct cskymac *dp, unsigned long mac_status)
{
	printk(KERN_ERR "cskymac_error: ");
	if (mac_status & CKMAC_INT_BER) 
	{
		printk("Bus error,");
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_BER;
	}
	else if (mac_status & CKMAC_INT_RXE)
	{
		printk("Receive error,");
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_RXE;
		// the detailed rx_errors status of BD
	}
	else if (mac_status & CKMAC_INT_TXE)
	{
		printk("Transmit error,");
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_TXE;
		// the detailed tx_errors status of BD
	}

	printk(" reset mac\n");
	cskymac_init(dp, 1);
}

/* CSKY MAC transmit complete service routines. */
static void cskymac_tx(struct cskymac *dp)
{
	PTXDESC			txbase = GetTxDescBaseAddr();
	struct net_device	*dev = dp->dev;
	int			elem;

	spin_lock(&dp->lock);

	elem = dp->tx_old;
	DTX(("cskymac_tx: tx_old[%d] ", elem));
	while (elem != dp->tx_new)
	{
		struct sk_buff	*skb;
		PTXDESC 	this = (PTXDESC)(&txbase[elem]);

		DTX(("this(%p) [flags(%08x)addr(%08x)]",
		     this, this->status, this->buf));

		if (this->status & TX_RD_BIT)
		{
			break;
		}
		skb = dp->tx_skbs[elem];
		dp->enet_stats.tx_packets++;
		dp->enet_stats.tx_bytes += skb->len;
		DTX(("skb(%p) ", skb));
		dp->tx_skbs[elem] = NULL;
		dev_kfree_skb_irq(skb);

		elem = NEXT_TX(elem);
	}
	DTX((" DONE, tx_old=%d\n", elem));
	dp->tx_old = elem;

	if (netif_queue_stopped(dev))// && TX_BUFFS_AVAIL(dp) > 0)
	{
		netif_wake_queue(dp->dev);
	}

	spin_unlock(&dp->lock);
}

/* CSKY MAC receive complete service routines. */
static void cskymac_rx(struct cskymac *dp)
{
	PRXDESC		rxbase = GetRxDescBaseAddr();
	PRXDESC		this;
#ifdef	CONFIG_CSKYMAC_VERSION101
	int 		elem = dp->rx_new;
#else
	int 		elem = 0;
#endif
	int		drops = 0;
	int		i;
	u32 		flags;
	int		rx_status;

	this = &rxbase[elem];
	while (!((flags = this->status) & RX_ETY_BIT)) 
	{
		struct sk_buff	*skb;
		int 		len = (flags & RXD_LENGTH) >> 16; 

		DRX(("cskymac_rx: rx_bd %d : 0x%x received a frame with length = %d from status\n", elem, this, len));
		//Get the size of the received data.
		skb = dp->rx_skbs[elem];
		len   = (skb->data[16] << 8) | skb->data[17];
		rx_status = (skb->data[12] << 8) | skb->data[13];
		if (rx_status == ETH_P_ARP)
		{
			len = 42;
		}
		else if (rx_status == ETH_P_IP)
		{
			len += 14;
		}
		else
		{
			this->status |= RX_ETY_BIT;
			continue;
		}

		DRX(("cskymac_rx: rx_bd %d : 0x%x received a frame with length = %d, status = 0x%x\n", elem, this, len, flags));
		/* Check for errors. */
		if (len < 42)//ETH_ZLEN)
		{
			dp->enet_stats.rx_errors++;
			dp->enet_stats.rx_length_errors++;
	drop_it:
			/* Return it to the CSKY MAC. */
			dp->enet_stats.rx_dropped++;
			this->status |= RX_ETY_BIT;
			goto next;
		}
		if (len > RX_LENGTH_THRESHOLD)
		{
			struct sk_buff *new_skb;

			/* Now refill the entry, if we can. */
			new_skb = alloc_skb(RX_BUF_ALLOC_SIZE, GFP_ATOMIC);
			if (new_skb == NULL) 
			{
				drops++;
				goto drop_it;
			}
			dp->rx_skbs[elem] = new_skb;
			new_skb->dev = dp->dev;
			skb_put(new_skb, ETH_FRAME_LEN);
			skb_reserve(new_skb, 34);
			this->buf = new_skb->data;
			this->status |= RX_ETY_BIT;

			/* Trim the original skb for the netif. */
			skb_trim(skb, len);
		}
		else
		{
			struct sk_buff *copy_skb = dev_alloc_skb(len + 2);

			if (copy_skb == NULL) 
			{
				drops++;
				goto drop_it;
			}
			copy_skb->dev = dp->dev;
			skb_reserve(copy_skb, 2);
			skb_put(copy_skb, len);
			eth_copy_and_sum(copy_skb, (unsigned char *)skb->data, len, 0);

			/* Reuse original ring buffer. */
			this->status |= RX_ETY_BIT;

			skb = copy_skb;
		}
#ifdef DEBUG_CSKY_MAC
		printk("\nReceiving data (%d): ", len);
		for (i = 0x00; i < len; i++)
		{
			printk("%2.2x ", skb->data[i] & 0xff);
		}
		printk("\n");
#endif

		/* No checksums done by the CSKY MAC ;-( */
		skb->protocol = eth_type_trans(skb, dp->dev);
		netif_rx(skb);
		dp->dev->last_rx = jiffies;
		dp->enet_stats.rx_packets++;
		dp->enet_stats.rx_bytes += len;
	next:
#ifndef CONFIG_CSKYMAC_VERSION101
		break;
#endif
		elem = NEXT_RX(elem);
		this = &rxbase[elem];
		if (elem == dp->rx_new)
		{
			break;
		}
	}
	dp->rx_new = elem;
	if (drops)
	{
		printk(KERN_NOTICE "%s: Memory squeeze, deferring packet.\n", dp->dev->name);
	}
}

static void cskymac_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct cskymac	*dp = (struct cskymac *) dev_id;
	u32		mac_status;

	DIRQ(("cskymac_interrupt: "));

	/* Latch status registers now. */
	mac_status = CKMAC_REG_BASEADDR[CKMAC_INT];
	DIRQ(("mac_status=%08x\n", mac_status));
	//Mac error!
	if (mac_status & (CKMAC_INT_TXE | CKMAC_INT_RXE | CKMAC_INT_BER))
	{
		cskymac_error(dp, mac_status);
		DIRQ(("cskymac_interrupt: error!\n"));
	}
	//Receive buffer description overflow
        if (mac_status & CKMAC_INT_BUSY)
        {
		cskymac_rx_reset();
                CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_BUSY;
		DIRQ(("cskymac_interrupt: busy!\n"));
		//Ready to send a control frame
        }
	//Having received a control frame
	if (mac_status & CKMAC_INT_RXC)
	{
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_RXC;
		DIRQ(("cskymac_interrupt: recieve a control frame!\n"));
	}
	//A control frame is sent
	if (mac_status & CKMAC_INT_TXC)
	{
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_TXC;
		DIRQ(("cskymac_interrupt: have sent a control frame!\n"));
	}
	//A data frame is sent
	if (mac_status & CKMAC_INT_TXB)
	{
		cskymac_tx(dp);
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_TXB;
		DIRQ(("cskymac_interrupt: have sent a frame!\n"));
	}
	//A data frame is received
	if (mac_status & CKMAC_INT_RXB)
	{
		cskymac_rx(dp);
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_RXB;
		DIRQ(("cskymac_interrupt: recieve a frame!\n"));
	}
}

static int cskymac_open(struct net_device *dev)
{
	struct cskymac	*dp = (struct cskymac *) dev->priv;
	int 		ret;

	printk("cskymac_open \n");

	init_timer(&dp->cskymac_timer);
	ret = cskymac_init(dp, 0);
	if (ret)
	{
		return ret;
	} 

	ret = request_irq(dev->irq, &cskymac_interrupt, SA_INTERRUPT, dev->name, dp);
        if (ret)
        {
		del_timer(&dp->cskymac_timer);
		dp->timer_state = asleep;
	        dp->timer_ticks = 0;
	        cskymac_stop();
       		cskymac_clean_rings(dp);
                printk(KERN_ERR "CSKY MAC: Can't order irq %d to go.\n", dev->irq);
        }

	return ret;
}

static int cskymac_close(struct net_device *dev)
{
	struct cskymac *dp = (struct cskymac *) dev->priv;
	printk("cskymac_close\n");

	del_timer(&dp->cskymac_timer);
	dp->timer_state = asleep;
	dp->timer_ticks = 0;

	cskymac_stop();
	cskymac_clean_rings(dp);
	free_irq(dev->irq, dp);
	return 0;
}

static void cskymac_tx_timeout(struct net_device *dev)
{
	struct cskymac *dp = (struct cskymac *) dev->priv;

	printk("cskymac_tx_timeout\n");
	dp->enet_stats.tx_errors++;
	cskymac_init(dp, 0);
	netif_wake_queue(dev);
}

/* 
Put a packet on the wire. 
Initial a transmision for device.
*/
static int cskymac_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct cskymac	*dp = (struct cskymac *) dev->priv;
	int		len, entry;
	PTXDESC		ptxbd;

	printk("cskymac_start_xmit\n");

	len = skb->len;
	ptxbd = GetTxDescBaseAddr();

	/* Avoid a race... */
	spin_lock_irq(&dp->lock);
	entry = dp->tx_new;
	DTX(("cskymac_start_xmit: len(%d) entry(%d)\n", len, entry));
	dp->tx_skbs[entry] = skb;
	ptxbd[entry].buf = skb->data;
        if (entry == TX_RING_SIZE - 1)
        {
		ptxbd[entry].status = (TX_RD_BIT | TX_IRQ_BIT | TX_PAD_BIT | TX_WR_BIT | (len << 16));
        }
        else
        {
		ptxbd[entry].status = (TX_RD_BIT | TX_IRQ_BIT | TX_PAD_BIT | (len << 16));
        }
	CKMAC_REG_BASEADDR[CKMAC_MODE] |= CKMAC_MODE_TXEN | CKMAC_MODE_RXEN;
	dp->tx_new = NEXT_TX(entry);
	if (TX_BUFFS_AVAIL(dp) <= 0)	//If there is no Buffer description, we must stop receive transmision from application.
	{
		netif_stop_queue(dev);
	}
	spin_unlock_irq(&dp->lock);

	dev->trans_start = jiffies;

	return 0;
}

static struct net_device_stats *cskymac_get_stats(struct net_device *dev)
{
	struct cskymac *dp = (struct cskymac *) dev->priv;

	printk("cskymac_get_stats\n");


	return &dp->enet_stats;
}

static void cskymac_set_multicast(struct net_device *dev)
{

	struct dev_mc_list	*dmi = dev->mc_list;
	char			*addrs;
	int 			i;
	u32 			tmp, crc;

	printk("cskymac_set_multicast\n");
	/* Disable the receiver.  The bit self-clears when
	 * the operation is complete.
	 */
	tmp = CKMAC_REG_BASEADDR[CKMAC_MODE];
	tmp &= ~(CKMAC_MODE_RXEN);
	CKMAC_REG_BASEADDR[CKMAC_MODE] = tmp;

	if ((dev->flags & IFF_ALLMULTI) || (dev->mc_count > CSKYMAC_TABLE_SIZE))
	{
		CKMAC_REG_BASEADDR[CKETH_HASH0] = 0xffffffff;
		CKMAC_REG_BASEADDR[CKETH_HASH1] = 0xffffffff;
		CKMAC_REG_BASEADDR[CKMAC_MODE] &= ~CKMAC_MODE_PRO;
		CKMAC_REG_BASEADDR[CKMAC_MODE] |= CKMAC_MODE_IAM;
	}
	else if (dev->flags & IFF_PROMISC)
	{
		tmp = CKMAC_REG_BASEADDR[CKMAC_MODE];
		tmp |= CKMAC_MODE_PRO;
		CKMAC_REG_BASEADDR[CKMAC_MODE] = tmp;
	}
	else if (dev->mc_count == 0)
	{
		tmp = CKMAC_REG_BASEADDR[CKMAC_MODE];
		tmp &= ~CKMAC_MODE_PRO;
		tmp &= ~CKMAC_MODE_IAM;
		CKMAC_REG_BASEADDR[CKMAC_MODE] = tmp;
	}
	else
	{
		u16 hash_table[2];

		for (i = 0; i < 2; i++)
		{
			hash_table[i] = 0;
		}

		for (i = 0; i < dev->mc_count; i++)
		{
			addrs = dmi->dmi_addr;
			dmi = dmi->next;

			if (!(*addrs & 1))
			{
				continue;
			}

			crc = ether_crc_le(6, addrs);
			crc >>= 26;
			hash_table[crc >> 5] |= 1 << (crc & 0x1f);
		}
		CKMAC_REG_BASEADDR[CKETH_HASH0] = hash_table[0];
		CKMAC_REG_BASEADDR[CKETH_HASH1] = hash_table[1];
		CKMAC_REG_BASEADDR[CKMAC_MODE] &= ~CKMAC_MODE_PRO;
		CKMAC_REG_BASEADDR[CKMAC_MODE] |= CKMAC_MODE_IAM;
	}

	/* Re-enable the receiver. */
	tmp = CKMAC_REG_BASEADDR[CKMAC_MODE];
	tmp |= CKMAC_MODE_RXEN;
	CKMAC_REG_BASEADDR[CKMAC_MODE] = tmp;
}

static int __init cskymac_ether_init()
{
	static int		version_printed;
	struct cskymac		*dp;
        struct net_device	*dev;
	int			i;

	/* Get a new device struct for this interface. */
	dev = init_etherdev(NULL, sizeof(struct cskymac));
	if (!dev)
	{
		return -ENOMEM;
	}
	if (version_printed++ == 0)
	{
		printk(KERN_INFO "%s", version);
	}
	SET_MODULE_OWNER(dev);

	/* Report what we have found to the user. */
	printk(KERN_INFO "%s: CSKY MAC 10/100baseT Ethernet ", dev->name);
	dev->base_addr = CKMAC_REG_BASEADDR;
	cskymac_get_macaddr(dev->dev_addr);
	dev->addr_len = 6;
	for (i = 0; i < 6; i++)
	{
		printk("%2.2x%c", dev->dev_addr[i], i == 5 ? ' ' : ':');
	}
	printk("aa\n");

	/* Setup softc, with backpointers to QEC and CSKY MAC SBUS device structs. */
	dp = dev->priv;

	spin_lock_init(&dp->lock);
	/* Stop the CSKY MAC. */
	cskymac_stop(dp);

	/* Init auto-negotiation timer state. */
	init_timer(&dp->cskymac_timer);
	dp->timer_state = asleep;
	dp->timer_ticks = 0;

	/* Backlink to generic net device struct. */
	dp->dev = dev;

	/* Set links to our CSKY MAC open and close routines. */
	dev->open = &cskymac_open;
	dev->stop = &cskymac_close;
	dev->hard_start_xmit = &cskymac_start_xmit;

	/* Set links to CSKY MAC statistic and multi-cast loading code. */
	dev->get_stats = &cskymac_get_stats;
	dev->set_multicast_list = &cskymac_set_multicast;

	dev->tx_timeout = &cskymac_tx_timeout;
	dev->watchdog_timeo = 5 * HZ;

	/* Finish net device registration. */
	dev->irq = CSKYMAC_IRQBASE;
	dev->dma = 0;
	ether_setup(dev);

	root_cskymac_dev = dp;

	return 0;

fail_and_cleanup:
	/* Something went wrong, undo whatever we did so far. */
	/* Free register mappings if any. */
	unregister_netdev(dev);
	kfree(dev);
	return -ENODEV;
}

static void __exit cskymac_cleanup(void)
{
	struct cskymac *dp = root_cskymac_dev;

	unregister_netdev(dp->dev);
	kfree(dp->dev);
}


module_init(cskymac_ether_init);
module_exit(cskymac_cleanup);
MODULE_LICENSE("GPL");
