     /** \file
 * This is the network dependent layer to handle network related functionality.
 * This file is tightly coupled to neworking frame work of linux 2.6.xx kernel.
 * The functionality carried out in this file should be treated as an example only
 * if the underlying operating system is not Linux. 
 * 
 * \note Many of the functions other than the device specific functions
 *  changes for operating system other than Linux 2.6.xx
 * \internal 
 *-----------------------------REVISION HISTORY-----------------------------------
 * Synopsys			01/Aug/2007				Created
 */


//GMACõ32λ
unsigned int *pDbgMem = 0xcfff0000;

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <asm/irq.h>

#include "synopGMAC_Host.h"
#include "synopGMAC_plat.h"
#include "synopGMAC_network_interface.h"
#include "synopGMAC_Dev.h"


#define IOCTL_READ_REGISTER  SIOCDEVPRIVATE+1
#define IOCTL_WRITE_REGISTER SIOCDEVPRIVATE+2
#define IOCTL_READ_IPSTRUCT  SIOCDEVPRIVATE+3
#define IOCTL_READ_RXDESC    SIOCDEVPRIVATE+4
#define IOCTL_READ_TXDESC    SIOCDEVPRIVATE+5
#define IOCTL_POWER_DOWN     SIOCDEVPRIVATE+6

static struct timer_list synopGMAC_cable_unplug_timer;
static u32 GMAC_Power_down=0; // This global variable is used to indicate the ISR whether the interrupts occured in the process of powering down the mac or not

/*these are the global data for base address and its size*/
extern u8* synopGMACMappedAddr;

/*These are the global pointers for their respecive structures*/
extern synopGMACPciNetworkAdapter * synopGMACadapter;

u32 synop_pci_using_dac = 0;


/*Sample Wake-up frame filter configurations*/

u32 synopGMAC_wakeup_filter_config0[] = {
					0x00000000,	// For Filter0 CRC is not computed may be it is 0x0000
					0x00000000,	// For Filter1 CRC is not computed may be it is 0x0000
					0x00000000,	// For Filter2 CRC is not computed may be it is 0x0000
					0x5F5F5F5F,     // For Filter3 CRC is based on 0,1,2,3,4,6,8,9,10,11,12,14,16,17,18,19,20,22,24,25,26,27,28,30 bytes from offset
					0x09000000,     // Filter 0,1,2 are disabled, Filter3 is enabled and filtering applies to only multicast packets
					0x1C000000,     // Filter 0,1,2 (no significance), filter 3 offset is 28 bytes from start of Destination MAC address 
					0x00000000,     // No significance of CRC for Filter0 and Filter1
					0xBDCC0000      // No significance of CRC for Filter2, Filter3 CRC is 0xBDCC
					};
u32 synopGMAC_wakeup_filter_config1[] = {
					0x00000000,	// For Filter0 CRC is not computed may be it is 0x0000
					0x00000000,	// For Filter1 CRC is not computed may be it is 0x0000
					0x7A7A7A7A,	// For Filter2 CRC is based on 1,3,4,5,6,9,11,12,13,14,17,19,20,21,25,27,28,29,30 bytes from offset
					0x00000000,     // For Filter3 CRC is not computed may be it is 0x0000
					0x00010000,     // Filter 0,1,3 are disabled, Filter2 is enabled and filtering applies to only unicast packets
					0x00100000,     // Filter 0,1,3 (no significance), filter 2 offset is 16 bytes from start of Destination MAC address 
					0x00000000,     // No significance of CRC for Filter0 and Filter1
					0x0000A0FE      // No significance of CRC for Filter3, Filter2 CRC is 0xA0FE
					};
u32 synopGMAC_wakeup_filter_config2[] = {
					0x00000000,	// For Filter0 CRC is not computed may be it is 0x0000
					0x000000FF,	// For Filter1 CRC is computed on 0,1,2,3,4,5,6,7 bytes from offset
					0x00000000,	// For Filter2 CRC is not computed may be it is 0x0000
					0x00000000,     // For Filter3 CRC is not computed may be it is 0x0000
					0x00000100,     // Filter 0,2,3 are disabled, Filter 1 is enabled and filtering applies to only unicast packets
					0x0000DF00,     // Filter 0,2,3 (no significance), filter 1 offset is 223 bytes from start of Destination MAC address 
					0xDB9E0000,     // No significance of CRC for Filter0, Filter1 CRC is 0xDB9E
					0x00000000      // No significance of CRC for Filter2 and Filter3 
					};

/*
The synopGMAC_wakeup_filter_config3[] is a sample configuration for wake up filter. 
Filter1 is used here
Filter1 offset is programmed to 50 (0x32)
Filter1 mask is set to 0x000000FF, indicating First 8 bytes are used by the filter
Filter1 CRC= 0x7EED this is the CRC computed on data 0x55 0x55 0x55 0x55 0x55 0x55 0x55 0x55

Refer accompanied software DWC_gmac_crc_example.c for CRC16 generation and how to use the same.
*/

u32 synopGMAC_wakeup_filter_config3[] = {
					0x00000000,	// For Filter0 CRC is not computed may be it is 0x0000
					0x000000FF,	// For Filter1 CRC is computed on 0,1,2,3,4,5,6,7 bytes from offset
					0x00000000,	// For Filter2 CRC is not computed may be it is 0x0000
					0x00000000,     // For Filter3 CRC is not computed may be it is 0x0000
					0x00000100,     // Filter 0,2,3 are disabled, Filter 1 is enabled and filtering applies to only unicast packets
					0x00003200,     // Filter 0,2,3 (no significance), filter 1 offset is 50 bytes from start of Destination MAC address 
					0x7eED0000,     // No significance of CRC for Filter0, Filter1 CRC is 0x7EED, 
					0x00000000      // No significance of CRC for Filter2 and Filter3 
					};
/**
 * Function used to detect the cable plugging and unplugging.
 * This function gets scheduled once in every second and polls
 * the PHY register for network cable plug/unplug. Once the 
 * connection is back the GMAC device is configured as per
 * new Duplex mode and Speed of the connection.
 * @param[in] u32 type but is not used currently. 
 * \return returns void.
 * \note This function is tightly coupled with Linux 2.6.xx.
 * \callgraph
 */

static void synopGMAC_linux_cable_unplug_function(u32 notused)
{
	s32 status;
	u16 data;
	synopGMACPciNetworkAdapter *adapter = (synopGMACPciNetworkAdapter *)synopGMAC_cable_unplug_timer.data;
	synopGMACdevice            *gmacdev = adapter->synopGMACdev;

int i;
	init_timer(&synopGMAC_cable_unplug_timer);
	synopGMAC_cable_unplug_timer.expires = CHECK_TIME + jiffies;
	status = synopGMAC_read_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,PHY_VSC8641_MODESTS, &data);


	//TR(" =timer check RTL8201_MODESTS: %08x\n",data);
	if((data & PHY_VSC8641_MODESTS_LNKSTS) == 0){
		TR(" =timer check=No Link: %08x\n",data);
	  	gmacdev->LinkState = 0;
		gmacdev->DuplexMode = 0;
		gmacdev->Speed = 0;
		gmacdev->LoopBackMode = 0; 

		add_timer(&synopGMAC_cable_unplug_timer);
	}
	else{
	/*
		//synopGMAC_read_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,PHY_VSC8641_100BTX_EXT_STS, &data);
		//printk("\n ==PHY_VSC8641_100BTX_EXT_STS %x===", data);


		//for(i = 1; i < 32; i++){
		//synopGMAC_read_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,i, &data);
		//	printk("\n =%x=PHY_VSC8641_AN_ADV %x===", i, data);}

		//for(i = 1; i < 32; i++){
		//synopGMAC_read_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,i, &data);
		//	printk("\n =%x=PHY_VSC8641_AN_ADV %x===", i, data);}
	data = 1;
	synopGMAC_write_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase, 31, data);

	for(i = 16; i < 31; i++){
	synopGMAC_read_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,i, &data);
		printk("\n =%x=extern %x===", i, data);}

	data = 0;
	synopGMAC_write_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase, 31, data);
*/
	
		TR(" =timer check=Link UP: %08x\n",data);		
		if(!gmacdev->LinkState){
			status = synopGMAC_check_phy_init(gmacdev);
			synopGMAC_mac_init(gmacdev);
		}
		add_timer(&synopGMAC_cable_unplug_timer);
	}
	
}

static void synopGMAC_linux_powerdown_mac(synopGMACdevice *gmacdev)
{
	TR0("Put the GMAC to power down mode..\n");
	// Disable the Dma engines in tx path
	GMAC_Power_down = 1;	// Let ISR know that Mac is going to be in the power down mode
	synopGMAC_disable_dma_tx(gmacdev);
	plat_delay(10000);		//allow any pending transmission to complete
	// Disable the Mac for both tx and rx
	synopGMAC_tx_disable(gmacdev);
	synopGMAC_rx_disable(gmacdev);
        plat_delay(10000); 		//Allow any pending buffer to be read by host
	//Disable the Dma in rx path
        synopGMAC_disable_dma_rx(gmacdev);

	//enable the power down mode
	//synopGMAC_pmt_unicast_enable(gmacdev);
	
	//prepare the gmac for magic packet reception and wake up frame reception
	synopGMAC_magic_packet_enable(gmacdev);
	synopGMAC_write_wakeup_frame_register(gmacdev, synopGMAC_wakeup_filter_config3);

	synopGMAC_wakeup_frame_enable(gmacdev);

	//gate the application and transmit clock inputs to the code. This is not done in this driver :).

	//enable the Mac for reception
	synopGMAC_rx_enable(gmacdev);

	//Enable the assertion of PMT interrupt
	synopGMAC_pmt_int_enable(gmacdev);
	//enter the power down mode
	synopGMAC_power_down_enable(gmacdev);
	return;
}

static void synopGMAC_linux_powerup_mac(synopGMACdevice *gmacdev)
{
	GMAC_Power_down = 0;	// Let ISR know that MAC is out of power down now
	if( synopGMAC_is_magic_packet_received(gmacdev))
		TR("GMAC wokeup due to Magic Pkt Received\n");
	if(synopGMAC_is_wakeup_frame_received(gmacdev))
		TR("GMAC wokeup due to Wakeup Frame Received\n");
	//Disable the assertion of PMT interrupt
	synopGMAC_pmt_int_disable(gmacdev);
	//Enable the mac and Dma rx and tx paths
	synopGMAC_rx_enable(gmacdev);
       synopGMAC_enable_dma_rx(gmacdev);

	synopGMAC_tx_enable(gmacdev);
	synopGMAC_enable_dma_tx(gmacdev);
	return;
}

/**
  * This sets up the transmit Descriptor queue in ring or chain mode.
  * This function is tightly coupled to the platform and operating system
  * Device is interested only after the descriptors are setup. Therefore this function
  * is not included in the device driver API. This function should be treated as an
  * example code to design the descriptor structures for ring mode or chain mode.
  * This function depends on the pcidev structure for allocation consistent dma-able memory in case of linux.
  * This limitation is due to the fact that linux uses pci structure to allocate a dmable memory
  *	- Allocates the memory for the descriptors.
  *	- Initialize the Busy and Next descriptors indices to 0(Indicating first descriptor).
  *	- Initialize the Busy and Next descriptors to first descriptor address.
  * 	- Initialize the last descriptor with the endof ring in case of ring mode.
  *	- Initialize the descriptors in chain mode.
  * @param[in] pointer to synopGMACdevice.
  * @param[in] pointer to pci_device structure.
  * @param[in] number of descriptor expected in tx descriptor queue.
  * @param[in] whether descriptors to be created in RING mode or CHAIN mode.
  * \return 0 upon success. Error code upon failure.
  * \note This function fails if allocation fails for required number of descriptors in Ring mode, but in chain mode
  *  function returns -ESYNOPGMACNOMEM in the process of descriptor chain creation. once returned from this function
  *  user should for gmacdev->TxDescCount to see how many descriptors are there in the chain. Should continue further
  *  only if the number of descriptors in the chain meets the requirements  
  */
s32 synopGMAC_setup_tx_desc_queue(synopGMACdevice * gmacdev,struct pci_dev * pcidev,u32 no_of_desc, u32 desc_mode)
{
s32 i;

DmaDesc *first_desc = NULL;
DmaDesc *second_desc = NULL;
dma_addr_t dma_addr;
gmacdev->TxDescCount = 0;

if(desc_mode == RINGMODE){
	TR("Total size of memory required for Tx Descriptors in Ring Mode = 0x%08x\n",((sizeof(DmaDesc) * no_of_desc)));
	first_desc = plat_alloc_consistent_dmaable_memory (pcidev, sizeof(DmaDesc) * no_of_desc,&dma_addr);
	if(first_desc == NULL){
		TR("Error in Tx Descriptors memory allocation\n");
		return -ESYNOPGMACNOMEM;
	}
	
	gmacdev->TxDescCount = no_of_desc;
	gmacdev->TxDesc      = first_desc;
	gmacdev->TxDescDma   = dma_addr;
	
	for(i =0; i < gmacdev -> TxDescCount; i++){
		synopGMAC_desc_init_ring(gmacdev->TxDesc + i, i == gmacdev->TxDescCount-1);
	}

}
else{
//Allocate the head descriptor
	first_desc = plat_alloc_consistent_dmaable_memory (pcidev, sizeof(DmaDesc),&dma_addr);
	if(first_desc == NULL){
		TR("Error in Tx Descriptor Memory allocation in Ring mode\n");
		return -ESYNOPGMACNOMEM;
	}
	gmacdev->TxDesc       = first_desc;
	gmacdev->TxDescDma    = dma_addr;

	TR("Tx===================================================================Tx\n");
	first_desc->buffer2   = gmacdev->TxDescDma;
 	first_desc->data2     = (u32)gmacdev->TxDesc;

	gmacdev->TxDescCount = 1;
	
	for(i =0; i <(no_of_desc-1); i++){
		second_desc = plat_alloc_consistent_dmaable_memory(pcidev, sizeof(DmaDesc),&dma_addr);
		if(second_desc == NULL){	
			TR("Error in Tx Descriptor Memory allocation in Chain mode\n");
			return -ESYNOPGMACNOMEM;
		}
		first_desc->buffer2  = dma_addr;
		first_desc->data2    = (u32)second_desc;

		/*Ϊʵѭ*/
		second_desc->buffer2 = gmacdev->TxDescDma;
		second_desc->data2   = (u32)gmacdev->TxDesc;
		
		synopGMAC_desc_init_chain(first_desc);
		TR("%02d %08x %08x %08x %08x %08x %08x %08x \n",gmacdev->TxDescCount, (u32)first_desc, first_desc->status, first_desc->length, first_desc->buffer1,first_desc->buffer2, first_desc->data1, first_desc->data2);
		gmacdev->TxDescCount += 1;
		first_desc = second_desc;
	}
		synopGMAC_desc_init_chain(first_desc);
		TR("%02d %08x %08x %08x %08x %08x %08x %08x \n",gmacdev->TxDescCount, (u32)first_desc, first_desc->status, first_desc->length, first_desc->buffer1,first_desc->buffer2, first_desc->data1, first_desc->data2);
	TR("Tx===================================================================Tx\n");
}

	gmacdev->TxNext = 0;
	gmacdev->TxBusy = 0;
	gmacdev->TxNextDesc = gmacdev->TxDesc;
	gmacdev->TxBusyDesc = gmacdev->TxDesc;

	return -ESYNOPGMACNOERR;
}


/**
  * This sets up the receive Descriptor queue in ring or chain mode.
  * This function is tightly coupled to the platform and operating system
  * Device is interested only after the descriptors are setup. Therefore this function
  * is not included in the device driver API. This function should be treated as an
  * example code to design the descriptor structures in ring mode or chain mode.
  * This function depends on the pcidev structure for allocation of consistent dma-able memory in case of linux.
  * This limitation is due to the fact that linux uses pci structure to allocate a dmable memory
  *	- Allocates the memory for the descriptors.
  *	- Initialize the Busy and Next descriptors indices to 0(Indicating first descriptor).
  *	- Initialize the Busy and Next descriptors to first descriptor address.
  * 	- Initialize the last descriptor with the endof ring in case of ring mode.
  *	- Initialize the descriptors in chain mode.
  * @param[in] pointer to synopGMACdevice.
  * @param[in] pointer to pci_device structure.
  * @param[in] number of descriptor expected in rx descriptor queue.
  * @param[in] whether descriptors to be created in RING mode or CHAIN mode.
  * \return 0 upon success. Error code upon failure.
  * \note This function fails if allocation fails for required number of descriptors in Ring mode, but in chain mode
  *  function returns -ESYNOPGMACNOMEM in the process of descriptor chain creation. once returned from this function
  *  user should for gmacdev->RxDescCount to see how many descriptors are there in the chain. Should continue further
  *  only if the number of descriptors in the chain meets the requirements  
  */
s32 synopGMAC_setup_rx_desc_queue(synopGMACdevice * gmacdev,struct pci_dev * pcidev,u32 no_of_desc, u32 desc_mode)
{
s32 i;

DmaDesc *first_desc = NULL;
DmaDesc *second_desc = NULL;
dma_addr_t dma_addr;
gmacdev->RxDescCount = 0;

if(desc_mode == RINGMODE){
	TR("total size of memory required for Rx Descriptors in Ring Mode = 0x%08x\n",((sizeof(DmaDesc) * no_of_desc)));
	first_desc = plat_alloc_consistent_dmaable_memory (pcidev, sizeof(DmaDesc) * no_of_desc, &dma_addr);
	if(first_desc == NULL){
		TR("Error in Rx Descriptor Memory allocation in Ring mode\n");
		return -ESYNOPGMACNOMEM;
	}
	gmacdev->RxDescCount = no_of_desc;
	gmacdev->RxDesc      = first_desc;
	gmacdev->RxDescDma   = dma_addr;
	
	for(i =0; i < gmacdev -> RxDescCount; i++){
		synopGMAC_desc_init_ring(gmacdev->RxDesc + i, i == gmacdev->RxDescCount-1);
	}

}
else{
//Allocate the head descriptor
	first_desc = plat_alloc_consistent_dmaable_memory (pcidev, sizeof(DmaDesc),&dma_addr);
	if(first_desc == NULL){
		TR("Error in Rx Descriptor Memory allocation in Ring mode\n");
		return -ESYNOPGMACNOMEM;
	}
	gmacdev->RxDesc       = first_desc;
	gmacdev->RxDescDma    = dma_addr;

	TR("Rx===================================================================Rx\n");
	first_desc->buffer2   = gmacdev->RxDescDma;
	first_desc->data2     = (u32) gmacdev->RxDesc;

	gmacdev->RxDescCount = 1;
	
	for(i =0; i < (no_of_desc-1); i++){
		second_desc = plat_alloc_consistent_dmaable_memory(pcidev, sizeof(DmaDesc),&dma_addr);
		if(second_desc == NULL){	
			TR("Error in Rx Descriptor Memory allocation in Chain mode\n");
			return -ESYNOPGMACNOMEM;
		}
		first_desc->buffer2  = dma_addr;
		first_desc->data2    = (u32)second_desc;
		
		second_desc->buffer2 = gmacdev->RxDescDma;
		second_desc->data2   = (u32)gmacdev->RxDesc;
		
		synopGMAC_desc_init_chain(first_desc);
		TR("%02d  %08x %08x %08x %08x %08x %08x %08x \n",gmacdev->RxDescCount, (u32)first_desc, first_desc->status, first_desc->length, first_desc->buffer1,first_desc->buffer2, first_desc->data1, first_desc->data2);
		gmacdev->RxDescCount += 1;
		first_desc = second_desc;
	}
		synopGMAC_desc_init_chain(first_desc);
		TR("%02d  %08x %08x %08x %08x %08x %08x %08x \n",gmacdev->RxDescCount, (u32)first_desc, first_desc->status, first_desc->length, first_desc->buffer1,first_desc->buffer2, first_desc->data1, first_desc->data2);
	TR("Rx===================================================================Rx\n");

}

	gmacdev->RxNext = 0;
	gmacdev->RxBusy = 0;
	gmacdev->RxNextDesc = gmacdev->RxDesc;
	gmacdev->RxBusyDesc = gmacdev->RxDesc;

	return -ESYNOPGMACNOERR;
}

/**
  * This gives up the receive Descriptor queue in ring or chain mode.
  * This function is tightly coupled to the platform and operating system
  * Once device's Dma is stopped the memory descriptor memory and the buffer memory deallocation,
  * is completely handled by the operating system, this call is kept outside the device driver Api.
  * This function should be treated as an example code to de-allocate the descriptor structures in ring mode or chain mode
  * and network buffer deallocation.
  * This function depends on the pcidev structure for dma-able memory deallocation for both descriptor memory and the
  * network buffer memory under linux.
  * The responsibility of this function is to 
  *     - Free the network buffer memory if any.
  *	- Fee the memory allocated for the descriptors.
  * @param[in] pointer to synopGMACdevice.
  * @param[in] pointer to pci_device structure.
  * @param[in] number of descriptor expected in rx descriptor queue.
  * @param[in] whether descriptors to be created in RING mode or CHAIN mode.
  * \return 0 upon success. Error code upon failure.
  * \note No referece should be made to descriptors once this function is called. This function is invoked when the device is closed.
  */
void synopGMAC_giveup_rx_desc_queue(synopGMACdevice * gmacdev, struct pci_dev *pcidev, u32 desc_mode)
{
s32 i;

DmaDesc *first_desc = NULL;
dma_addr_t first_desc_dma_addr;
u32 status;
dma_addr_t dma_addr1;
dma_addr_t dma_addr2;
u32 length1;
u32 length2;
u32 data1;
u32 data2;

if(desc_mode == RINGMODE){
	for(i =0; i < gmacdev -> RxDescCount; i++){
		synopGMAC_get_desc_data(gmacdev->RxDesc + i, &status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
		if((length1 != 0) && (data1 != 0)){
			//--pci_unmap_single(pcidev,dma_addr1,0,PCI_DMA_FROMDEVICE);
			dev_kfree_skb((struct sk_buff *) data1);	// free buffer1
			TR("(Ring mode) rx buffer1 %08x of size %d from %d rx descriptor is given back\n",data1, length1, i);
		}
		if((length2 != 0) && (data2 != 0)){
			//--pci_unmap_single(pcidev,dma_addr2,0,PCI_DMA_FROMDEVICE);
			dev_kfree_skb((struct sk_buff *) data2);	//free buffer2
			TR("(Ring mode) rx buffer2 %08x of size %d from %d rx descriptor is given back\n",data2, length2, i);
		}
	}
	plat_free_consistent_dmaable_memory(pcidev,(sizeof(DmaDesc) * gmacdev->RxDescCount),gmacdev->RxDesc,gmacdev->RxDescDma); //free descriptors memory
	TR("Memory allocated %08x  for Rx Desriptors (ring) is given back\n",(u32)gmacdev->RxDesc);
}
else{
	TR("rx-------------------------------------------------------------------rx\n");
	first_desc          = gmacdev->RxDesc;
	first_desc_dma_addr = gmacdev->RxDescDma;
	for(i =0; i < gmacdev -> RxDescCount; i++){
		synopGMAC_get_desc_data(first_desc, &status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
		TR("%02d %08x %08x %08x %08x %08x %08x %08x\n",i,(u32)first_desc,first_desc->status,first_desc->length,first_desc->buffer1,first_desc->buffer2,first_desc->data1,first_desc->data2);
		if((length1 != 0) && (data1 != 0)){
			//--pci_unmap_single(pcidev,dma_addr1,0,PCI_DMA_FROMDEVICE);
			dev_kfree_skb((struct sk_buff *) data1);	// free buffer1
			TR("(Chain mode) rx buffer1 %08x of size %d from %d rx descriptor is given back\n",data1, length1, i);
		}
		plat_free_consistent_dmaable_memory(pcidev,(sizeof(DmaDesc)),first_desc,first_desc_dma_addr); //free descriptors
		TR("Memory allocated %08x for Rx Descriptor (chain) at  %d is given back\n",data2,i);

		first_desc = (DmaDesc *)data2;
		first_desc_dma_addr = dma_addr2;
	}

	TR("rx-------------------------------------------------------------------rx\n");
}
gmacdev->RxDesc    = NULL;
gmacdev->RxDescDma = 0;
return;
}

/**
  * This gives up the transmit Descriptor queue in ring or chain mode.
  * This function is tightly coupled to the platform and operating system
  * Once device's Dma is stopped the memory descriptor memory and the buffer memory deallocation,
  * is completely handled by the operating system, this call is kept outside the device driver Api.
  * This function should be treated as an example code to de-allocate the descriptor structures in ring mode or chain mode
  * and network buffer deallocation.
  * This function depends on the pcidev structure for dma-able memory deallocation for both descriptor memory and the
  * network buffer memory under linux.
  * The responsibility of this function is to 
  *     - Free the network buffer memory if any.
  *	- Fee the memory allocated for the descriptors.
  * @param[in] pointer to synopGMACdevice.
  * @param[in] pointer to pci_device structure.
  * @param[in] number of descriptor expected in tx descriptor queue.
  * @param[in] whether descriptors to be created in RING mode or CHAIN mode.
  * \return 0 upon success. Error code upon failure.
  * \note No reference should be made to descriptors once this function is called. This function is invoked when the device is closed.
  */
void synopGMAC_giveup_tx_desc_queue(synopGMACdevice * gmacdev,struct pci_dev * pcidev, u32 desc_mode)
{
s32 i;

DmaDesc *first_desc = NULL;
dma_addr_t first_desc_dma_addr;
u32 status;
dma_addr_t dma_addr1;
dma_addr_t dma_addr2;
u32 length1;
u32 length2;
u32 data1;
u32 data2;

if(desc_mode == RINGMODE){
	for(i =0; i < gmacdev -> TxDescCount; i++){
		synopGMAC_get_desc_data(gmacdev->TxDesc + i,&status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
		if((length1 != 0) && (data1 != 0)){
			//--pci_unmap_single(pcidev,dma_addr1,0,PCI_DMA_TODEVICE);
			dev_kfree_skb((struct sk_buff *) data1);	// free buffer1
			TR("(Ring mode) tx buffer1 %08x of size %d from %d rx descriptor is given back\n",data1, length1, i);
		}
		if((length2 != 0) && (data2 != 0)){
			//--pci_unmap_single(pcidev,dma_addr2,0,PCI_DMA_TODEVICE);
			dev_kfree_skb((struct sk_buff *) data2);	//free buffer2
			TR("(Ring mode) tx buffer2 %08x of size %d from %d rx descriptor is given back\n",data2, length2, i);
		}
	}
	plat_free_consistent_dmaable_memory(pcidev,(sizeof(DmaDesc) * gmacdev->TxDescCount),gmacdev->TxDesc,gmacdev->TxDescDma); //free descriptors
	TR("Memory allocated %08x for Tx Desriptors (ring) is given back\n",(u32)gmacdev->TxDesc);
}
else{
	TR("tx-------------------------------------------------------------------tx\n");
	first_desc          = gmacdev->TxDesc;
	first_desc_dma_addr = gmacdev->TxDescDma;
	for(i =0; i < gmacdev -> TxDescCount; i++){
		synopGMAC_get_desc_data(first_desc, &status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
		TR("%02d %08x %08x %08x %08x %08x %08x %08x\n",i,(u32)first_desc,first_desc->status,first_desc->length,first_desc->buffer1,first_desc->buffer2,first_desc->data1,first_desc->data2);
		if((length1 != 0) && (data1 != 0)){
			//--pci_unmap_single(pcidev,dma_addr1,0,PCI_DMA_TODEVICE);
			dev_kfree_skb((struct sk_buff *) data2);	// free buffer1
			TR("(Chain mode) tx buffer1 %08x of size %d from %d rx descriptor is given back\n",data1, length1, i);
		}
		plat_free_consistent_dmaable_memory(pcidev,(sizeof(DmaDesc)),first_desc,first_desc_dma_addr); //free descriptors
		TR("Memory allocated %08x for Tx Descriptor (chain) at  %d is given back\n",data2,i);

		first_desc = (DmaDesc *)data2;
		first_desc_dma_addr = dma_addr2;
	}
	TR("tx-------------------------------------------------------------------tx\n");

}
gmacdev->TxDesc    = NULL;
gmacdev->TxDescDma = 0;
return;
}


/**
 * Function to handle housekeeping after a packet is transmitted over the wire.
 * After the transmission of a packet DMA generates corresponding interrupt 
 * (if it is enabled). It takes care of returning the sk_buff to the linux
 * kernel, updating the networking statistics and tracking the descriptors.
 * @param[in] pointer to net_device structure. 
 * \return void.
 * \note This function runs in interrupt context
 */
 //ݷɴжϴе
void synop_handle_transmit_over(struct net_device *netdev)
{
	synopGMACPciNetworkAdapter *adapter;
	synopGMACdevice * gmacdev;
	struct pci_dev *pcidev;
	s32 desc_index;
	u32 data1, data2;
	u32 status;
	u32 length1, length2;
	u32 dma_addr1, dma_addr2;

	adapter = netdev->priv;
	if(adapter == NULL){
		TR("Unknown Device\n");
		return;
	}
	
	gmacdev = adapter->synopGMACdev;
	if(gmacdev == NULL){
		TR("GMAC device structure is missing\n");
		return;
	}

 	pcidev  = (struct pci_dev *)adapter->synopGMACpcidev;	
	/*Handle the transmit Descriptors*/
	do {
		desc_index = synopGMAC_get_tx_qptr(gmacdev, &status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
		//desc_index = synopGMAC_get_tx_qptr(gmacdev, &status, &dma_addr, &length, &data1);
		if(desc_index >= 0 && data1 != 0){
			TR("Finished Transmit at Tx Descriptor %d for skb 0x%08x and buffer = %08x whose status is %08x \n", desc_index,data1,dma_addr1,status);
			#ifdef	IPC_OFFLOAD
			if(synopGMAC_is_tx_ipv4header_checksum_error(gmacdev, status)){
			TR0("Harware Failed to Insert IPV4 Header Checksum\n");
			}
			if(synopGMAC_is_tx_payload_checksum_error(gmacdev, status)){
			TR0("Harware Failed to Insert Payload Checksum\n");
			}
			#endif			

			//--pci_unmap_single(pcidev,dma_addr1,length1,PCI_DMA_TODEVICE); 
			dev_kfree_skb_irq((struct sk_buff *)data1);
			
			if(synopGMAC_is_desc_valid(status)){
				adapter->synopGMACNetStats.tx_bytes += length1;
				adapter->synopGMACNetStats.tx_packets++;
			}
			else {	
				TR("Error in Status %08x\n",status);
				adapter->synopGMACNetStats.tx_errors++;
				adapter->synopGMACNetStats.tx_aborted_errors += synopGMAC_is_tx_aborted(status);
				adapter->synopGMACNetStats.tx_carrier_errors += synopGMAC_is_tx_carrier_error(status);
			}
		}	

		adapter->synopGMACNetStats.collisions += synopGMAC_get_tx_collision_count(status);

	} while(desc_index >= 0);



	
	netif_wake_queue(netdev);


	*(pDbgMem+2) = jiffies;
	//printk("\n=drv icmp tx over jiffies %d=", *(pDbgMem+2));

	
}


/**
 * Function to Receive a packet from the interface.
 * After Receiving a packet, DMA transfers the received packet to the system memory
 * and generates corresponding interrupt (if it is enabled). This function prepares
 * the sk_buff for received packet after removing the ethernet CRC, and hands it over
 * to linux networking stack.
 * 	- Updataes the networking interface statistics
 *	- Keeps track of the rx descriptors
 * @param[in] pointer to net_device structure. 
 * \return void.
 * \note This function runs in interrupt context.
 */
//ݵĺжϴе
void synop_handle_received_data(struct net_device *netdev)
{
	synopGMACPciNetworkAdapter *adapter;
	synopGMACdevice * gmacdev;
	struct pci_dev *pcidev;
	s32 desc_index;
	
	u32 data1;
	u32 data2;
	u32 len;
	u32 status;
	u32 dma_addr1;
	u32 dma_addr2;

    
	struct sk_buff *skb; //This is the pointer to hold the received data
	
	TR("%s\n",__FUNCTION__);	
	
	adapter = netdev->priv;
	if(adapter == NULL){
		TR("Unknown Device\n");
		return;
	}
	
	gmacdev = adapter->synopGMACdev;
	if(gmacdev == NULL){
		TR("GMAC device structure is missing\n");
		return;
	}

 	pcidev  = (struct pci_dev *)adapter->synopGMACpcidev;	
	/*Handle the Receive Descriptors*/
	do{
		desc_index = synopGMAC_get_rx_qptr(gmacdev, &status,&dma_addr1,NULL, &data1,&dma_addr2,NULL,&data2);
		
		if(desc_index >= 0 && data1 != 0){
			TR("Received Data at Rx Descriptor %d for skb 0x%08x whose status is %08x\n",desc_index,data1,status);
			
			//--/*At first step unmapped the dma address*/
			//--pci_unmap_single(pcidev,dma_addr1,0,PCI_DMA_FROMDEVICE);

			skb = (struct sk_buff *)data1;
			if(synopGMAC_is_rx_desc_valid(status)){				
				len =  synopGMAC_get_rx_desc_frame_length(status) - 4; //Not interested in Ethernet CRC bytes
				skb_put(skb,len);

				//debug
				if (*(skb->data+23) == 1) 
				{
					*(pDbgMem+3) = jiffies ;
					//printk("\n=drv icmp rx jiffies %d  %d=", jiffies, skb->nh.iph->protocol);
				}


				#if 0
				if (*(skb->data+0)==0x00 && *(skb->data+1)==0x80 &&*(skb->data+2)==0x00 
					&&*(skb->data+3)==0x81 &&*(skb->data+4)==0x00 &&*(skb->data+5)==0x33)
				{
						int counter;
					
						printk("==synop_handle_received_data\n");
						
						printk("DST MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+0),*(skb->data+1),*(skb->data+2),*(skb->data+3),*(skb->data+4),*(skb->data+5));
						printk("SRC MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+6),*(skb->data+7),*(skb->data+8),*(skb->data+9),*(skb->data+10),*(skb->data+11));
						printk("Len/type    :%02x %02x\n",*(skb->data+12),*(skb->data+13));


                                if (*(skb->data+12) == 0x08 && *(skb->data+13) == 0x06){
                                        printk("h %02x %02x p %02x %02x\n",*(skb->data+14),*(skb->data+15),*(skb->data+16),*(skb->data+17));
                                        printk("hL %02x PL %02x O:%02x %02x\n",*(skb->data+18),*(skb->data+19),*(skb->data+20),*(skb->data+21));
                                        printk("SHA %02x %02x %02x %02x %02x %02x\n",*(skb->data+22),*(skb->data+23),*(skb->data+24),*(skb->data+25), *(skb->data+26),*(skb->data+27));
                                        printk("SIA %02x %02x %02x %02x\n",*(skb->data+28),*(skb->data+29),*(skb->data+30),*(skb->data+31));
                                        printk("THA:%02x %02x %02x %02x %02x %02x\n",*(skb->data+32),*(skb->data+33),*(skb->data+34),*(skb->data+35),*(skb->data+36),*(skb->data+37));
                                        printk("TIA %02x %02x %02x %02x\n",*(skb->data+38),*(skb->data+39),*(skb->data+40),*(skb->data+41));
                                }
                                else
                                {
						//debug
						if (*(skb->data+23) == 1) 
						{
							*(pDbgMem+3) = jiffies ;
							printk("\n=drv icmp rx jiffies %d  %d=", jiffies, skb->nh.iph->protocol);
						}

                                    printk(KERN_CRIT"IPV4 Header:\n");
                                    printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+14),*(skb->data+15),*(skb->data+16),*(skb->data+17));
                                    printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+18),*(skb->data+19),*(skb->data+20),*(skb->data+21));
                                    printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+22),*(skb->data+23),*(skb->data+24),*(skb->data+25));
                                    printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+26),*(skb->data+27),*(skb->data+28),*(skb->data+29));
                                    printk(KERN_CRIT"%02x %02x %02x %02x\n\n",*(skb->data+30),*(skb->data+31),*(skb->data+32),*(skb->data+33));
                                    for(counter = 34; counter < skb->len; counter++)
                                        printk("%02X ",*(skb->data + counter));

                                        
                                }
				}
				#endif

			
				#ifdef IPC_OFFLOAD
				// Now lets check for the IPC offloading
				/*  Since we have enabled the checksum offloading in hardware, lets inform the kernel
				    not to perform the checksum computation on the incoming packet. Note that ip header 
  				    checksum will be computed by the kernel immaterial of what we inform. Similary TCP/UDP/ICMP
				    pseudo header checksum will be computed by the stack. What we can inform is not to perform
				    payload checksum. 		
   				    When CHECKSUM_UNNECESSARY is set kernel bypasses the checksum computation.		    
				*/
				TR("Checksum Offloading will be done now\n");
				skb->ip_summed = CHECKSUM_UNNECESSARY;

				//Linux Kernel doesnot care for ipv4 header checksum. So we will simply proceed by printing a warning ....
			      if(synopGMAC_is_rx_ipv4header_checksum_error(gmacdev, status)){
					TR("Error in 16bit IPv4 Header Checksum\n");
				}
				
				// Control should not come here if we allow DMA to drop the packets if error
				if(synopGMAC_is_rx_payload_checksum_error(gmacdev, status)){
        				TR0("Error in Payload Checksum Error %x\n", status);
                			/* In case control is here tell the networking stack to recompute the checksum since hardware reported
                				   some issues.   This is done by informing the stack saying Checksum is not cont computed
                  				   and CHECKSUM_NONE is set.
                		*/
        				skb->ip_summed = CHECKSUM_NONE;
				}
				#endif
				
				skb->dev = netdev;
				skb->protocol = eth_type_trans(skb, netdev);
				netif_rx(skb);

				netdev->last_rx = jiffies;
				adapter->synopGMACNetStats.rx_packets++;
				adapter->synopGMACNetStats.rx_bytes += len;
			}
			else{
				adapter->synopGMACNetStats.rx_errors++;
				adapter->synopGMACNetStats.collisions       += synopGMAC_is_rx_frame_collision(status);
				adapter->synopGMACNetStats.rx_crc_errors    += synopGMAC_is_rx_crc(status);
				adapter->synopGMACNetStats.rx_frame_errors  += synopGMAC_is_frame_dribbling_errors(status);
				adapter->synopGMACNetStats.rx_length_errors += synopGMAC_is_rx_frame_length_errors(status);
			
				//add by phosor 
				printk("\n =========synopGMAC_is_rx_desc_ not valid=============");
				dev_kfree_skb_irq(skb);
			}
			
			//Now lets allocate the skb for the emptied descriptor
			
			skb = dev_alloc_skb(PKT_BUF_SIZE);
			if(skb == NULL){
				TR("SKB memory allocation failed \n");
				adapter->synopGMACNetStats.rx_dropped++;
			}
			skb->dev = netdev;
			skb_reserve(skb,ETHERNET_EXTRA);	


			//--dma_addr1 = pci_map_single(pcidev,skb->data,skb_tailroom(skb),PCI_DMA_FROMDEVICE);
			dma_addr1 = skb->data;
			
			desc_index = synopGMAC_set_rx_qptr(gmacdev,dma_addr1, skb_tailroom(skb), (u32)skb,0,0,0);
			//desc_index = synopGMAC_set_rx_qptr(gmacdev,dma_addr ,skb_tailroom(skb),(u32)skb);
			if(desc_index < 0){
				TR("Cannot set Rx Descriptor for skb %08x\n",(u32)skb);
				dev_kfree_skb_irq(skb);
			}
					
		}
	}while(desc_index >= 0);
}


/**жϴ
 * Interrupt service routing.
 * This is the function registered as ISR for device interrupts.
 * @param[in] interrupt number. 
 * @param[in] void pointer to device unique structure (Required for shared interrupts in Linux).
 * @param[in] pointer to pt_regs (not used).
 * \return Returns IRQ_NONE if not device interrupts IRQ_HANDLED for device interrupts.
 * \note This function runs in interrupt context
 *
 */
#define NET_DBG
//#undef NET_DBG
#ifndef NET_DBG
irqreturn_t synopGMAC_intr_handler(s32 intr_num, void * dev_id, struct pt_regs *regs)
#else
static void synopGMAC_intr_handler(s32 intr_num, void * dev_id, struct pt_regs *regs)
#endif
{       
    
    
    
	/*Kernels passes the netdev structure in the dev_id. So grab it*/
        struct net_device *netdev;
        synopGMACPciNetworkAdapter *adapter;
        synopGMACdevice * gmacdev;
	struct pci_dev *pcidev;
        u32 interrupt;
	s32 status;
	u32 dma_addr;
	//char tmp;

    //tmp = 0x00;
   // printk("%c", tmp);
_net_st:

        netdev  = (struct net_device *) dev_id;
        if(netdev == NULL){
                TR("Unknown Device\n");
                #ifndef NET_DBG
                    return -1;
                #else
                    return;
                #endif
        }

        adapter  = netdev->priv;
        if(adapter == NULL){
                TR("Adapter Structure Missing\n");
                #ifndef NET_DBG
                    return -1;
                #else
                    return;
                #endif
        }

        gmacdev = adapter->synopGMACdev;
        if(gmacdev == NULL){
                TR("GMAC device structure Missing\n");
                #ifndef NET_DBG
                    return -1;
                #else
                    return;
                #endif
        }

	pcidev  = (struct pci_dev *)adapter->synopGMACpcidev;	

	/*Read the Dma interrupt status to know whether the interrupt got generated by our device or not*/
	interrupt = synopGMACReadReg((u32 *)gmacdev->DmaBase, DmaStatus);
	
	if(interrupt == 0)
    {
        #ifndef NET_DBG
            return IRQ_NONE;
        #else
            return;
        #endif
    }

        synopGMAC_disable_interrupt_all(gmacdev);

	
	if(interrupt & GmacPmtIntr){
		TR("%s:: Interrupt due to PMT module\n",__FUNCTION__);
		synopGMAC_linux_powerup_mac(gmacdev);
	}
	
	if(interrupt & GmacMmcIntr){
		TR("%s:: Interrupt due to MMC module\n",__FUNCTION__);
		TR("%s:: synopGMAC_rx_int_status = %08x\n",__FUNCTION__,synopGMAC_read_mmc_rx_int_status(gmacdev));
		TR("%s:: synopGMAC_tx_int_status = %08x\n",__FUNCTION__,synopGMAC_read_mmc_tx_int_status(gmacdev));
	}

	if(interrupt & GmacLineIntfIntr){
		TR("%s:: Interrupt due to GMAC LINE module\n",__FUNCTION__);
	}


	/*Now lets handle the DMA interrupts*/  
        interrupt = synopGMAC_get_interrupt_type(gmacdev);

        if(interrupt & synopGMACDmaError){
		TR("%s::Fatal Bus Error Inetrrupt Seen\n",__FUNCTION__);
		synopGMAC_disable_dma_tx(gmacdev);
                synopGMAC_disable_dma_rx(gmacdev);
                
		synopGMAC_take_desc_ownership_tx(gmacdev);
		synopGMAC_take_desc_ownership_rx(gmacdev);
		
		synopGMAC_init_tx_rx_desc_queue(gmacdev);
		
		synopGMAC_reset(gmacdev);//reset the DMA engine and the GMAC ip
		
		synopGMAC_dma_bus_mode_init(gmacdev,DmaFixedBurstEnable| DmaBurstLength8 | DmaDescriptorSkip2);
	 	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward);	
		synopGMAC_init_rx_desc_base(gmacdev);
		synopGMAC_init_tx_desc_base(gmacdev);
		synopGMAC_mac_init(gmacdev);
		synopGMAC_enable_dma_rx(gmacdev);
		synopGMAC_enable_dma_tx(gmacdev);

        }

	//ȴ
	if(interrupt & synopGMACDmaRxNormal){	
		TR("%s:: Rx Normal \n", __FUNCTION__);
		synop_handle_received_data(netdev);
	}

        if(interrupt & synopGMACDmaRxAbnormal){
	        TR0("%s::Abnormal Rx Interrupt Seen\n",__FUNCTION__);
		#if 1
	       if(GMAC_Power_down == 0){	// If Mac is not in powerdown
                adapter->synopGMACNetStats.rx_over_errors++;
		synopGMAC_disable_dma_rx(gmacdev);
	        synopGMAC_take_desc_ownership_rx(gmacdev);
                synopGMAC_resume_dma_rx(gmacdev);
		}
		#endif
	}

        if(interrupt & synopGMACDmaRxStopped){
        	TR0("%s::Receiver stopped seeing Rx interrupts\n",__FUNCTION__); //Receiver gone in to stopped state
		#if 1
	        if(GMAC_Power_down == 0){	// If Mac is not in powerdown
		adapter->synopGMACNetStats.rx_over_errors++;
		do{
			struct sk_buff *skb = dev_alloc_skb(PKT_BUF_SIZE);
			if(skb == NULL){
				TR("%s::ERROR in skb buffer allocation Better Luck Next time\n",__FUNCTION__);
				break;
				//			return -ESYNOPGMACNOMEM;
			}
			skb->dev = netdev;
			skb_reserve(skb, ETHERNET_EXTRA);		

			//--dma_addr = pci_map_single(pcidev,skb->data,skb_tailroom(skb),PCI_DMA_FROMDEVICE);
			dma_addr = skb->data;
			status = synopGMAC_set_rx_qptr(gmacdev,dma_addr, skb_tailroom(skb), (u32)skb,0,0,0);
			//status = synopGMAC_set_rx_qptr(gmacdev,dma_addr, skb_tailroom(skb), (u32)skb);
			TR("%s::Set Rx Descriptor no %08x for skb %08x \n",__FUNCTION__,status,(u32)skb);
			if(status < 0)
				dev_kfree_skb_irq(skb);
		
		}while(status >= 0);
		
		synopGMAC_enable_dma_rx(gmacdev);
		}
		#endif
	}

	//ٴ
	//spin_lock(&gmacdev->lock);
	
	if(interrupt & synopGMACDmaTxNormal){
		//xmit function has done its job
		TR("%s::Finished Normal Transmission \n",__FUNCTION__);
                synop_handle_transmit_over(netdev);//Do whatever you want after the transmission is over
	}

        if(interrupt & synopGMACDmaTxAbnormal){
		TR0("%s::Abnormal Tx Interrupt Seen\n",__FUNCTION__);
		#if 1
	       if(GMAC_Power_down == 0){	// If Mac is not in powerdown
                synop_handle_transmit_over(netdev);
		}
		#endif
	}

	if(interrupt & synopGMACDmaTxStopped){
		TR0("%s::Transmitter stopped sending the packets\n",__FUNCTION__);
		#if 1
	       if(GMAC_Power_down == 0){	// If Mac is not in powerdown
		synopGMAC_disable_dma_tx(gmacdev);
                synopGMAC_take_desc_ownership_tx(gmacdev);
		
		synopGMAC_enable_dma_tx(gmacdev);
		TR("%s::Transmission Resumed\n",__FUNCTION__);
		}
		#endif
	}
	
	//spin_unlock(&gmacdev->lock);


	TR("%s::Clearing the interrupts (Still unservice interrupt: %08x)\n",__FUNCTION__,interrupt);
	synopGMAC_clear_interrupt(gmacdev);
        synopGMAC_enable_interrupt(gmacdev,DmaIntEnable);
        
        goto _net_st;

/*
    #ifndef NET_DBG
        return IRQ_HANDLED;
    #else
        return;
    #endif*/
}


/**
 * Function used when the interface is opened for use.
 * We register synopGMAC_linux_open function to linux open(). Basically this 
 * function prepares the the device for operation . This function is called whenever ifconfig (in Linux)
 * activates the device (for example "ifconfig eth0 up"). This function registers
 * system resources needed 
 * 	- Attaches device to device specific structure
 * 	- Programs the MDC clock for PHY configuration
 * 	- Check and initialize the PHY interface 
 *	- ISR registration
 * 	- Setup and initialize Tx and Rx descriptors
 *	- Initialize MAC and DMA
 *	- Allocate Memory for RX descriptors (The should be DMAable)
 * 	- Initialize one second timer to detect cable plug/unplug
 *	- Configure and Enable Interrupts
 *	- Enable Tx and Rx
 *	- start the Linux network queue interface
 * @param[in] pointer to net_device structure. 
 * \return Returns 0 on success and error status upon failure.
 * \callgraph
 */

s32 synopGMAC_linux_open(struct net_device *netdev)
{
	s32 status = 0;
	s32 retval = 0;
	s32 ijk;
	//s32 reserve_len=2;
	u32 dma_addr;
	struct sk_buff *skb;
        synopGMACPciNetworkAdapter *adapter;
        synopGMACdevice * gmacdev;
	struct pci_dev *pcidev;
	TR0("%s called \n",__FUNCTION__);
	adapter = (synopGMACPciNetworkAdapter *) netdev->priv;
	gmacdev = (synopGMACdevice *)adapter->synopGMACdev;
 	pcidev  = (struct pci_dev *)adapter->synopGMACpcidev;	
	
	/*Now platform dependent initialization.*/

	//spin_lock_init(&gmacdev->lock);//add by phosor

	/*Lets reset the IP*/
	TR("adapter= %08x gmacdev = %08x netdev = %08x pcidev= %08x\n",(u32)adapter,(u32)gmacdev,(u32)netdev,(u32)pcidev);
	synopGMAC_reset(gmacdev);
	/*Lets read the version of ip in to device structure*/	
	synopGMAC_read_version(gmacdev);
	
	/*Attach the device to MAC struct This will configure all the required base addresses
	  such as Mac base, configuration base, phy base address(out of 32 possible phys )*/
	synopGMAC_attach(synopGMACadapter->synopGMACdev,(u32) synopGMACMappedAddr + MACBASE,(u32) synopGMACMappedAddr + DMABASE, DEFAULT_PHY_BASE);
	synopGMAC_get_mac_addr(synopGMACadapter->synopGMACdev,GmacAddr0High,GmacAddr0Low, netdev->dev_addr); //ȡMACַ
	/*Now set the broadcast address*/	
	for(ijk = 0; ijk <6; ijk++){
		netdev->broadcast[ijk] = 0xff;
	}
	for(ijk = 0; ijk <6; ijk++){
		TR("netdev->dev_addr[%d] = %02x and netdev->broadcast[%d] = %02x\n",ijk,netdev->dev_addr[ijk],ijk,netdev->broadcast[ijk]);
	}
	
	/*Check for Phy initialization*/
	synopGMAC_set_mdc_clk_div(gmacdev,GmiiCsrClk3);
	gmacdev->ClockDivMdc = synopGMAC_get_mdc_clk_div(gmacdev);
	status = synopGMAC_check_phy_init(gmacdev);
	//synopGMAC_phy_loopback(gmacdev, 1); //add for debug
	
	
	/*Request for an shared interrupt. Instead of using netdev->irq lets use pcidev->irq*/
	if(request_irq (pcidev->irq, synopGMAC_intr_handler, 0, netdev->name,netdev)){//SA_SHIRQ | SA_INTERRUPT
 		TR0("Error in request_irq\n");
		goto error_in_irq;	
	}
	enable_irq(pcidev->irq);
 	TR("%s owns a shared interrupt on line %d\n",netdev->name, pcidev->irq);

	/*Set up the tx and rx descriptor queue/ring*/
	//synopGMAC_setup_tx_desc_queue(gmacdev,pcidev,TRANSMIT_DESC_SIZE, CHAINMODE);
	synopGMAC_setup_tx_desc_queue(gmacdev,pcidev,TRANSMIT_DESC_SIZE, RINGMODE);
	synopGMAC_init_tx_desc_base(gmacdev);	//Program the transmit descriptor base address in to DmaTxBase addr

	//synopGMAC_setup_rx_desc_queue(gmacdev,pcidev,RECEIVE_DESC_SIZE, CHAINMODE);
	synopGMAC_setup_rx_desc_queue(gmacdev,pcidev,RECEIVE_DESC_SIZE, RINGMODE);
	synopGMAC_init_rx_desc_base(gmacdev);	//Program the transmit descriptor base address in to DmaTxBase addr
	
	synopGMAC_dma_bus_mode_init(gmacdev,DmaFixedBurstEnable| DmaBurstLength8 | DmaDescriptorSkip2); 
	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward);	
	
	/*Initialize the mac interface*/
	synopGMAC_mac_init(gmacdev);
	#ifdef IPC_OFFLOAD
	/*IPC Checksum offloading is enabled for this driver. Should only be used if Full Ip checksumm offload engine is configured in the hardware*/
	synopGMAC_enable_rx_chksum_offload(gmacdev);  	//Enable the offload engine in the receive path

	//շУʹֱԵ
	synopGMAC_rx_tcpip_chksum_drop_enable(gmacdev); // This is default configuration, DMA drops the packets if error in encapsulated ethernet payload
							// The FEF bit in DMA control register is configured to 0 indicating DMA to drop the errored frames.
	/*Inform the Linux Networking stack about the hardware capability of checksum offloading*/
	netdev->features = NETIF_F_HW_CSUM;//Ŀǰں˲֧֣ҪʹHW CHECKSUMںЭջCHECKSUMش޸;
	#endif


     do{
		skb = dev_alloc_skb(PKT_BUF_SIZE);
		if(skb == NULL){
			TR0("ERROR in skb buffer allocation\n");
			return -ESYNOPGMACNOMEM;
		}
		skb->dev = netdev;
		skb_reserve(skb, ETHERNET_EXTRA);
		TR("skb = %08x skb->tail = %08x skb_tailroom(skb)=%08x skb->data = %08x\n",(u32)skb,(u32)skb->tail,(skb_tailroom(skb)),(u32)skb->data);

		//--dma_addr = pci_map_single(pcidev,skb->data,skb_tailroom(skb),PCI_DMA_FROMDEVICE);
		dma_addr = skb->data;
		status = synopGMAC_set_rx_qptr(gmacdev,dma_addr, skb_tailroom(skb), (u32)skb,0,0,0);
		if(status < 0)
			dev_kfree_skb(skb);
			
	}while(status >= 0);
	

	TR("Setting up the cable unplug timer\n");
	init_timer(&synopGMAC_cable_unplug_timer);
	synopGMAC_cable_unplug_timer.function = (void *)synopGMAC_linux_cable_unplug_function;
	synopGMAC_cable_unplug_timer.data = (u32) adapter;
	synopGMAC_cable_unplug_timer.expires = CHECK_TIME + jiffies;
	add_timer(&synopGMAC_cable_unplug_timer);

	synopGMAC_clear_interrupt(gmacdev);
	synopGMAC_enable_interrupt(gmacdev,DmaIntEnable);
	synopGMAC_enable_dma_rx(gmacdev);
	synopGMAC_enable_dma_tx(gmacdev);
	
	netif_start_queue(netdev);

	return retval;

error_in_irq:
	/*Lets free the allocated memory*/
	plat_free_memory(gmacdev);
	return -ESYNOPGMACBUSY;
}

/**
 * Function used when the interface is closed.
 *
 * This function is registered to linux stop() function. This function is 
 * called whenever ifconfig (in Linux) closes the device (for example "ifconfig eth0 down").
 * This releases all the system resources allocated during open call.
 * system resources int needs 
 * 	- Disable the device interrupts
 * 	- Stop the receiver and get back all the rx descriptors from the DMA
 * 	- Stop the transmitter and get back all the tx descriptors from the DMA 
 * 	- Stop the Linux network queue interface
 *	- Free the irq (ISR registered is removed from the kernel)
 * 	- Release the TX and RX descripor memory
 *	- De-initialize one second timer rgistered for cable plug/unplug tracking
 * @param[in] pointer to net_device structure. 
 * \return Returns 0 on success and error status upon failure.
 * \callgraph
 */

s32 synopGMAC_linux_close(struct net_device *netdev)
{
	
//	s32 status = 0;
//	s32 retval = 0;
//	u32 dma_addr;
	synopGMACPciNetworkAdapter *adapter;
        synopGMACdevice * gmacdev;
	struct pci_dev *pcidev;
	
	TR0("%s\n",__FUNCTION__);
	adapter = (synopGMACPciNetworkAdapter *) netdev->priv;
	if(adapter == NULL){
		TR0("OOPS adapter is null\n");
		return -1;
	}

	gmacdev = (synopGMACdevice *) adapter->synopGMACdev;
	if(gmacdev == NULL){
		TR0("OOPS gmacdev is null\n");
		return -1;
	}

	pcidev = (struct pci_dev *)adapter->synopGMACpcidev;
	if(pcidev == NULL){
		TR("OOPS pcidev is null\n");
		return -1;
	}

	/*Disable all the interrupts*/
	synopGMAC_disable_interrupt_all(gmacdev);
	TR("the synopGMAC interrupt has been disabled\n");

	/*Disable the reception*/	
	synopGMAC_disable_dma_rx(gmacdev);
        synopGMAC_take_desc_ownership_rx(gmacdev);
	TR("the synopGMAC Reception has been disabled\n");

	/*Disable the transmission*/
	synopGMAC_disable_dma_tx(gmacdev);
        synopGMAC_take_desc_ownership_tx(gmacdev);

	TR("the synopGMAC Transmission has been disabled\n");
	netif_stop_queue(netdev);
	/*Now free the irq: This will detach the interrupt handler registered*/
	free_irq(pcidev->irq, netdev);
	TR("the synopGMAC interrupt handler has been removed\n");
	
	/*Free the Rx Descriptor contents*/
	TR("Now calling synopGMAC_giveup_rx_desc_queue \n");
	synopGMAC_giveup_rx_desc_queue(gmacdev, pcidev, RINGMODE);
//	synopGMAC_giveup_rx_desc_queue(gmacdev, pcidev, CHAINMODE);
	TR("Now calling synopGMAC_giveup_tx_desc_queue \n");
	synopGMAC_giveup_tx_desc_queue(gmacdev, pcidev, RINGMODE);
//	synopGMAC_giveup_tx_desc_queue(gmacdev, pcidev, CHAINMODE);
	
	TR("Freeing the cable unplug timer\n");	
	del_timer(&synopGMAC_cable_unplug_timer);

	return -ESYNOPGMACNOERR;

//	TR("%s called \n",__FUNCTION__);
}

/**
 * Function to transmit a given packet on the wire.
 * Whenever Linux Kernel has a packet ready to be transmitted, this function is called.
 * The function prepares a packet and prepares the descriptor and 
 * enables/resumes the transmission.
 * @param[in] pointer to sk_buff structure. 
 * @param[in] pointer to net_device structure.
 * \return Returns 0 on success and Error code on failure. 
 * \note structure sk_buff is used to hold packet in Linux networking stacks.
 */
s32 synopGMAC_linux_xmit_frames(struct sk_buff *skb, struct net_device *netdev)
{
	s32 status = 0;
	s32 counter =0;
	u32 offload_needed = 0;
	u32 dma_addr;
	//u32 flags;
	synopGMACPciNetworkAdapter *adapter;
	synopGMACdevice * gmacdev;
	struct pci_dev * pcidev;
	
	TR("%s called \n",__FUNCTION__);
	if(skb == NULL){
		TR0("skb is NULL What happened to Linux Kernel? \n ");
		return -1;
	}
	
	adapter = (synopGMACPciNetworkAdapter *) netdev->priv;
	if(adapter == NULL)
		return -1;

	gmacdev = (synopGMACdevice *) adapter->synopGMACdev;
	if(gmacdev == NULL)
		return -1;


	//spin_lock_irq(&gmacdev->lock);//add by phosor

 	pcidev  = (struct pci_dev *)adapter->synopGMACpcidev;	
	/*Stop the network queue*/	
	netif_stop_queue(netdev);

	if(skb->ip_summed == CHECKSUM_HW){
		/*	
		   In Linux networking, if kernel indicates skb->ip_summed = CHECKSUM_HW, then only checksum offloading should be performed
		   Make sure that the OS on which this code runs have proper support to enable offloading.
		*/
		offload_needed = 0x00000001;
		
		#if 0
		printk(KERN_CRIT"skb->ip_summed = CHECKSUM_HW\n");
		printk(KERN_CRIT"skb->h.th=%08x skb->h.th->check=%08x\n",(u32)(skb->h.th),(u32)(skb->h.th->check));
		printk(KERN_CRIT"skb->h.uh=%08x skb->h.uh->check=%08x\n",(u32)(skb->h.uh),(u32)(skb->h.uh->check));
		printk(KERN_CRIT"\n skb->len = %d skb->mac_len = %d skb->data = %08x skb->csum = %08x skb->h.raw = %08x\n",skb->len,skb->mac_len,(u32)(skb->data),skb->csum,(u32)(skb->h.raw));
		printk(KERN_CRIT"DST MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+0),*(skb->data+1),*(skb->data+2),*(skb->data+3),*(skb->data+4),*(skb->data+5));
		printk(KERN_CRIT"SRC MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+6),*(skb->data+7),*(skb->data+8),*(skb->data+9),*(skb->data+10),*(skb->data+11));
		printk(KERN_CRIT"Len/type    :%02x %02x\n",*(skb->data+12),*(skb->data+13));
		if(((*(skb->data+14)) & 0xF0) == 0x40){
			printk(KERN_CRIT"IPV4 Header:\n");
			printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+14),*(skb->data+15),*(skb->data+16),*(skb->data+17));
			printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+18),*(skb->data+19),*(skb->data+20),*(skb->data+21));
			printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+22),*(skb->data+23),*(skb->data+24),*(skb->data+25));
			printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+26),*(skb->data+27),*(skb->data+28),*(skb->data+29));
			printk(KERN_CRIT"%02x %02x %02x %02x\n\n",*(skb->data+30),*(skb->data+31),*(skb->data+32),*(skb->data+33));
			for(counter = 34; counter < skb->len; counter++)
				printk("%02X ",*(skb->data + counter));
		}
		else{
			printk(KERN_CRIT"IPV6 FRAME:\n");
			for(counter = 14; counter < skb->len; counter++)
				printk("%02X ",*(skb->data + counter));
		}
		#endif
	}
	else{//-->>this
		#ifdef	IPC_OFFLOAD
		offload_needed = 0x00000001;
		#endif			

		#if 0
		{
			printk("\n ==xmit skb->len==%d======", skb->len);
			printk("skb->ip_summed = CHECKSUM_SW\n");
			printk("skb->h.uh=%08x skb->h.uh->check=%08x\n",(u32)(skb->h.uh),(u32)(skb->h.uh->check));
			printk("\n skb->len = %d skb->data_len = %d skb->data = %08x skb->csum = %08x skb->h.raw = %08x\n",skb->len,skb->data_len,(u32)(skb->data),skb->csum,(u32)(skb->h.raw));
			printk("DST MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+0),*(skb->data+1),*(skb->data+2),*(skb->data+3),*(skb->data+4),*(skb->data+5));
			printk("SRC MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+6),*(skb->data+7),*(skb->data+8),*(skb->data+9),*(skb->data+10),*(skb->data+11));
			printk("Len/type    :%02x %02x\n",*(skb->data+12),*(skb->data+13));
			if (*(skb->data+12) == 0x08 && *(skb->data+13) == 0x06){

					printk("h %02x %02x p %02x %02x\n",*(skb->data+14),*(skb->data+15),*(skb->data+16),*(skb->data+17));
					printk("hL %02x PL %02x O:%02x %02x\n",*(skb->data+18),*(skb->data+19),*(skb->data+20),*(skb->data+21));
					printk("SHA %02x %02x %02x %02x %02x %02x\n",*(skb->data+22),*(skb->data+23),*(skb->data+24),*(skb->data+25), *(skb->data+26),*(skb->data+27));
					printk("SIA %02x %02x %02x %02x\n",*(skb->data+28),*(skb->data+29),*(skb->data+30),*(skb->data+31));
					printk("THA:%02x %02x %02x %02x %02x %02x\n",*(skb->data+32),*(skb->data+33),*(skb->data+34),*(skb->data+35),*(skb->data+36),*(skb->data+37));
					printk("TIA %02x %02x %02x %02x\n",*(skb->data+38),*(skb->data+39),*(skb->data+40),*(skb->data+41));
			}
			else
			{
				printk(KERN_CRIT"IPV4 Header:\n");
				printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+14),*(skb->data+15),*(skb->data+16),*(skb->data+17));
				printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+18),*(skb->data+19),*(skb->data+20),*(skb->data+21));
				printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+22),*(skb->data+23),*(skb->data+24),*(skb->data+25));
				printk(KERN_CRIT"%02x %02x %02x %02x\n",*(skb->data+26),*(skb->data+27),*(skb->data+28),*(skb->data+29));
				printk(KERN_CRIT"%02x %02x %02x %02x\n\n",*(skb->data+30),*(skb->data+31),*(skb->data+32),*(skb->data+33));
				for(counter = 34; counter < skb->len; counter++)
					printk("%02X ",*(skb->data + counter));
			}
		}
		#endif
	}


	#ifdef	IPC_OFFLOAD
			//ֹCHECKSUM0ΪҪCOEĻ뱣֤CHECKSUMΪ0
	if (*(skb->data+12) == 0x08 && *(skb->data+13) == 0x06){
		}
	else{
		//ֹCHECKSUM0
		//IP
		if (skb->nh.iph->check) {
			//printk("\n==ip check sum ==%x", skb->nh.iph->check);
			*(skb->data+24) = *(skb->data+25) = 0;
		}
			
		//icmp
		if (skb->nh.iph->protocol == 1) {
			if (*(skb->data+36) != 0 ||  *(skb->data+37) != 0) {
				//printk("\==icmp checksum %x %x==", *(skb->data+36), *(skb->data+37));
				*(skb->data+36) = *(skb->data+37) = 0;
			}
		}
		//udp
		else if(skb->nh.iph->protocol == 17) {
			if (*(skb->data+40) != 0 ||  *(skb->data+41) != 0) {
				//printk("\==udp checksum %x %x==", *(skb->data+40), *(skb->data+41));
				*(skb->data+40) = *(skb->data+41) = 0;
			}
		}
		//tcp
		else if(skb->nh.iph->protocol == 6) {
			if (*(skb->data+50) != 0 ||  *(skb->data+51) != 0) {
				//printk("\===tcp checksum %x %x==", *(skb->data+50), *(skb->data+51));
				*(skb->data+50) = *(skb->data+51) = 0;
			}
		}
	}
	#endif	


	if (*(skb->data+23) == 1)
    	{
		*(pDbgMem+1) = jiffies;
		//printk("\n=drv= tx jiffies %d=%d=", *(pDbgMem+1), skb->nh.iph->protocol);
	}


	//--/*Now we have skb ready and OS invoked this function. Lets make our DMA know about this*/
	//--dma_addr = pci_map_single(pcidev,skb->data,skb->len,PCI_DMA_TODEVICE);
	dma_addr = skb->data;

	//offload_needed˷ʱҪҪCOECHECKSUM
	status = synopGMAC_set_tx_qptr(gmacdev, dma_addr, skb->len, (u32)skb,0,0,0,offload_needed);
	if(status < 0){
		TR0("%s No More Free Tx Descriptors\n",__FUNCTION__);
		return -EBUSY;
	}
	
	/*Now force the DMA to start transmission*/	
	synopGMAC_resume_dma_tx(gmacdev);
	netdev->trans_start = jiffies;

	//spin_unlock_irq(&gmacdev->lock);//add by phosor

	
	/*Now start the netdev queue*/
	netif_wake_queue(netdev);//ͻǷ

	return -ESYNOPGMACNOERR;
}

/**
 * Function provides the network interface statistics.
 * Function is registered to linux get_stats() function. This function is 
 * called whenever ifconfig (in Linux) asks for networkig statistics
 * (for example "ifconfig eth0").
 * @param[in] pointer to net_device structure. 
 * \return Returns pointer to net_device_stats structure.
 * \callgraph
 */
struct net_device_stats *  synopGMAC_linux_get_stats(struct net_device *netdev)
{
	TR("%s called \n",__FUNCTION__);
	return( &(((synopGMACPciNetworkAdapter *)(netdev->priv))->synopGMACNetStats) );
}

/**
 * Function to set multicast and promiscous mode.
 * @param[in] pointer to net_device structure. 
 * \return returns void.
 * \todo Function not yet implemented.
 */
void synopGMAC_linux_set_multicast_list(struct net_device *netdev)
{
	TR("%s called \n",__FUNCTION__);
	return;
}

/**
 * Function to set ethernet address of the NIC.
 * @param[in] pointer to net_device structure. 
 * @param[in] pointer to an address structure. 
 * \return Returns 0 on success Errorcode on failure.
 */
s32 synopGMAC_linux_set_mac_address(struct net_device *netdev, void * macaddr)
{

	synopGMACPciNetworkAdapter *adapter = NULL;
	synopGMACdevice * gmacdev = NULL;
	struct sockaddr *addr = macaddr;

	adapter = (synopGMACPciNetworkAdapter *) netdev->priv;
	if(adapter == NULL)
		return -1;

	gmacdev = adapter->synopGMACdev;
	if(gmacdev == NULL)
		return -1;

	if(!is_valid_ether_addr(addr->sa_data))
		return -EADDRNOTAVAIL;

	synopGMAC_set_mac_addr(gmacdev,GmacAddr0High,GmacAddr0Low, addr->sa_data); 
	synopGMAC_get_mac_addr(synopGMACadapter->synopGMACdev,GmacAddr0High,GmacAddr0Low, netdev->dev_addr); 

	TR("%s called \n",__FUNCTION__);
	return 0;
}

/**
 * Function to change the Maximum Transfer Unit.
 * @param[in] pointer to net_device structure. 
 * @param[in] New value for maximum frame size.
 * \return Returns 0 on success Errorcode on failure.
 * \todo Function not yet implemented.
 */
s32 synopGMAC_linux_change_mtu(struct net_device *netdev, s32 newmtu)
{
	TR("%s called \n",__FUNCTION__);
	return 0;

}

/**
 * IOCTL interface.
 * This function is mainly for debugging purpose.
 * This provides hooks for Register read write, Retrieve descriptor status
 * and Retreiving Device structure information.
 * @param[in] pointer to net_device structure. 
 * @param[in] pointer to ifreq structure.
 * @param[in] ioctl command. 
 * \return Returns 0 on success Error code on failure.
 */
s32 synopGMAC_linux_do_ioctl(struct net_device *netdev, struct ifreq *ifr, s32 cmd)
{
	s32 retval = 0;
	u16 temp_data = 0;
	synopGMACPciNetworkAdapter *adapter = NULL;
	synopGMACdevice * gmacdev = NULL;
	struct ifr_data_struct
	{
		u32 unit;
		u32 addr;
		u32 data;
	} *req;


	if(netdev == NULL)
		return -1;
	if(ifr == NULL)
		return -1;

	req = (struct ifr_data_struct *)ifr->ifr_data;

	adapter = (synopGMACPciNetworkAdapter *) netdev->priv;
	if(adapter == NULL)
		return -1;

	gmacdev = adapter->synopGMACdev;
	if(gmacdev == NULL)
		return -1;
	//TR("%s :: on device %s req->unit = %08x req->addr = %08x req->data = %08x cmd = %08x \n",__FUNCTION__,netdev->name,req->unit,req->addr,req->data,cmd);

	switch(cmd)
	{
		case IOCTL_READ_REGISTER:		//IOCTL for reading IP registers : Read Registers
			if      (req->unit == 0)	// Read Mac Register
				req->data = synopGMACReadReg((u32 *)gmacdev->MacBase,req->addr);
			else if (req->unit == 1)	// Read DMA Register
				req->data = synopGMACReadReg((u32 *)gmacdev->DmaBase,req->addr);
			else if (req->unit == 2){	// Read Phy Register
				retval = synopGMAC_read_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,req->addr,&temp_data);
				req->data = (u32)temp_data;
				if(retval != -ESYNOPGMACNOERR)
					TR("ERROR in Phy read\n");	
			}
			break;

		case IOCTL_WRITE_REGISTER:		//IOCTL for reading IP registers : Read Registers
			if      (req->unit == 0)	// Write Mac Register
				synopGMACWriteReg((u32 *)gmacdev->MacBase,req->addr,req->data);
			else if (req->unit == 1)	// Write DMA Register
				synopGMACWriteReg((u32 *)gmacdev->DmaBase,req->addr,req->data);
			else if (req->unit == 2){	// Write Phy Register
				retval = synopGMAC_write_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,req->addr,req->data);
				if(retval != -ESYNOPGMACNOERR)
					TR("ERROR in Phy read\n");	
			}
			break;

		case IOCTL_READ_IPSTRUCT:		//IOCTL for reading GMAC DEVICE IP private structure
		        memcpy(ifr->ifr_data, gmacdev, sizeof(synopGMACdevice));
			break;

		case IOCTL_READ_RXDESC:			//IOCTL for Reading Rx DMA DESCRIPTOR
			memcpy(ifr->ifr_data, gmacdev->RxDesc + ((DmaDesc *) (ifr->ifr_data))->data1, sizeof(DmaDesc) );
			break;

		case IOCTL_READ_TXDESC:			//IOCTL for Reading Tx DMA DESCRIPTOR
			memcpy(ifr->ifr_data, gmacdev->TxDesc + ((DmaDesc *) (ifr->ifr_data))->data1, sizeof(DmaDesc) );
			break;
		case IOCTL_POWER_DOWN:
			if	(req->unit == 1){	//power down the mac
				TR("============I will Power down the MAC now =============\n");
				// If it is already in power down don't power down again
				retval = 0;
				if(((synopGMACReadReg((u32 *)gmacdev->MacBase,GmacPmtCtrlStatus)) & GmacPmtPowerDown) != GmacPmtPowerDown){
				synopGMAC_linux_powerdown_mac(gmacdev);			
				retval = 0;
				}
			}
			if	(req->unit == 2){	//Disable the power down  and wake up the Mac locally
				TR("============I will Power up the MAC now =============\n");
				//If already powered down then only try to wake up
				retval = -1;
				if(((synopGMACReadReg((u32 *)gmacdev->MacBase,GmacPmtCtrlStatus)) & GmacPmtPowerDown) == GmacPmtPowerDown){
				synopGMAC_power_down_disable(gmacdev);
				synopGMAC_linux_powerup_mac(gmacdev);
				retval = 0;
				}
			}
			break;
		default:
			retval = -1;

	}


	return retval;
}

/**
 * Function to handle a Tx Hang.
 * This is a software hook (Linux) to handle transmitter hang if any.
 * We get transmitter hang in the device interrupt status, and is handled
 * in ISR. This function is here as a place holder.
 * @param[in] pointer to net_device structure 
 * \return void.
 * \todo Function not yet implemented
 */
void synopGMAC_linux_tx_timeout(struct net_device *netdev)
{
	s32 status = 0;
	u32 dma_addr;
	struct sk_buff *skb;
        synopGMACPciNetworkAdapter *adapter;
        synopGMACdevice * gmacdev;
	struct pci_dev *pcidev;

	
	TR0("%s called \n",__FUNCTION__);
	adapter = (synopGMACPciNetworkAdapter *) netdev->priv;
	gmacdev = (synopGMACdevice *)adapter->synopGMACdev;
 	pcidev  = (struct pci_dev *)adapter->synopGMACpcidev;	
	
	
	synopGMAC_giveup_tx_desc_queue(gmacdev,pcidev, RINGMODE);
	synopGMAC_setup_tx_desc_queue(gmacdev,pcidev,TRANSMIT_DESC_SIZE, RINGMODE);
	synopGMAC_init_tx_desc_base(gmacdev);	//Program the transmit descriptor base address in to DmaTxBase addr

	synopGMAC_giveup_rx_desc_queue(gmacdev,pcidev, RINGMODE);
	synopGMAC_setup_rx_desc_queue(gmacdev,pcidev,RECEIVE_DESC_SIZE, RINGMODE);
	synopGMAC_init_rx_desc_base(gmacdev);	//Program the transmit descriptor base address in to DmaTxBase addr

	synopGMAC_dma_bus_mode_init(gmacdev,DmaFixedBurstEnable| DmaBurstLength8 | DmaDescriptorSkip2);//ʹRINGģʽѡDmaDescriptorSkip0 
	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward);	
	
	synopGMAC_mac_init(gmacdev);

     do{
		skb = dev_alloc_skb(PKT_BUF_SIZE);
		if(skb == NULL){
			TR0("ERROR in skb buffer allocation\n");
			return -ESYNOPGMACNOMEM;
		}
		skb->dev = netdev;
		skb_reserve(skb,ETHERNET_EXTRA);

		//--dma_addr = pci_map_single(pcidev,skb->data,skb_tailroom(skb),PCI_DMA_FROMDEVICE);
		dma_addr = skb->data;
		status = synopGMAC_set_rx_qptr(gmacdev,dma_addr, skb_tailroom(skb), (u32)skb,0,0,0);
		if(status < 0)
			dev_kfree_skb(skb);
			
	}while(status >= 0);
	
	synopGMAC_clear_interrupt(gmacdev);
	synopGMAC_enable_interrupt(gmacdev,DmaIntEnable);
	synopGMAC_enable_dma_rx(gmacdev);
	synopGMAC_enable_dma_tx(gmacdev);
	
    netif_wake_queue (netdev);

    return;
}

/**
 * Function to initialize the Linux network interface.
 * 
 * Linux dependent Network interface is setup here. This provides 
 * an example to handle the network dependent functionality.
 *
 * \return Returns 0 on success and Error code on failure.
 */
s32 __init synopGMAC_init_network_interface(void)
{
	s32 err;
	struct net_device *netdev;
	
	unsigned int *p;
	
	printk("MAX_ORDER = %#x\n", MAX_ORDER );
	
	p = kmalloc( 32 * 1024 * 1024, GFP_DMA );
	
	if ( NULL == p )
    {
        printk("ffffff\n");
    }
    else
    {
        printk("p = %p\n", p);
    }
    
    kfree( p );
    
    p = kmalloc( 1 * 1024 * 1024, GFP_KERNEL );
	
	if ( NULL == p )
    {
        printk("ffffff\n");
    }
    else
    {
        printk("p = %p\n", p);
    }

    kfree( p );

	TR("Now Going to Call register_netdev to register the network interface for GMAC core\n");
	/*
	Lets allocate and set up an ethernet device, it takes the sizeof the private structure. This is mandatory as a 32 byte 
	allignment is required for the private data structure.
	*/
	//netdev = init_etherdev(NULL, sizeof(synopGMACPciNetworkAdapter));
	netdev = alloc_etherdev(sizeof(synopGMACPciNetworkAdapter));
	if(!netdev){
	err = -ESYNOPGMACNOMEM;
	goto err_alloc_etherdev;
	}	
	
	synopGMACadapter = netdev->priv;//synopGMACPciNetworkAdapter˽к豸ṹ
	synopGMACadapter->synopGMACnetdev = netdev;


	synopGMACadapter->synopGMACpcidev = NULL;
	synopGMACadapter->synopGMACpcidev =  (struct pci_dev *)plat_alloc_memory(sizeof(struct pci_dev));
	//ûóʼΪGMACûйPCIsynopGMACadapter->synopGMACpcidevҲЧ
	if(!synopGMACadapter->synopGMACpcidev){
		TR0("Error in Memory Allocataion \n");
	}
	synopGMACadapter->synopGMACpcidev->irq = 40;

	
	/*Allocate Memory for the the GMACip structure*/
	synopGMACadapter->synopGMACdev    = NULL;
	synopGMACadapter->synopGMACdev = (synopGMACdevice *) plat_alloc_memory(sizeof (synopGMACdevice));
	if(!synopGMACadapter->synopGMACdev){
		TR0("Error in Memory Allocataion \n");
	}

	/*Attach the device to MAC struct This will configure all the required base addresses
	  such as Mac base, configuration base, phy base address(out of 32 possible phys )*/
	synopGMAC_attach(synopGMACadapter->synopGMACdev,(u32) synopGMACMappedAddr + MACBASE,(u32) synopGMACMappedAddr + DMABASE, DEFAULT_PHY_BASE);
	synopGMAC_reset(synopGMACadapter->synopGMACdev);

	if(synop_pci_using_dac){
		TR("netdev->features = %08x\n",netdev->features);
		TR("synop_pci_using dac is %08x\n",synop_pci_using_dac);
		netdev->features |= NETIF_F_HIGHDMA;
		TR("netdev->features = %08x\n",netdev->features);
	}

	netdev->open = &synopGMAC_linux_open; 
	netdev->stop = &synopGMAC_linux_close;
	netdev->hard_start_xmit = &synopGMAC_linux_xmit_frames;
	netdev->get_stats = &synopGMAC_linux_get_stats;
	netdev->set_multicast_list = &synopGMAC_linux_set_multicast_list;
	netdev -> set_mac_address = &synopGMAC_linux_set_mac_address;
	netdev -> change_mtu = &synopGMAC_linux_change_mtu;
	netdev -> do_ioctl = &synopGMAC_linux_do_ioctl;
	netdev -> tx_timeout = &synopGMAC_linux_tx_timeout; 
	netdev->watchdog_timeo = 5 * HZ;

	/*Now start the network interface*/
	TR("Now Registering the netdevice\n");

	//ether_setup(netdev);
	if((err = register_netdev(netdev)) != 0) {
		TR0("Error in Registering netdevice\n");
		return err;
	}

	
 	return 0;
err_alloc_etherdev:
	TR0("Problem in alloc_etherdev()..Take Necessary action\n");
	return err;
}


/**
 * Function to initialize the Linux network interface.
 * Linux dependent Network interface is setup here. This provides 
 * an example to handle the network dependent functionality.
 * \return Returns 0 on success and Error code on failure.
 */
void __exit synopGMAC_exit_network_interface(void)
{
	TR0("Now Calling network_unregister\n");
	unregister_netdev(synopGMACadapter->synopGMACnetdev);	
}

