/*
 * linux/drivers/video/csky/ck1000fb.c -- driver for CK1000 framebuffer device
 *
 * by Li Xin
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/fb.h>
#include <linux/init.h>

#include <video/fbcon.h> 
#include <video/fbcon-mfb.h>
#include <video/fbcon-cfb2.h>
#include <video/fbcon-cfb4.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>
#include <video/fbcon-cfb32.h>

#include "ck1000fb.h"

#define FBCON_HAS_CFB16

CK_INSTANTIATE_REGISTERS(static,aCKRegs);

    /*
     *  This is just simple sample code.
     *
     *  No warranty that it actually compiles.
     *  Even less warranty that it actually works :-)
     */


struct ck1000fb_info {
    /*
     *  Choose _one_ of the two alternatives:
     *
     *    1. Use the generic frame buffer operations (fbgen_*).
     */
    struct fb_info_gen gen;
    unsigned char *VmemAddr; 
    volatile unsigned char *RegAddr; 
#ifdef FBCON_HAS_CFB16
    u16 cfb16[16];
#endif

    /*
     *    2. Provide your own frame buffer operations.
     */
//    struct fb_info info;

    /* Here starts the frame buffer device dependent part */
    /* You can use this to store e.g. the board number if you support */
    /* multiple boards */
};


struct ck1000fb_par {
    /*
     *  The hardware specific data in this structure uniquely defines a video
     *  mode.
     *
     *  If your hardware supports only one video mode, you can leave it empty.
     */
};


    /*
     *  If your driver supports multiple boards, you should make these arrays,
     *  or allocate them dynamically (using kmalloc()).
     */

static struct ck1000fb_info fb_info;
static struct ck1000fb_par current_par;
static int current_par_valid = 0;
static struct display disp;
static char __initdata default_fontname[40] = { 0 }; 
static void ck1000fb_handle_irq(int irq, void *dev_id, struct pt_regs *regs);
static struct fb_var_screeninfo default_var;

//static int currcon = 0;
//static int inverse = 0;

int __init ck1000fb_init(char*);
int __init ck1000fb_setup(char *options, int *ints);

static struct fb_ops ck1000fb_ops = {
	owner:		THIS_MODULE,
//	fb_open:	ck1000fb_open,    /* only if you need it to do something */
//	fb_release:	ck1000fb_release, /* only if you need it to do something */
	fb_get_fix:	fbgen_get_fix,
	fb_get_var:	fbgen_get_var,
	fb_set_var:	fbgen_set_var,
	fb_get_cmap:	fbgen_get_cmap,
	fb_set_cmap:	fbgen_set_cmap,
	fb_pan_display:	fbgen_pan_display,
//	fb_ioctl:	ck1000fb_ioctl,   /* optional */
};


/* ------------------- chipset specific functions -------------------------- */


static void ck1000_detect(void)
{
    /*
     *  This function should detect the current video mode settings and store
     *  it as the default video mode
     */

//    struct ck1000fb_par par;

    /* ... */
//    ck1000_get_par(&par);
//    ck1000_encode_var(&default_var, &par);
}

static int ck1000_encode_fix(struct fb_fix_screeninfo *fix, struct ck1000fb_par *par,
			  const struct fb_info *info)
{
    /*
     *  This function should fill in the 'fix' structure based on the values
     *  in the `par' structure.
     */
    memset(fix, 0, sizeof(struct fb_fix_screeninfo));
    strcpy(fix->id, fb_info.gen.info.modename);

    fix->type = FB_TYPE_PACKED_PIXELS;  /* see FB_TYPE_* in fb.h    */       
    fix->type_aux = 0;                  /* Interleave for interleaved Planes */

    /* set fix->line_length to the length of the line in bytes */
    fix->line_length = CK_DISPLAY_SCANLINE_BYTES;

#ifdef FBCON_HAS_CFB16
    fix->visual = FB_VISUAL_TRUECOLOR;
#endif

    /* Start and length of Memory Mapped I/O */ 
    fix->mmio_start = CK_PHYSICAL_REG_ADDR;     
    fix->mmio_len   = CK_PHYSICAL_REG_SIZE;    

    /* Start and length of frame buffer mem */ 
    fix->smem_start = CK_PHYSICAL_VMEM_ADDR;    
    fix->smem_len   = CK_PHYSICAL_VMEM_SIZE;   
    
    fix->xpanstep = fix->ypanstep = fix->ywrapstep = 0;
    fix->accel = FB_ACCEL_NONE;

    /* ... */
    return 0;
}

static int ck1000_decode_var(struct fb_var_screeninfo *var, struct ck1000fb_par *par,
			  const struct fb_info *info)
{
    /*
     *  Get the video params out of 'var'. If a value doesn't fit, round it up,
     *  if it's too big, return -EINVAL.
     *
     *  Suggestion: Round up in the following order: bits_per_pixel, xres,
     *  yres, xres_virtual, yres_virtual, xoffset, yoffset, grayscale,
     *  bitfields, horizontal timing, vertical timing.
     */

    /* ... */

    /* pixclock in picos, htotal in pixels, vtotal in scanlines */
//    if (!fbmon_valid_timings(pixclock, htotal, vtotal, info))
//	    return -EINVAL;

    return 0;
}

static int ck1000_encode_var(struct fb_var_screeninfo *var, struct ck1000fb_par *par,
			  const struct fb_info *info)
{
    /*
     *  Fill the 'var' structure based on the values in 'par' and maybe other
     *  values read out of the hardware.
     */
    memset(var, 0, sizeof(struct fb_var_screeninfo));

    var->xres =  CK_DISPLAY_WIDTH;     //visible resolution
    var->yres =  CK_DISPLAY_HEIGHT;

    var->xres_virtual = var->xres;
    var->yres_virtual = var->yres;
    var->xoffset = var->yoffset = 0;
    var->bits_per_pixel = CK_DISPLAY_BPP;
    var->grayscale = 0;
    var->nonstd = 0;                 /* != 0 Non standard pixel format */
    var->activate = FB_ACTIVATE_NOW; /* see FB_ACTIVATE_*              */
    var->height = -1;                /* height of picture in mm        */
    var->width  = -1;                /* width of picture in mm         */
    var->accel_flags = 0;            /* acceleration flags (hints      */
    var->pixclock = CK_DISPLAY_PCLK;
    var->right_margin = 0;
    var->lower_margin = 0;
    var->hsync_len = 0;
    var->vsync_len = 0;
    var->left_margin = 0;
    var->upper_margin = 0;
    var->sync = 0;
    var->vmode = FB_VMODE_NONINTERLACED;

#ifdef FBCON_HAS_CFB16  // RGB 555
    var->red.offset = 10;
    var->red.length = 5;
    var->green.offset = 5;
    var->green.length = 5;
    var->blue.offset = 0;
    var->blue.length = 5;
#endif

    var->red.msb_right = var->green.msb_right = var->blue.msb_right = 0;
    var->transp.offset = var->transp.length = var->transp.msb_right = 0;
    /* ... */
    return 0;
}

static void ck1000_get_par(struct ck1000fb_par *par, const struct fb_info *info)
{
    /*
     *  Fill the hardware's 'par' structure.
     */

//    if (current_par_valid)
//	*par = current_par;
//    else {
	/* ... */
//    }
}

static void ck1000_set_par(struct ck1000fb_par *par, const struct fb_info *info)
{
    /*
     *  Set the hardware according to 'par'.
     */

//    current_par = *par;
//    current_par_valid = 1;
    /* ... */
}

static int ck1000_getcolreg(unsigned regno, unsigned *red, unsigned *green,
			 unsigned *blue, unsigned *transp,
			 const struct fb_info *info)
{
    /*
     *  Read a single color register and split it into colors/transparent.
     *  The return values must have a 16 bit magnitude.
     *  Return != 0 for invalid regno.
     */
    u32 r,g,b;

    if (regno >= CK_PALETTE_SIZE) 
        return 1;

    CK_READ_PALETTE(fb_info.RegAddr,regno,r,g,b);  

    *red    = (r << 8) | r; 
    *green  = (g << 8) | g; 
    *blue   = (b << 8) | b; 
    *transp = 0; 

    /* ... */
    return 0;
}

static int ck1000_setcolreg(unsigned regno, unsigned red, unsigned green,
			 unsigned blue, unsigned transp,
			 const struct fb_info *info)
{
    /*
     *  Set a single color register. The values supplied have a 16 bit
     *  magnitude.
     *  Return != 0 for invalid regno.
     */
    if (regno >= CK_PALETTE_SIZE) 
        return 1;

    CK_WRITE_PALETTE(fb_info.RegAddr,regno,red>>8,green>>8,blue>>8); 

#if defined(FBCON_HAS_CFB16) 
    // Make the first 16 colors of the palette available to fbcon 

    if (regno < 16) 
        {
        fb_info.cfb16[regno] = 
            ((red & 0xf800) >> 10)|((green & 0xf800) >> 5)|(blue & 0xf800); 
        }
#endif
    
//    if (regno < 16) {
	/*
	 *  Make the first 16 colors of the palette available to fbcon
	 */
//	if (is_cfb16)		/* RGB 555 */
//	    info->fbcon_cmap.cfb16[regno] = ((red & 0xf800) >> 1) |
//					 ((green & 0xf800) >> 6) |
//					 ((blue & 0xf800) >> 11);
//	if (is_cfb16)		/* RGB 565 */
//	    ...fbcon_cmap.cfb16[regno] = (red & 0xf800) |
//					 ((green & 0xfc00) >> 5) |
//					 ((blue & 0xf800) >> 11);					 
//	if (is_cfb24)		/* RGB 888 */
//	    fbcon_cmap.cfb24[regno] = ((red & 0xff00) << 8) |
//					 (green & 0xff00) |
//					 ((blue & 0xff00) >> 8);
//	if (is_cfb32)		/* RGBA 8888 */
//	    fbcon_cmap.cfb32[regno] = ((red & 0xff00) << 16) |
//					 ((green & 0xff00) << 8) |
//					 (blue & 0xff00) |
//					 ((transp & 0xff00) >> 8);
//    }
    /* ... */
    return 0;
}

static int ck1000_pan_display(struct fb_var_screeninfo *var,
			   struct ck1000fb_par *par, const struct fb_info *info)
{
    /*
     *  Pan (or wrap, depending on the `vmode' field) the display using the
     *  `xoffset' and `yoffset' fields of the `var' structure.
     *  If the values don't fit, return -EINVAL.
     */

    /* ... */
    return 0;
}

static int ck1000_blank(int blank_mode, const struct fb_info *info)
{
    /*
     *  Blank the screen if blank_mode != 0, else unblank. If blank == NULL
     *  then the caller blanks by setting the CLUT (Color Look Up Table) to all
     *  black. Return 0 if blanking succeeded, != 0 if un-/blanking failed due
     *  to e.g. a video mode which doesn't support it. Implements VESA suspend
     *  and powerdown modes on hardware that supports disabling hsync/vsync:
     *    blank_mode == 2: suspend vsync
     *    blank_mode == 3: suspend hsync
     *    blank_mode == 4: powerdown
     */

    /* ... */
    return 0;
}

static void ck1000_set_disp(const void *par, struct display *disp,
			 struct fb_info_gen *info)
{
    /*
     *  Fill in a pointer with the virtual address of the mapped frame buffer.
     *  Fill in a pointer to appropriate low level text console operations (and
     *  optionally a pointer to help data) for the video mode `par' of your
     *  video hardware. These can be generic software routines, or hardware
     *  accelerated routines specifically tailored for your hardware.
     *  If you don't have any appropriate operations, you must fill in a
     *  pointer to dummy operations, and there will be no text output.
     */
    disp->screen_base = fb_info.VmemAddr;

#ifdef FBCON_HAS_CFB16
    disp->dispsw = &fbcon_cfb16;
    disp->dispsw_data = fb_info.cfb16;  /* console palette */
#endif

    disp->scrollmode = SCROLL_YREDRAW;
}


/* ------------ Interfaces to hardware functions ------------ */

struct fbgen_hwswitch ck1000_switch = {
    ck1000_detect, 
    ck1000_encode_fix, 
    ck1000_decode_var, 
    ck1000_encode_var, 
    ck1000_get_par,
    ck1000_set_par, 
    ck1000_getcolreg, 
    ck1000_setcolreg, 
    ck1000_pan_display, 
    ck1000_blank,
    ck1000_set_disp
};


/* ------------ Hardware Independent Functions ------------ */


    /*
     *  Initialization
     */

int __init ck1000fb_init(char *dummy)
{
    int i;
    int * add;
    int *p; 
    int ret;
    
    CK_INDEX CKReg;
    CK_VALUE CKValue;
    
	fb_info.VmemAddr = (unsigned char*) ioremap_nocache(CK_PHYSICAL_VMEM_ADDR,CK_PHYSICAL_VMEM_SIZE); 
//	        printk("fb_info.VmemAddr:0x%x\n",fb_info.VmemAddr); 

    fb_info.RegAddr  = (unsigned char*) ioremap(CK_PHYSICAL_REG_ADDR,CK_PHYSICAL_REG_SIZE); 

    if (!fb_info.VmemAddr || !fb_info.RegAddr)  
        { 
        printk("CK1000_lcd_controller_init: ioremap() returned NULL\n"); 
        return -EINVAL;
        }

    for (i = 0; i < sizeof(aCKRegs)/sizeof(aCKRegs[0]); i++)
        {
        CKReg   = aCKRegs[i].Index;
        CKValue = aCKRegs[i].Value;
 
//        if (CKReg == CK_REGDELAYOFF || CKReg == CK_REGDELAYON)
//            mdelay((int)CKValue);
//        else
            {
            ((CK_VALUE*)fb_info.RegAddr)[CKReg/sizeof(CK_VALUE)] = CKValue; 
//            printk("CK1000: reg:%X val=%X\n",CKReg,CKValue);
            }
        }
        
    strcpy(fb_info.gen.info.modename, "CK1000"); 
    fb_info.gen.info.changevar = NULL; 
    fb_info.gen.info.node = -1; 
    fb_info.gen.info.fbops = &ck1000fb_ops; 
    fb_info.gen.info.disp = &disp; 

    fb_info.gen.info.switch_con = &fbgen_switch;
    fb_info.gen.info.updatevar = &fbgen_update_var;
    fb_info.gen.info.blank = &fbgen_blank;

    strcpy(fb_info.gen.info.fontname, default_fontname);
    fb_info.gen.parsize = 0;
    fb_info.gen.info.flags = FBINFO_FLAG_DEFAULT; 
    fb_info.gen.fbhw = &ck1000_switch; 
    fb_info.gen.fbhw->detect(); 

    fbgen_get_var(&disp.var, -1, &fb_info.gen.info); 
    disp.var.activate = FB_ACTIVATE_NOW;
    fbgen_do_set_var(&disp.var, 1, &fb_info.gen); 
    fbgen_set_disp(-1, &fb_info.gen); 
    fbgen_install_cmap(0, &fb_info.gen); 
 
    if (register_framebuffer(&fb_info.gen.info) < 0) 
        return -EINVAL;

    printk("ck1000: CK1000 frame buffer device\n"); 

    printk("ck1000: Display %d x %d %dBpp\n", 
        CK_DISPLAY_WIDTH,CK_DISPLAY_HEIGHT,CK_DISPLAY_BPP);
        

    
    /* uncomment this if your driver cannot be unloaded */
    /* MOD_INC_USE_COUNT; */
    
/*//	p = 0x10000008;
//	*p = 4;
//	printk("0x%d\n",*p);
	p = 0x10000000;
	*p = 1;
//	printk("0x%d\n",*p);
	p = 0x10000004;
	*p = 1;
//	printk("0x%d\n",*p);
	p = 0x1000000c;
	*p = 3;
//	printk("0x%d\n",*p);

//	p = 0x10000048;
//	*p = 3;
//	printk("0x%d\n",*p);*/
    add = CK_PHYSICAL_VMEM_ADDR;
    for(i=0;i<38400;i++)
    {
        (*add) = 0x0;
        add++;
    }

    add = CK_PHYSICAL_VMEM_ADDR;
    for (i=0;i<160;i++){
    (*add) = 0x7c007c00;add++;}
    for(i=0;i<38080;i++){
        (*add) = 0x1f001f;add++;}
    for (i=0;i<160;i++){
        (*add) = 0x3e003e0;add++;}
        
    add = CK_PHYSICAL_VMEM_ADDR+0x25800;
    for (i=0;i<160;i++){
    (*add) = 0x7c007c00;add++;}
    for(i=0;i<38080;i++){
        (*add) = 0x1f001f;add++;}
    for (i=0;i<160;i++){
        (*add) = 0x3e003e0;add++;}    
    
/*    
    for (i=0;i<80;i++)
    {
    for (j=0;j<320;j++)
    {
        (*add) = 0x3e003e0;
        *(add+160) = 0x1f001f;
        *(add+320) = 0x7c007c00;
        add++;
    }
    add+=320;
    }

    for(i=0;i<12800;i++)        //green
    {
        (*add) = 0x3e003e0;
        add = add + 1;
    }

    for(;i<25600;i++)           //blue
    {
        (*add) = 0x1f001f;
        add = add + 1;
    }

    for(;i<37760;i++)           //red
    {
        (*add) = 0x7c007c00;
        add = add + 1;
    }
*/
    ret = request_irq(CK_LCD_IRQ, ck1000fb_handle_irq, SA_INTERRUPT,
			  "ck1000fb", NULL);
	if (ret) {
		printk(KERN_ERR "ck1000fb: request_irq failed: %d\n", ret);
	}


    return 0;
}


    /*
     *  ck1000fb_handle_irq: Handle interrupts.
     */

static void ck1000fb_handle_irq(int irq, void *dev_id, struct pt_regs *regs)
{
    int * int_reg = 0x10003020;
//    int * ctl_reg = 0x10003000;
//    int * bas_reg = 0x10003010;
//    int i =10;
    
//    printk("int = %x\n",*int_reg); 
    
    if (((*int_reg)&2)==2)
    {
        
//        *(ctl_reg)=0xc48;
//        if(*(bas_reg)==0x28fb0000)
//        *(bas_reg)=0x28fd5800;
//        else
//        *(bas_reg)=0x28fb0000;
        *(int_reg)=0xf;
//        while(i)
//        i--;
//        *(ctl_reg)=0xc49;
//        printk("base = %x\n",*bas_reg); 

    }

    return;
}

    /*
     *  Cleanup
     */

void ck1000fb_cleanup(struct fb_info *info)
{
    /*
     *  If your driver supports multiple boards, you should unregister and
     *  clean up all instances.
     */

    unregister_framebuffer(info);
    /* ... */
}


    /*
     *  Setup
     */

int __init ck1000fb_setup(char *options, int *ints)
{
    /* Parse user speficied options (`video=ck1000fb:') */
}


/* ------------------------------------------------------------------------- */


    /*
     *  Frame buffer operations
     */

/* If all you need is that - just don't define ->fb_open */
static int ck1000fb_open(const struct fb_info *info, int user)
{
    return 0;
}

/* If all you need is that - just don't define ->fb_release */
static int ck1000fb_release(const struct fb_info *info, int user)
{
    return 0;
}


    /*
     *  In most cases the `generic' routines (fbgen_*) should be satisfactory.
     *  However, you're free to fill in your own replacements.
     */




/* ------------------------------------------------------------------------- */


    /*
     *  Modularization
     */

#ifdef MODULE
MODULE_LICENSE("GPL");
int init_module(void)
{
    ck1000fb_init();
    return 0;
}

void cleanup_module(void)
{
    unregister_framebuffer(&fb_info.gen.info); 
}
#endif /* MODULE */
