/*****************************************************************************/

/*
 *	pcf8566.c -- driver for PCF8566 Real Time Clock.
 *
 * 	(C) Copyright 2008, ZhengXJ (zheng_xingjian@dahuatech.com)
 */

/*****************************************************************************/
#include <linux/i2c.h>
#include "vs28xx.h"
#include "log.h"
#include "iic.h"

static struct i2c_driver pcf8566_driver;
static struct i2c_client *pcf8566_c;
static unsigned int pcf8566_adapterid;

static unsigned short ignore[] = {I2C_CLIENT_END};
static unsigned short normal_addr[] = {0x28, I2C_CLIENT_END};

static struct i2c_client_address_data addr_data = {
	normal_i2c:		    normal_addr,
	normal_i2c_range:	ignore,
	probe:			    ignore,
    probe_range:		ignore,
    ignore:			    ignore,
    ignore_range:		ignore,
    force:			    ignore,
};


static int pcf8566_attach(struct i2c_adapter *adap, int addr, unsigned short flags, int kind)
{
	struct i2c_client *c;

	c = kmalloc(sizeof(*c), GFP_KERNEL);
	if (!c)
		return -ENOMEM;

	strcpy(c->name, "PCF8566");
	c->id		= pcf8566_driver.id;
	c->flags	= I2C_CLIENT_ALLOW_USE;
	c->addr		= addr;
	c->adapter	= adap;
	c->driver	= &pcf8566_driver;
	c->data		= NULL;

	pcf8566_adapterid = adap->id;

	return i2c_attach_client(c);
}


static int pcf8566_probe(struct i2c_adapter *adap)
{
	return i2c_probe(adap, &addr_data, pcf8566_attach);
}


static int pcf8566_detach(struct i2c_client *client)
{
	i2c_detach_client(client);
	kfree(client);
	return 0;
}


/*
**  һֵ
**  addr:0x28ʾ0x500x29ʾ0x52
*/
unsigned long pcf8566_readbus(unsigned char addr, unsigned char pos)
{
	unsigned char val[1], reg[1];
	struct i2c_msg msgs[2] =
	{
		{0x28, 0,        1, reg},
		{0x28, I2C_M_RD, 1, val}
	};

	if (pcf8566_c != NULL)
	{
	    msgs[0].addr = addr;
	    msgs[1].addr = addr;
		reg[0] = pos;

		if (2 == i2c_transfer(pcf8566_c->adapter, msgs, 2))
		{
		    return val[0];
		}
	}

	return -1;
}


/*
**  дһֵ
**  addr:0x28ʾ0x500x29ʾ0x52
*/
ssize_t pcf8566_writereg(unsigned char addr, unsigned char pos, unsigned char data)
{
	unsigned char i2cbuf[2];
	struct i2c_msg msgs[1];

	if(NULL == pcf8566_c)
	{
	    return -1;
	}

	i2cbuf[0] = pos;                                /* Ĵַ */
	i2cbuf[1] = data;                               /* Ĵֵ */

	msgs[0].addr = addr;
	msgs[0].flags = 0;
	msgs[0].len = 2;
	msgs[0].buf = i2cbuf;

	if (1 == i2c_transfer(pcf8566_c->adapter, msgs, 1))
	{
        return 1;
	}

	return -1;
}


/*
**  дֵ
**  addr:0x28ʾ0x500x29ʾ0x52
*/
ssize_t pcf8566_writebus(unsigned char addr, unsigned char pos, unsigned char *buf, int count)
{
	unsigned char i2cbuf[256];
	struct i2c_msg msgs[1];

	if(NULL == pcf8566_c)
	{
	    return -1;
	}

	i2cbuf[0] = pos;                                /* buf[0]ΪʼĴ255 */
	memcpy(i2cbuf + 1, buf, count);

	msgs[0].addr = addr;
	msgs[0].flags = 0;
	msgs[0].len = count + 1;
	msgs[0].buf = i2cbuf;

	if (1 == i2c_transfer(pcf8566_c->adapter, msgs, 1))
	{
        return count;
	}

	return -1;
}


static int pcf8566_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	return -EINVAL;
}


static struct i2c_driver pcf8566_driver = {
	name:		    "PCF8566",
	id:		        I2C_DRIVERID_PCF8563,
	flags:		    I2C_DF_NOTIFY,
	attach_adapter:	pcf8566_probe,
	detach_client:	pcf8566_detach,
	command:	    pcf8566_command
};


/*
**  0xd2ֵΪ0x32ʾ8·ƵӿģʽΪI2SһƬplaybackinADATM(ĿǰӦò)
**  0xd3-0xdaʾƵ˳򣬹ϵ
**  0xcfֵΪ0x41ʾ2815(IRQ)ʽƬΪ2
**  ֮ǰ0xcfֵΪ0x800x81ʾALINKʽƬΪ12
**  0xcc 0xcdʾͨĿǰ˳
**
**  201007030xF9ֵ0x03ĳ0x83ȺɫȵķΧ
*/
static unsigned char tw2864_public[] =
{
 /* ַ        0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F */
    0x80,   0x09,   0x3F, 0x02, 0x10, 0xDC, 0x00, 0x30, 0x44, 0x50, 0x42,
    0x8A,   0x06,                                                               0xD8, 0xBC, 0xB8, 0x44, 0x2A, 0x00,
    0x90,   0x10,   0x00, 0x68, 0x4C, 0x30, 0x14, 0xA5, 0xE4, 0x05, 0x00, 0x28, 0x44, 0x44, 0x20, 0x90, 0x62, 0x55,
    0xA4,   0x0C,                           0x1A, 0x1A, 0x1A, 0x1A, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0x00,
    0xB0,   0x01,   0x00,
    0xC4,   0x0C,                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0xB1, 0xE4, 0x40, 0x41,
    0xD0,   0x10,   0x88, 0x88, 0x32, 0x20, 0x64, 0xA8, 0xEC, 0x31, 0x75, 0xB9, 0xFD, 0xE1, 0x00, 0x00, 0x00, 0x60,
    0xE0,   0x04,   0x10, 0xC0, 0xAA, 0xAA,
    0xF0,   0x0C,   0x83, 0xB5, 0x09, 0x78, 0x85, 0x00, 0x01, 0x20, 0x64, 0x83, 0x45, 0x8F,
    0xFF,   0x00
};

/* 100401palʹdvrɫ */
static unsigned char tw2864_pal_channel[16] =
{
    0x00, 0x1A, 0x50, 0x35,
    0x80, 0x80, 0x00, 0x12,
    0x19, 0x20, 0x0A, 0xD0,
    0x00, 0x00, 0x01, 0x7F
};

static unsigned char tw2864_ntsc_channel[16] =
{
    0x00, 0x00, 0x80, 0x3B,
    0x80, 0x80, 0x00, 0x02,
    0x14, 0xF0, 0x0E, 0xD0,
    0x00, 0x00, 0x00, 0x7F
};


#ifdef IIC_THREAD
/*
**  ʽ
**  chΪͨ
**  nΪ0ʾPALnΪ1ʾNTSC
**  Ӧжãi2c߳д
**  0x58
*/
void tw2864_set_standard(int ch, int n)
{
    int i;
    int chip;
    int base;
    unsigned char *puc;
    unsigned long *pul;
    unsigned long buf[49];

    FUNC_TRACK(0x58);

    /* 1оƬ4ͨ */
    chip = 0x50 + ((ch >> 2) << 1);
    ch &= 0x3;
    base = ch * 0x10;

    if (n)
    {
        puc = tw2864_ntsc_channel;
    }
    else
    {
        puc = tw2864_pal_channel;
    }

    /*
    **  ǿĳʽʱ0x*eĴҪǰã
    **  ֮ǰõļĴֵӰ:0x*7-0x*b
    */
    buf[1] = chip;
    buf[2] = base + 0xe;
    buf[3] = puc[0xe];

    pul = buf + 4;
    for (i = 0; i < 14; i++)
    {
        pul[0] = chip;
        pul[1] = base + i;
        pul[2] = puc[i];
        pul += 3;
    }

    pul[0] = chip;
    pul[1] = base + 0xf;
    pul[2] = puc[0xf];

    buf[0] = 16;
    tell_i2c_thread((unsigned char *)buf, sizeof(buf));

    return;
}
#endif


/*
**  2864ò
**  0x57
*/
void tw2864_set_common(int chip)
{
    unsigned char *reg;

    FUNC_TRACK(0x57);

    reg = tw2864_public;
    while(1)
    {
        if(0xff == reg[0] && 0 == reg[1])
        {
            break;
        }

        pcf8566_writebus(chip, reg[0], reg + 2, reg[1]);
        reg += (2 + reg[1]);
    }

    pcf8566_writereg(chip, 0xCA, REG_CA);           /* Ƶʼ˫ͨ */
    pcf8566_writereg(chip, 0xCB, REG_CB);
    pcf8566_writereg(chip, 0x89, REG_89);

    return;
}


/*
**  ʽΪPAL
**  0x56
*/
void tw2864_set_pal(int chip, int ch)
{
    FUNC_TRACK(0x56);

    /*
    **  ǿĳʽʱ0x*eĴҪǰã
    **  ֮ǰõļĴֵӰ:0x*7-0x*b
    */
    pcf8566_writereg(chip, ch * 0x10 + 0xe, tw2864_pal_channel[0xe]);

    pcf8566_writebus(chip, ch * 0x10, tw2864_pal_channel, 14);

    pcf8566_writereg(chip, ch * 0x10 + 0xf, tw2864_pal_channel[0xf]);

    return;
}


/*
**  ʽΪNTSC
**  0x55
*/
void tw2864_set_ntsc(int chip, int ch)
{
    FUNC_TRACK(0x55);

    /*
    **  ǿĳʽʱ0x*eĴҪǰã
    **  ֮ǰõļĴֵӰ:0x*7-0x*b
    */
    pcf8566_writereg(chip, ch * 0x10 + 0xe, tw2864_ntsc_channel[0xe]);

    pcf8566_writebus(chip, ch * 0x10, tw2864_ntsc_channel, 14);

    pcf8566_writereg(chip, ch * 0x10 + 0xf, tw2864_ntsc_channel[0xf]);

    return;
}


/*
**  2864
**  0x54
*/
static void config2864(void)
{
    int ch;
    int chip;

    FUNC_TRACK(0x54);

    for(chip = FIRST_CHIP; chip <= END_CHIP; chip++)
    {
        for(ch = 0; ch < 4; ch++)                   /* ͨ */
        {
#ifdef DEFAULT_NTSC
            tw2864_set_ntsc(chip, ch);
#else
            tw2864_set_pal(chip, ch);
#endif
        }

        tw2864_set_common(chip);                    /* ͨò */
    }

    PRINT(INFO, "config2864\n");
}


#ifdef IIC_THREAD

wait_queue_head_t i2c_wqh;
unsigned long g_ulI2cBuffer[1001];                  /* 0Ԫرʾ״̬0ʾУʾ */

/*
**  ֪ͨ߳дi2c
**  0x53
*/
void tell_i2c_thread(unsigned char *pucData, int nLength)
{
    FUNC_TRACK(0x53);

    if (nLength < 1 || nLength > 4096)
    {
        FUNC_COUNTER(0x53, 0x00);
        PRINT(ERR, "i2c msglen err %x\n", nLength);
        return;
    }

    if (g_ulI2cBuffer[0] != 0)
    {
        FUNC_COUNTER(0x53, 0x01);
        PRINT(ERR, "i2c_thread busy\n");
        return;
    }

    memcpy(g_ulI2cBuffer + 1, pucData, nLength);
    g_ulI2cBuffer[0] = nLength;
    wake_up_interruptible(&i2c_wqh);

    return;
}


/*
**  дi2cĴ߳
**  0x52
*/
static int i2c_thread(void *__i2c)
{
    unsigned long ulCount;
    unsigned long ulAddr;
    unsigned long *pul;
    unsigned long d;

    FUNC_TRACK(0x52);

    PRINT(INFO, "i2c_thread\n");

	do
	{
	    PRINT(INFO, "wake 0x%lx!\n", g_ulI2cBuffer[0]);

	    if (g_ulI2cBuffer[0] != 0)
	    {
	        /* g_ulI2cBufferԪ0ܳȣԪ1ÿ3ulong */
	        ulCount = byte_swap4(g_ulI2cBuffer[1]);
	        if ((ulCount * 12 + 4) == g_ulI2cBuffer[0])
	        {
	            pul = g_ulI2cBuffer + 2;
	            for (d = 0; d < ulCount; d++)
	            {
                    ulAddr = byte_swap4(pul[0]);
                    ulAddr /= 2;                    /* i2cַ0x50ʵӦ0x28 */
                    pcf8566_writereg(ulAddr, byte_swap4(pul[1]), byte_swap4(pul[2]));

                    pul += 3;
	            }
	            PRINT(INFO, "iic config 0x%lx\n", ulCount);
	        }
	        else
	        {
                FUNC_COUNTER(0x52, 0x00);
	        }

	        g_ulI2cBuffer[0] = 0;
	    }

		wait_event_interruptible(i2c_wqh, g_ulI2cBuffer[0] != 0);
	} while (!signal_pending(current));

    return 0;
}
#endif


/*
**  iicģע
**  0x51
*/
void iic_exit(void)
{
    FUNC_TRACK(0x51);

	pcf8566_c = NULL;

	i2c_del_driver(&pcf8566_driver);

	return;
}


/*
**  iicģʼ
**  0x50
*/
int iic_init(void)
{
	int ret;

    FUNC_TRACK(0x50);

	ret = i2c_add_driver(&pcf8566_driver);
	if (ret != 0)
	{
	    FUNC_COUNTER(0x50, 0x00);
		PRINT(ERR, "i2c_add_driver fail\n" );
		return -1;
	}

	pcf8566_c = i2c_get_client(I2C_DRIVERID_PCF8563, pcf8566_adapterid, 0);
	if (NULL == pcf8566_c)
	{
	    i2c_del_driver(&pcf8566_driver);
	    FUNC_COUNTER(0x50, 0x01);
		PRINT(ERR, "i2c_add_driver fail\n" );
		return -1;
	}

#ifdef CONFIG_2864
	config2864();                                   /* 2864 */
#endif

#ifdef IIC_THREAD
    g_ulI2cBuffer[0] = 0;                           /* ʾҪĸΪ0 */
    init_waitqueue_head(&i2c_wqh);                  /* ʼi2c_threadȴ */

    /* ΪжԼʱﲻдi2c¿һ߳дi2c */
	ret = kernel_thread(i2c_thread, NULL, CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	if (ret < 0)
	{
	    FUNC_COUNTER(0x50, 0x02);
	    PRINT(WARN, "create i2c_thread fail\n");
	}
#endif

	return 0;
}
