/*
 *
 * Delay routines, using a pre-computed "loops_per_second" value.
 * 
 */

#ifndef _CKCORENOMMU_DELAY_H
#define _CKCORENOMMU_DELAY_H

#include <asm/param.h>

extern __inline__ void __delay(unsigned long loops)
{
	__asm__ __volatile__ (	"1: decgt %0\n\t"
				"bt 1b"
		              : "=r" (loops) 
                              : "0" (loops));
}

/*
 * Use only for very small delays ( < 20 msec).  Should probably use a
 * lookup table, really, as the multiplications take much too long with
 * short delays.  This is a "reasonable" implementation, though (and the
 * first constant multiplications gets optimized away if the delay is
 * a constant)  

 * Division by multiplication and shifts.
 *
 *  We want the number of loops to complete, where loops is:
 *      (us * (HZ * loops_per_jiffy)) / 10^6
 *  or
 *      (ns * (HZ * loops_per_jiffy)) / 10^9
 *
 *  Since we don't want to do long division, we multiply both numerator
 *  and denominator by (2^27 / 10^6):
 *
 *      (us * (2^27 / 10^6) * HZ * loops_per_jiffy) / 2^27
 *
 *  =>  (us * (2^27 * HZ / 10^6) * loops_per_jiffy) / 2^27
 *
 *  ~>  (((us * (2^27 * HZ / 10^6)) / 2^11) * (loops_per_jiffy / 2^11)) / 2^5
 *         (for large loops_per_jiffy >> 2^12)
 *
 *  Note: maximum loops_per_jiffy = 20000000 (bogomips = 400)
 *        minimum loops_per_jiffy = 20000 (bogomips = 0.4)
 *
 * Note: we rely on HZ = 100.
 */

#define UDELAY_FACTOR  13421  /* 2^27 * HZ / 10^6 */
#define NDELAY_FACTOR  13     /* 2^27 * HZ / 10^9 */

extern unsigned long loops_per_jiffy;

extern __inline__ void udelay(unsigned long usecs)
{
        register unsigned long temp = loops_per_jiffy >> 11;
	usecs *= UDELAY_FACTOR;

	__asm__ (
                "lsri %0, 11\n\t"
                "mult %0, %2"
		: "=r" (usecs) 
		: "0" (usecs), "r" (temp));
        usecs >>= 5;
	__delay(usecs);
}

/* #define ndelay(n)	udelay((n) * 5) */
extern __inline__ void ndelay(unsigned long nsecs)
{
        register unsigned long temp = loops_per_jiffy >> 11;
        nsecs *= NDELAY_FACTOR; 

	__asm__ (
                "lsri %0, 11\n\t"
                "mult %0, %2"
		: "=r" (nsecs) 
		: "0" (nsecs), "r" (temp));
        nsecs >>= 5;
	__delay(nsecs);
}


#endif /* defined(_CKCORENOMMU_DELAY_H) */
